///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/field_element.h"
using namespace rheolef;
using namespace std;

namespace rheolef {

// cstor by specifying the quadrature formulae
void
field_element::initialize (const space& X, const quadrature_option_type& qopt) const {
  _X = X;
  _X.freeze();
  if (qopt.get_order() != std::numeric_limits<quadrature_option_type::size_type>::max()) {
    _quad.set_order (qopt.get_order());
  } else {
    size_type k = X.degree();
    _quad.set_order (k);
  }
  _quad.set_family (qopt.get_family());

warning_macro ("quadrature : " << _quad.get_family_name() << " order " << _quad.get_order());

  _b_table.set (_quad, X.get_basis());
  _piola.set (_quad, X);
  _initialized = true;
}
// compute Mk(i,q) such that mk(phi_i,f) = sum_q Mk(i,q)*f(q)*wq
template<>
void
field_element::eval (const geo_element& K,
	const vector<Float>& fq,
	vector<Float>& uk) const
{
  reference_element hat_K = K;
  size_t b_size = get_basis().size(hat_K);
  uk.resize (b_size);
  fill (uk.begin(), uk.end(), Float(0.0));
  quadrature::const_iterator first_quad = _quad.begin(hat_K);
  quadrature::const_iterator last_quad  = _quad.end  (hat_K);
  for (size_type q = 0; first_quad != last_quad; first_quad++, q++) {
    // TODO: passer (hat_x, K) a` field_o_characteristic: ca evite de localiser
    // si on a vh sur le meme mailage
    Float wq = _piola.weight_coordinate_system (K, q);
    wq *= _piola.det_jacobian_transformation (K,q);
    wq *= (*first_quad).w;
    Float fwq = fq[q]*wq;
    basis_on_quadrature::const_iterator first_phi = _b_table.begin (hat_K, q);
    basis_on_quadrature::const_iterator last_phi  = _b_table.end (hat_K, q);
    for (size_t i = 0; first_phi != last_phi; first_phi++, i++) {
      Float phi_wq = (*first_phi)*wq;
      uk[i] += (*first_phi)*fwq;
    }
  }
}
// idem for vector-valued functions
template<>
void
field_element::eval (const geo_element& K,
	const vector<point>& fq,
	vector<Float>& uk) const
{
  size_t n_comp = K.dimension(); // TODO: pb when 3d surface triangulation
  reference_element hat_K = K;
  size_t b_size = get_basis().size(hat_K);
  uk.resize (n_comp*b_size);
  fill (uk.begin(), uk.end(), Float(0.0));
  quadrature::const_iterator first_quad = _quad.begin(hat_K);
  quadrature::const_iterator last_quad  = _quad.end  (hat_K);
  for (size_type q = 0; first_quad != last_quad; first_quad++, q++) {
    // TODO: passer (hat_x, K) a` field_o_characteristic: ca evite de localiser
    // si on a vh sur le meme mailage
    Float wq = _piola.weight_coordinate_system (K, q);
    wq *= _piola.det_jacobian_transformation (K,q);
    wq *= (*first_quad).w; 
    point fwq = fq[q]*wq;
    basis_on_quadrature::const_iterator first_phi = _b_table.begin (hat_K, q);
    basis_on_quadrature::const_iterator last_phi  = _b_table.end (hat_K, q);
    for (size_t i = 0; first_phi != last_phi; first_phi++, i++) {
      Float phi_wq = (*first_phi)*wq;
      for (size_t i_comp = 0; i_comp < n_comp; i_comp++) {
        uk[i_comp*b_size + i] += (*first_phi)*fwq[i_comp];
      }
    }
  }
}
template<>
void 
field_element::operator() (const geo_element& K,
	const field_o_characteristic& f,
	vector<Float>& uk) const
{
  switch (f.get_field().get_valued_type()) {
    case fem_helper::scalar: {
      vector<Float> fq(_quad.size(K));
      quadrature::const_iterator first = _quad.begin(K);
      quadrature::const_iterator last  = _quad.end  (K);
      for (size_type q = 0; first != last; first++, q++) {
        point xq = get_space().get_geo().dehatter ((*first).x, K.index());
        fq[q] = f(xq);
      }
      eval (K, fq, uk);
      break;
    }
    case fem_helper::vectorial: {
      vector<point> fq(_quad.size(K));
      quadrature::const_iterator first = _quad.begin(K);
      quadrature::const_iterator last  = _quad.end  (K);
      for (size_type q = 0; first != last; first++, q++) {
        point xq = get_space().get_geo().dehatter ((*first).x, K.index());
        fq[q] = f.vector_evaluate(xq);
      }
      eval (K, fq, uk);
      break;
    }
    default:
      error_macro ("unsupported");
  }
}
template<>
void 
field_element::operator() (const geo_element& K,
	const field& fh,
	vector<Float>& uk) const
{
  switch (fh.get_valued_type()) {
    case fem_helper::scalar: {
      vector<Float> fq(_quad.size(K));
      quadrature::const_iterator first = _quad.begin(K);
      quadrature::const_iterator last  = _quad.end  (K);
      for (size_type q = 0; first != last; first++, q++) {
        meshpoint xq (K.index(), (*first).x);
        fq[q] = fh.evaluate (xq);
      }
      eval (K, fq, uk);
      break;
    }
    case fem_helper::vectorial: {
      size_t d = fh.dimension();
      vector<point> fq(_quad.size(K));
      quadrature::const_iterator first = _quad.begin(K);
      quadrature::const_iterator last  = _quad.end  (K);
      for (size_type q = 0; first != last; first++, q++) {
        meshpoint xq (K.index(), (*first).x);
	for (size_t i = 0; i < d; i++) {
          fq[q][i] = fh.evaluate (xq, i);
        }
      }
      eval (K, fq, uk);
      break;
    }
    default:
      error_macro ("unsupported yet");
  }
}
}// namespace rheolef
