#ifndef _RHEO_REFERENCE_ELEMENT_H
#define _RHEO_REFERENCE_ELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/compiler.h"

/*Class:reference_element
NAME: @code{reference_element} - reference element
@cindex  reference element
@clindex reference_element
SYNOPSYS:
  @noindent
  The @code{reference_element} class defines all supported types of 
  geometrical elements in one, two and three dimensions. The set of
  supported elements are designate by a letter
  @table @samp
       	@itemx p
       		point (dimension 0)
       	@itemx e
       		edge (dimension 1)
	@itemx t
		triangle(dimension 2)
	@itemx q
		quadrangle(dimension 2)
	@itemx T
		tetrahedron(dimension 3)
        @itemx P
		prism(dimension 3)
	@itemx H
		hexaedron(dimension 3)
  @end table

AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   30 november 2003
End:
*/

namespace rheolef { 
//<reference_element:
class reference_element {
public:


// typedefs:

    typedef std::vector<int>::size_type size_type;

    // defines enum_type { p, t, q ..., H, ...};
    // in an automatically generated file :

    typedef enum { 
	p = 0, 
	e = 1, 
	t = 2, 
	q = 3, 
	T = 4, 
	P = 5, 
	H = 6, 
	max_size = 7 
    } enum_type; 
    
    typedef enum {
	Lagrange = 0,
	Hermite = 1,
	dof_family_max =2
    } dof_family_type;
 
// constants:

    static const size_type not_set;
    static const size_type max_n_subgeo      = 12; 
    static const size_type max_subgeo_vertex = 8; 

// allocators/deallocators:

    reference_element (enum_type x = max_size);

// accessors:

    enum_type type() const;
    char name() const;
    size_type dimension() const;
    friend Float measure (reference_element hat_K);
    size_type size() const;
    size_type n_edge() const;
    size_type n_face() const;
    size_type n_subgeo(size_type subgeo_dim) const;
    size_type subgeo_size(size_type subgeo_dim, size_type i_subgeo) const;
    size_type subgeo_local_vertex(size_type subgeo_dim, size_type i_subgeo, size_type i_subgeo_vertex) const;
    size_type heap_size() const;
    size_type heap_offset(size_type subgeo_dim) const;

    void set_type (enum_type x);
    void set_type (size_type n_vertex, size_type dim);
    void set_name (char name);

// data:
protected:

    enum_type _x;

// constants:

    // these constants are initialized in
    //   "reference_element_declare.c"
    // automatically generated.

    static const char _name                     [max_size];
    static const size_type _dimension           [max_size];
    static const size_type _n_subgeo            [max_size][4];
    static const size_type _subgeo_size         [max_size*4*max_n_subgeo];
    static const size_type _subgeo_local_vertex [max_size*4*max_n_subgeo*max_subgeo_vertex];
    static const size_type _heap_size           [max_size];
    static const size_type _heap_offset         [max_size][4];
    friend std::istream& operator>>(std::istream&, class geo_element&);
};
//>reference_element:
inline
reference_element::reference_element (enum_type x)
  : _x(x)
{
    assert_macro (x >= 0 && x <= max_size, "invalid type " << x);
}
inline
reference_element::enum_type
reference_element::type() const
{
    return _x;
}
inline
void
reference_element::set_type (enum_type x)
{
  _x = x;
}
inline
char
reference_element::name() const
{
    return _name[_x];
}
inline
reference_element::size_type
reference_element::dimension() const
{
    return _dimension[_x];
}
inline
reference_element::size_type
reference_element::n_subgeo(size_type subgeo_dim) const
{
    return _n_subgeo [_x][subgeo_dim];
}
inline
reference_element::size_type
reference_element::subgeo_size(size_type subgeo_dim, 
	size_type i_subgeo) const
{
    size_type p = (4*_x + subgeo_dim)*max_n_subgeo + i_subgeo;
    return _subgeo_size [p];
}
inline
reference_element::size_type
reference_element::size() const
{
    return n_subgeo (0);
}
inline
reference_element::size_type
reference_element::n_edge() const
{
    return n_subgeo (1);
}
inline
reference_element::size_type
reference_element::n_face() const
{
    return n_subgeo (2);
}
inline
reference_element::size_type
reference_element::subgeo_local_vertex(size_type d, size_type k, size_type j) const
{
    size_type p = ((_x*4 + d)*max_n_subgeo + k)*max_subgeo_vertex + j;
    return _subgeo_local_vertex [p];
}
inline
reference_element::size_type
reference_element::heap_size() const
{
    return _heap_size [_x];
}
inline
reference_element::size_type
reference_element::heap_offset(size_type subgeo_dim) const
{
    return _heap_offset [_x][subgeo_dim];
}
}// namespace rheolef
#endif // _RHEO_REFERENCE_ELEMENT_H
