#ifndef _RHEO_ARRAY_H
#define _RHEO_ARRAY_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

# include "rheolef/parallel.h"
# include "rheolef/distributor.h"
# include "rheolef/parstream.h"
# include "rheolef/heap_allocator.h"

/// @brief array element output helper
template <class T>
struct _array_put_element_type {
  std::ostream& operator() (std::ostream& os, const T& x) { return os << x; }
};
/// @brief array element input helper
template <class T>
struct _array_get_element_type {
  std::istream& operator() (std::istream& is, T& x) { return is >> x; }
};
#ifdef _RHEOLEF_HAVE_MPI
// -------------------------------------------------------------
// send a pair<size_t,T> via mpi & serialization
// -------------------------------------------------------------
// non-intrusive version of serialization for pair<size_t,T> 
namespace boost {
 namespace serialization {
  template<class T, class Archive>
  void serialize (Archive & ar, std::pair<size_t, T>& g, const unsigned int version) {
    ar & g.first;
    ar & g.second;
  }
 } // namespace serialization
} // namespace boost

// Some serializable types, like pair<size_t,double>, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <>
  struct is_mpi_datatype<std::pair<size_t,double> > : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI

#ifndef TO_CLEAN
namespace rheolef {
// for debug
template<class T>
inline
std::ostream&
operator<< (std::ostream& o, const std::pair<size_t,T>& x) {
  return o << "pair{" << x.first << ", "<< x.second << "}";
}
} // namespace rheolef
#endif // TO_CLEAN
// -------------------------------------------------------------
// the sequential representation
// -------------------------------------------------------------
namespace rheolef {

template <class T>
class array_seq_rep : public std::vector<T> {
public:
    typedef T                                       value_type;
    typedef typename std::vector<T>::size_type      size_type;
    typedef typename std::vector<T>::iterator       iterator;
    typedef typename std::vector<T>::const_iterator const_iterator;
    typedef typename std::vector<T>::const_reference const_reference;
    typedef typename std::vector<T>::reference      reference;
    typedef distributor::communicator_type          communicator_type;
    typedef sequential                              memory_type;

    array_seq_rep (size_type loc_size = 0,       const T& init_val = T());
    array_seq_rep (const distributor& ownership, const T& init_val = T());
    array_seq_rep (const array_seq_rep<T>& x);
    void resize   (const distributor& ownership, const T& init_val = T());
    void resize   (size_type loc_size = 0,       const T& init_val = T());

    size_type size() const { return std::vector<T>::size(); }
    iterator begin() { return std::vector<T>::begin(); }
    const_iterator begin() const { return std::vector<T>::begin(); }
    iterator end() { return std::vector<T>::end(); }
    const_iterator end() const { return std::vector<T>::end(); }
    const distributor& ownership() const { return _ownership; }

    reference       operator[] (size_type i)       { return std::vector<T>::operator[] (i); }
    const_reference operator[] (size_type i) const { return std::vector<T>::operator[] (i); }
   
    size_type par_size () const { return std::vector<T>::size(); }
    size_type first_index () const { return 0; }
    size_type last_index () const { return std::vector<T>::size(); }
    void set (size_type i, const T& xi) { operator[](i) = xi; }
    void assembly_begin () {}
    void assembly_end () {}
    void repartition (					      // old_numbering for *this
        const array_seq_rep<size_type>& partition,	      // old_ownership
        array_seq_rep<T>&               new_array,	      // new_ownership (created)
        array_seq_rep<size_type>&       old_numbering,	      // new_ownership
        array_seq_rep<size_type>&       new_numbering) const  // old_ownership
    {
	error_macro ("not yet");
    }
    iparstream& get (iparstream& s);
    oparstream& put (oparstream& s) const;
    template <class GetFunction> iparstream& get (iparstream& ips, GetFunction get_element);
    template <class PutFunction> oparstream& put (oparstream& ops, PutFunction put_element) const;
    void dump (std::string name) const;
protected:
// data:
    distributor      _ownership;
};
// -------------------------------------------------------------
// the distributed representation
// -------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
class array_mpi_rep : public array_seq_rep<T>
{
public:
    typedef typename array_seq_rep<T>::value_type     value_type;
    typedef typename array_seq_rep<T>::size_type      size_type;
    typedef typename array_seq_rep<T>::iterator       iterator;
    typedef typename array_seq_rep<T>::const_iterator const_iterator;
    typedef distributor::communicator_type          communicator_type;
    typedef distributed                               memory_type;

    array_mpi_rep (const array_mpi_rep<T>& x);
    array_mpi_rep (
    	size_type par_size = 0,
	const T&  init_val = T(),
    	size_type loc_size = distributor::decide); 
    void resize (
    	size_type par_size = 0,
	const T&  init_val = T(),
    	size_type loc_size = distributor::decide);
    array_mpi_rep (
    	const distributor& ownership,
	const T&  init_val = T());
    void resize (
    	const distributor& ownership,
	const T&  init_val = T());

    size_type size() const       { return array_seq_rep<T>::size(); }
    const_iterator begin() const { return array_seq_rep<T>::begin(); }
    const_iterator end() const   { return array_seq_rep<T>::end(); }
    iterator begin()             { return array_seq_rep<T>::begin(); }
    iterator end()               { return array_seq_rep<T>::end(); }

    const distributor& ownership() const  { return array_seq_rep<T>::_ownership; }
    const mpi::communicator& comm() const { return ownership().comm(); }
    size_type first_index () const        { return ownership().first_index(); }
    size_type last_index () const         { return ownership().last_index(); }
    size_type par_size () const           { return ownership().par_size(); }
    void set (size_type i, const T& xi);
    void assembly_begin ();
    void assembly_end ();
    void repartition (					      // old_numbering for *this
        const array_mpi_rep<size_type>& partition,	      // old_ownership
        array_mpi_rep<T>&               new_array,	      // new_ownership (created)
        array_mpi_rep<size_type>&       old_numbering,	      // new_ownership
        array_mpi_rep<size_type>&       new_numbering) const; // old_ownership
    template<class Set, class Map>
    void scatter (const Set& ext_idx_set, Map& ext_idx_map) const;

    iparstream& get (iparstream& s);
    oparstream& put (oparstream& s) const;
    template <class GetFunction> iparstream& get (iparstream& ips, GetFunction get_element);
    template <class PutFunction> oparstream& put (oparstream& ops, PutFunction put_element) const;
    template <class PutFunction> oparstream& permuted_put (oparstream& ops, const array_mpi_rep<size_type>& perm,
		PutFunction put_element) const;
    void dump (std::string name) const;
protected:
// data:
    // for communication during assembly_begin(), assembly_end()
    typedef std::map <size_type, T, std::less<size_type>, heap_allocator<std::pair<size_type,T> > >   map_type;
    struct message_type {
        std::list<std::pair<size_t,mpi::request> >  waits;
        std::vector<std::pair<size_type,T> >        data;
    };
    map_type        _stash;
    message_type    _send;
    message_type    _receive;
    size_type       _receive_max_size;
};
#endif // _RHEOLEF_HAVE_MPI
// -------------------------------------------------------------
// the basic class with a smart pointer to representation
// the user-level class with memory-model parameter
// -------------------------------------------------------------
/*Class:array
NAME:  array - container in distributed environment (@PACKAGE@-@VERSION@)
SYNOPSYS:       
 STL-like vector container for a distributed memory machine model.
EXAMPLE:
   A sample usage of the class is:
   @example
     int main(int argc, char**argv) {
        environment parallel(argc, argv);
        array<double> x(100, 3.14);
        pcout << x << endl;
     }
   @end example
SEE ALSO: "parstream"(1)
End:
*/
template <class R>
class basic_array : public smart_pointer<R> {
public:

// typedefs:

    typedef typename R::value_type value_type;
    typedef typename R::value_type T;
    typedef typename R::size_type  size_type;
    typedef typename R::memory_type memory_type;
    typedef typename R::iterator iterator;
    typedef typename R::const_iterator const_iterator;
    typedef typename R::communicator_type communicator_type;
    typedef typename R::reference reference;
    typedef typename R::const_reference const_reference;

// allocators/deallocators:

    basic_array (size_type par_size = 0,       const T&  init_val = T());
    void resize (size_type par_size = 0,       const T&  init_val = T());
    basic_array (const distributor& ownership, const T&  init_val = T());
    void resize (const distributor& ownership, const T&  init_val = T());

// accessors:

    // sizes
    size_type size () const;
    size_type par_size () const;
    const distributor& ownership() const;
    const communicator_type& comm() const;

    // range on local memory
    size_type first_index () const;
    size_type last_index () const;

    // local iterators
    iterator begin();
    const_iterator begin() const;
    iterator end();
    const_iterator end() const;

    reference operator[] (size_type i);
    const_reference operator[] (size_type i) const;

// global modifiers:

    void set (size_type i, const T& xi);
    void assembly();
    void assembly_begin ();
    void assembly_end ();

// apply a partition:
 
    template<class RepSize>
    void repartition (			      // old_numbering for *this
        const RepSize&  partition,	      // old_ownership
        basic_array<R>& new_array,	      // new_ownership (created)
        RepSize&        old_numbering,	      // new_ownership
        RepSize&        new_numbering) const;  // old_ownership

// i/o:

    void dump (std::string name) const;
};
template <class T, class M = rheo_default_memory_model>
class array {
public:
    typedef M memory_type;
};
//<verbatim:
template <class T>
class array<T,sequential> : public basic_array<array_seq_rep<T> > {
public:
    typedef sequential memory_type;
    typedef typename basic_array<array_seq_rep<T> >::size_type size_type;
    typedef typename basic_array<array_seq_rep<T> >::reference reference;
    typedef typename basic_array<array_seq_rep<T> >::const_reference const_reference;
    array (size_type par_size = 0,       const T& init_val = T());
    array (const distributor& ownership, const T& init_val = T());
    reference       operator[] (size_type i)       { return basic_array<array_seq_rep<T> >::operator[] (i); }
    const_reference operator[] (size_type i) const { return basic_array<array_seq_rep<T> >::operator[] (i); }
    oparstream& put (oparstream& ops) const { return basic_array<array_seq_rep<T> >::data().put(ops); }
    iparstream& get (iparstream& ips)       { return basic_array<array_seq_rep<T> >::data().get(ips); }
    template <class GetFunction>
    iparstream& get (iparstream& ips, GetFunction get_element) { return basic_array<array_seq_rep<T> >::data().get(ips, get_element); }
    template <class PutFunction>
    oparstream& put (oparstream& ops, PutFunction put_element) const { return basic_array<array_seq_rep<T> >::data().put(ops, put_element); }
};
//>verbatim:
#ifdef _RHEOLEF_HAVE_MPI
//<verbatim:
template <class T>
class array<T,distributed> : public basic_array<array_mpi_rep<T> > {
public:
    typedef distributed memory_type;
    typedef typename basic_array<array_mpi_rep<T> >::size_type size_type;
    typedef typename basic_array<array_mpi_rep<T> >::reference reference;
    typedef typename basic_array<array_mpi_rep<T> >::const_reference const_reference;
    array (size_type par_size = 0,       const T&  init_val = T());
    array (const distributor& ownership, const T&  init_val = T());
    reference       operator[] (size_type i)       { return basic_array<array_mpi_rep<T> >::operator[] (i); }
    const_reference operator[] (size_type i) const { return basic_array<array_mpi_rep<T> >::operator[] (i); }
    oparstream& put (oparstream& ops) const { return basic_array<array_mpi_rep<T> >::data().put(ops); }
    iparstream& get (iparstream& ips)       { return basic_array<array_mpi_rep<T> >::data().get(ips); }
    template<class Set, class Map>
    void scatter (const Set& ext_idx_set, Map& ext_idx_map) const {
	return basic_array<array_mpi_rep<T> >::data().scatter (ext_idx_set, ext_idx_map);
    }
    template <class GetFunction>
    iparstream& get (iparstream& ips, GetFunction get_element) {
	return basic_array<array_mpi_rep<T> >::data().get(ips, get_element); }
    template <class PutFunction>
    oparstream& put (oparstream& ops, PutFunction put_element) const {
	return basic_array<array_mpi_rep<T> >::data().put(ops, put_element); }
    template <class PutFunction>
    oparstream& permuted_put (oparstream& ops, const array<size_type,distributed>& perm, PutFunction put_element) const {
	return basic_array<array_mpi_rep<T> >::data().permuted_put (ops, perm.data(), put_element); }
};
//>verbatim:
#endif // _RHEOLEF_HAVE_MPI

// inputs/outputs:
template <class T, class M>
iparstream& operator >> (iparstream& s, array<T,M>& x);

template <class T, class M>
oparstream& operator << (oparstream& s, const array<T,M>& x);
// -------------------------------------------------------------
// inline'd : basic_array
// -------------------------------------------------------------
template <class R>
inline
basic_array<R>::basic_array (
    	size_type          par_size,
	const value_type&  init_val)
 : smart_pointer<R> (new_macro(R(par_size,init_val)))
{
}
template <class R>
inline
void
basic_array<R>::resize (
    	size_type 	   par_size,
	const value_type&  init_val)
{
    smart_pointer<R>::data().resize(par_size,init_val);
}
template <class R>
inline
basic_array<R>::basic_array (
    	const distributor& ownership,
	const T&           init_val)
 : smart_pointer<R> (new_macro(R(ownership,init_val)))
{
}
template <class R>
inline
void
basic_array<R>::resize (
    	const distributor& ownership,
	const T&           init_val)
{
    smart_pointer<R>::data().resize(ownership,init_val);
}
template <class R>
inline
const distributor&
basic_array<R>::ownership () const
{
    return smart_pointer<R>::data().ownership();
}
template <class R>
inline
const typename basic_array<R>::communicator_type&
basic_array<R>::comm () const
{
    return ownership().comm();
}
template <class R>
inline
typename basic_array<R>::size_type
basic_array<R>::last_index () const 
{
    return smart_pointer<R>::data().last_index();
}
template <class R>
inline
typename basic_array<R>::size_type
basic_array<R>::first_index () const 
{
    return smart_pointer<R>::data().first_index();
}
template <class R>
inline
typename basic_array<R>::size_type
basic_array<R>::size () const
{
    return smart_pointer<R>::data().size();
}
template <class R>
inline
typename basic_array<R>::size_type
basic_array<R>::par_size () const
{
    return smart_pointer<R>::data().par_size();
}
template <class R>
inline
typename basic_array<R>::iterator
basic_array<R>::begin ()
{
    return smart_pointer<R>::data().begin();
}
template <class R>
inline
typename basic_array<R>::iterator
basic_array<R>::end ()
{
    return smart_pointer<R>::data().end();
}
template <class R>
inline
typename basic_array<R>::const_iterator
basic_array<R>::begin () const
{
    return smart_pointer<R>::data().begin();
}
template <class R>
inline
typename basic_array<R>::const_iterator
basic_array<R>::end () const
{
    return smart_pointer<R>::data().end();
}
template <class R>
inline
void
basic_array<R>::assembly_begin ()
{
    smart_pointer<R>::data().assembly_begin();
}
template <class R>
inline
void
basic_array<R>::assembly_end ()
{
    smart_pointer<R>::data().assembly_end();
}
template <class R>
inline
void
basic_array<R>::assembly()
{
    smart_pointer<R>::data().assembly_begin();
    smart_pointer<R>::data().assembly_end();
}
template<class R>
template<class RepSize>
inline
void
basic_array<R>::repartition (
	const RepSize&  partition,
	basic_array<R>& new_array,
	RepSize&        old_numbering,
	RepSize&        new_numbering) const
{
    smart_pointer<R>::data().repartition(
    	partition.data(),
    	new_array.data(),
    	old_numbering.data(),
    	new_numbering.data());
}
template <class R>
inline
void
basic_array<R>::set (size_type i, const value_type& val)
{
    smart_pointer<R>::data().set(i, val);
}
template <class R>
inline
typename basic_array<R>::reference
basic_array<R>::operator[] (size_type i) 
{
    return smart_pointer<R>::data().operator[] (i);
}
template <class R>
inline
typename basic_array<R>::const_reference
basic_array<R>::operator[] (size_type i) const
{
    return smart_pointer<R>::data().operator[] (i);
}
template <class R>
inline
void
basic_array<R>::dump (std::string name) const
{
    return smart_pointer<R>::data().dump (name);
}
// ===========================================================
// inlined : array
// ===========================================================
template <class T>
inline
array<T,sequential>::array (size_type par_size, const T&  init_val)
  : basic_array<array_seq_rep<T> >(par_size,init_val)
{
}
template <class T>
inline
array<T,sequential>::array (const distributor&  ownership, const T&  init_val)
  : basic_array<array_seq_rep<T> >(ownership,init_val)
{
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
inline
array<T,distributed>::array (size_type par_size, const T&  init_val)
  : basic_array<array_mpi_rep<T> >(par_size,init_val)
{
}
template <class T>
inline
array<T,distributed>::array (const distributor&  ownership, const T&  init_val)
  : basic_array<array_mpi_rep<T> >(ownership,init_val)
{
}
#endif // _RHEOLEF_HAVE_MPI

template <class T>
inline
iparstream&
operator >> (iparstream& ips,  array<T,sequential>& x)
{ 
    return x.get(ips); 
}
template <class T> 
inline
oparstream&
operator << (oparstream& ops, const array<T,sequential>& x)
{
    return x.put(ops);
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
inline
iparstream&
operator >> (iparstream& ips,  array<T,distributed>& x)
{ 
    return x.get(ips); 
}
template <class T> 
inline
oparstream&
operator << (oparstream& ops, const array<T,distributed>& x)
{
    return x.put(ops);
}
#endif // _RHEOLEF_HAVE_MPI
} // namespace rheolef
// -------------------------------------------------------------
// not inlined : longer code
// -------------------------------------------------------------
#include "rheolef/array_seq.icc"
#include "rheolef/array_mpi.icc"
#endif // _RHEO_ARRAY_H
