#ifndef _RHEO_MATRIX_SOLVER_H
#define _RHEO_MATRIX_SOLVER_H
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:matrix_solver
NAME: @code{cg_solver}, @code{ldlt_solver} - wrappers for algoritms
@clindex cg_solver
@clindex ldlt_solver
@clindex iterative_stokes_solver
DESCRIPTION: 
  @noindent
  These class offers an uniform invocation
  for linear system solver, and thus are suitable for 
  upper-lever template algorithm developments.
  These classes call a matrix inversion method and 
  can contains some parameters.

  @noindent
  These classes are used by @code{iterative_stokes_solver}.
  @code{cg_solver}, @code{ldlt_solver} are examples of
  base classes @code{iterative_matrix_solver}
  and
  @code{direct_matrix_solver}, respectively.
EXAMPLE: 
  @example
	cg_solver my_algo(100, 1.e-14, cout);
  	my_algo (a1,m_a1,x1,b1);
  	my_algo (a2,m_a2,x2,b2);  
  @end example
SEE ALSO: 
  class "iterative_solver"
AUTHOR: 
  LMC-IMAG, 38041 Grenoble cedex 9, France
  | Nicolas.Roquet@imag.fr
DATE:   24 sept 1997
METHODS: @matrixsolver
End:
*/

#include "rheolef/gen_solver.h"
#include "rheolef/cg.h"
namespace rheolef { 

//<matrixsolver:
// iterative conjugate gradient solver
template<
    class MatrixPreconditionner,
    class Matrix, 
    class InputVector, 
    class OutputVector, 
    class Real>
class cg_solver
  : public iterative_solver<Real>
{
public:  

// allocator:

  cg_solver(int n_iter_max_init, Real epsilon_init,
	  std::ostream* p_clog_init = 0, std::ostream* p_cres_init = 0);

// solve method:

  int operator()(
	const Matrix&                     a, 
        const MatrixPreconditionner&      ma, 
	OutputVector&                     x, 
        const InputVector&                b)
        const;
};
//>matrixsolver:

template<
    class MatrixPreconditionner,
    class Matrix, 
    class InputVector, 
    class OutputVector, 
    class Real>
inline
cg_solver<MatrixPreconditionner,Matrix,InputVector,OutputVector,Real>::cg_solver(
    int n_iter_max_init, 
    Real epsilon_init,
    std::ostream* p_clog_init, 
    std::ostream* p_cres_init)
: iterative_solver<Real>( 
	n_iter_max_init, 
	epsilon_init, 
	p_cres_init, 
	p_clog_init )
{
}
template<
    class MatrixPreconditionner,
    class Matrix, 
    class InputVector, 
    class OutputVector, 
    class Real>
inline
int
cg_solver<MatrixPreconditionner,Matrix,InputVector,OutputVector,Real>::operator()(
	const Matrix&                     A, 
        const MatrixPreconditionner&      M_A, 
	OutputVector&                     x, 
        const InputVector&                b)
        const
{
    int n_iter  = iterative_solver<Real>::_n_iter_max ;
    Real residu = iterative_solver<Real>::_epsilon    ;
    int status = cg( A, x, b, M_A, n_iter, residu ) ;
    bilan(status, n_iter, residu) ;
    return status ;
}

//<matrixsolver:
// Direct matrix solver : Choleski factorization
template<
    class MatrixFactorized, 
    class Matrix, 
    class InputVector,
    class OutputVector>
class ldlt_solver 
  : public direct_solver
{
public:

// allocator:

    ldlt_solver();

// solve method:

    int operator()(
	const Matrix&           a, 
	const MatrixFactorized& m, 
	OutputVector&           x, 
	const InputVector&      b)
        const;
};
//>matrixsolver:

template<
    class MatrixFactorized, 
    class Matrix, 
    class InputVector,
    class OutputVector>
inline
ldlt_solver<MatrixFactorized,Matrix,InputVector,OutputVector>::ldlt_solver()
  : direct_solver()
{
}
template<
    class MatrixFactorized, 
    class Matrix, 
    class InputVector,
    class OutputVector>
inline
int
ldlt_solver<MatrixFactorized,Matrix,InputVector,OutputVector>::operator()(
	const Matrix&           A, 
	const MatrixFactorized& m, 
	OutputVector&           x, 
	const InputVector&      b )
        const
{
    m.solve( b, x ) ;
    return 0 ;
}

}// namespace rheolef
#endif // _RHEO_MATRIX_SOLVER_H
