///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:geo-dom
NAME: "geo-dom"
SYNOPSIS:
DESCRIPTION:
EXAMPLE: 
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Nicolas.Roquet@imag.fr
SEE ALSO:
DATE:
    11 dec 1998
End:
*/

// ============================================================================
//  Includes
// ============================================================================ 
#include "rheolef/rheolef.h"
#include "rheolef/trace.h"
using namespace rheolef;
using namespace std;

// ============================================================================
//  Usage
// ============================================================================

void usage()
{
  cerr << "geo-dom: usage: geo-dom "
       << "[-hb|-matlab] "
       << "{-Igeodir}* "
       << "-app P1|P2 "
       << "{-|mesh[.geo]}"
       << "{domain}+"
       << endl ;
  exit (1) ;
}
// ============================================================================
//  Utilities
// ============================================================================

Float pol_un (point x)
{
  return 10.*x[0] + 100.*x[1] + 1000. ;
}
void put_hb(const csr<Float>& a, string name) {

    string fname = name + ".hb";
    ofstream os(fname.c_str()); 
    os << hb << a; 
    cerr << "! file \"" << fname << "\" created." << endl;
}
// ============================================================================
//  Main
// ============================================================================
int main(int argc, char**argv) {
  // --------------------------------------------------------------------------
  //  Read geo & domains
  // -------------------------------------------------------------------------- 
  geo g ;
  geo::size_type n_dom=0 ;
  vector<geo::size_type> io_dom(10) ;
  const char* approx = "P1"; 
  bool mesh_done = false ;
  bool do_matlab = false ;
  bool do_hb     = false ; 
  if (argc <= 1) usage() ;
  for (int io=1 ; io < argc ; io++ ) {
      if (argv [io][0] == '-' && argv [io][1] == 'I')
	append_dir_to_rheo_path (argv[io]+2) ;
      else if (strcmp(argv[io],"-app") == 0)
	approx = argv[++io] ;
      else if (strcmp(argv[io],"-hb") == 0)
	do_hb = true;
      else if (strcmp(argv[io],"-matlab") == 0)
	do_matlab = true;
      else if (strcmp(argv[io], "-") == 0) {
	  // input geo on standard input
	  if (mesh_done) usage() ;
	  cerr << " ! geo-dom : geo on stdin\n" ;
	  cin >> g ;
	  mesh_done = true ;
      } else if (!mesh_done) {
	  // input geo on file
	  g = geo(argv[io]) ;
	  mesh_done = true  ;
      } else {
	  // add a domain
	  cerr << "! geo-dom: add domain `" << argv[io] << "'\n" ;
	  io_dom[n_dom]=io ;
	  n_dom++ ;
      }
  }
  // for non-regression outoput, when using bigfloat and such
  numeric_flags<Float>::output_as_double(true);
  // --------------------------------------------------------------------------
  //  Un (quasi-)bug :
  //
  //  domain d1(g["titi"]) ;
  //  domain d2 ;
  //  d2 = d1 ;    // <-- c'est ici que ca coince
  //  cerr << d2 ;
  // --------------------------------------------------------------------------
  if (n_dom==0) {
      cerr << " ! geo-dom : no domains.\n" ;
      exit(1) ;
  }
  domain d(g[argv[io_dom[0]]]) ;
  for (geo::size_type i_dom=1 ; i_dom < n_dom ; i_dom++)
    d.cat(g[argv[io_dom[i_dom]]]) ;
  // --------------------------------------------------------------------------
  //  Les tests
  // --------------------------------------------------------------------------
  geo dg(g,d) ;
  space V_h(g, approx)    ; 
  V_h.freeze() ;
  space S_h(g, d, approx) ; S_h.freeze() ;
  if (!do_matlab) {
      cout << "geo du bord:\n"    << dg  << "\n"
           << "espace du bord:\n" << S_h << "\n" ;
  }
  field u__h(V_h) ;
  for (unsigned int dof = 0 ; dof < u__h.u.size() ; dof++) {
    u__h.u(dof) = Float(dof) ;
  }
  field z__h(S_h) ;
  for (unsigned int dof = 0 ; dof < z__h.u.size() ; dof++) {
    z__h.u(dof) = Float(dof) ;
  }
  trace t_h(V_h, S_h) ;
  form i_h(S_h, S_h, "mass") ;
  form Ab_h(V_h, V_h, "mass", d) ;
  form gamma_h(V_h, S_h, "mass") ;
  if (do_matlab) {
      cout << ml
       << "% Les matrices et vecteurs en matlab:\n" 
       << "th_uu = "       << t_h.uu        << ";\n"
       << "ih_uu = "       << i_h.uu        << ";\n"
       << "Abh_uu = "      << Ab_h.uu       << ";\n"
       << "gammah_uu = "      << gamma_h.uu << ";\n"
       << "th_T_uu = "     << trans(t_h).uu << ";\n"
       << "ih_th_uu = "    << (i_h*t_h).uu  << ";\n"

       << "% La trace de u__h (valeur = n_dof):\n"
       << "th_uh_u = "     << (t_h*u__h).u << ";\n"

       << "% Le vecteur de z__h (valeur = n_dof):\n"
       << "z__h_u = "       << z__h.u << ";\n" 
    ;
  }
  if (do_hb) {
    put_hb (t_h.uu, "t");
    put_hb (i_h.uu, "i");
    put_hb (Ab_h.uu, "Ab");
    put_hb (gamma_h.uu, "gamma");
  }
  field un(S_h) ;
  un = 1. ;
  if (!do_matlab) {
      cout << "dot((i_h*un).u,un.u) = " 
           <<  double( dot((i_h*un).u,un.u) )
           << endl; 
  }
}
