///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is sequential in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/geo.h"
#include "rheolef/geo_connectivity.h"
#include "rheolef/rheostream.h"

namespace rheolef {

// =========================================================================
// geo_rep members
// =========================================================================
template <class T, class M>
geo_rep<T,M>::~geo_rep()
{
  warning_macro ("dstor geo_rep: dim="<<dimension()<<", size="<<size());
}
template <class T, class M>
geo_rep<T,M>::geo_rep (const geo_rep<T,M>& omega2)
{
  error_macro ("physical copy of geo_rep: dim="<<omega2.dimension()<<", par_size="<<omega2.par_size());
}
template <class T, class M>
distributor
geo_rep<T,M>::subgeo_ownership (size_type dim) const
{
  if (dim == _dimension) { return ownership(); }
  switch (dim) {
    case 0: return _idx_vertex.ownership();
    case 1: return _edges.ownership();
    case 2: return _faces.ownership();
    default: {
      error_macro ("unexpected subgeo dimension="<<dim);
      return distributor(); // not reached
    }
  }
}
template <class T, class M>
const geo_element&
geo_rep<T,M>::subgeo (size_type dim, size_type ie) const
{
  if (dim == _dimension) { return operator[] (ie); }
  // edge or face ?
  switch (dim) {
    case 1: return _edges[ie];
    case 2: return _faces[ie];
    default: {
      error_macro ("unsupported subgeo dimension="<<dim);
      geo_element* p = 0; return *p; // not reached
    }
  }
}
template <class T, class M>
const domain_basic<T,M>&
geo_rep<T,M>::operator[] (const std::string& name) const
{
#ifdef TO_CLEAN
  comm().barrier();
#endif // TO_CLEAN
  warning_macro ("domain.size="<<_domains.size());
  for (typename std::vector<domain_basic<T,M> >::const_iterator
        iter = _domains.begin(),
        last = _domains.end(); iter != last; iter++) {
    const domain_basic<T,M>& dom = *iter;
    warning_macro ("looking for domain \""<<name<<": is \"" << dom.name() << "\" ?");
    if (name == dom.name()) return dom;
  }
#ifdef TO_CLEAN
  comm().barrier();
#endif // TO_CLEAN
  error_macro ("undefined domain \""<<name<<"\" in mesh \"" << _name << "\"");
  return *(_domains.begin()); // not reached
}
// =========================================================================
// geo_seq_rep members
// =========================================================================
/// @brief io for geo
template <class T>
iparstream&
geo_seq_rep<T>::get (iparstream& ips, const geo_basic<T,sequential>& smart_ptr_this) {
  using namespace std;
  istream& is = ips.is();
  if (!is.good()) error_macro("bad input stream for geo.");

  if (!scatch(is,"\nmesh"))
    error_macro("input stream does not contains a geo.");
  //
  // 1) get header
  //
  size_type n_vert;
  size_type n_elt;
  size_type n_edg = 0;
  size_type n_fac = 0;

  base::_name = "unnamed";

  is >> base::_version
     >> base::_dimension
     >> n_vert
     >> n_elt;
  if (base::_version < 2) {
    warning_macro ("mesh version < 2 is obsolete");
  } else {
    if (base::_dimension >= 3) {
      is >> n_fac;
    }
    if (base::_dimension >= 2) {
      is >> n_edg;
    }
  }
  //
  // 2) get coordinates
  //
  base::_idx_vertex.resize (n_vert);
  base::_idx_vertex.get (ips, _point_get<T>(geo_rep<T,sequential>::_dimension));
  for (size_type iv = 0; iv < n_vert; iv++) {
    base::_idx_vertex [iv].first = iv;
  }
  if (!is.good()) return ips;
  //
  // 3) get elements
  //
  polymorphic_array<geo_element,sequential>::resize (n_elt);
  polymorphic_array<geo_element,sequential>::get (ips);
  for (size_type ie = 0; ie < n_elt; ie++) {
    geo_element& K = operator[] (ie);
    K.set_par_old_ie (ie);
  }
  //
  // 4) get faces & edges
  //
  if   (base::_version  >= 2 && base::_dimension >= 3) {
    base::_faces.resize (n_fac);
    base::_faces.get (ips);
    for (size_type ifac = 0; ifac < n_fac; ifac++) {
      geo_element& F = base::_faces [ifac];
      F.set_par_old_ie (ifac);
    }
  }
  if   (base::_version  >= 2 && base::_dimension >= 2) {
    base::_edges.resize (n_edg);
    base::_edges.get (ips);
    for (size_type iedg = 0; iedg < n_edg; iedg++) {
      geo_element& E = base::_edges [iedg];
      E.set_par_old_ie (iedg);
    }
  }
  //
  // 5) get domain, until end-of-file
  //
  vector<set<size_type> > ball [4];
  domain_basic<T,sequential> dom;
  while (dom.get (ips, smart_ptr_this, ball)) {
     base::_domains.push_back (dom);
  }
  return ips;
}
template <class T>
oparstream&
geo_seq_rep<T>::put (oparstream& ops)  const {
  using namespace std;
  size_type n_vert = geo_rep<T,sequential>::_idx_vertex.par_size ();
  size_type n_elt  = polymorphic_array<geo_element,sequential>::par_size ();
  //
  // put header
  //
  ops << "#!geo" << endl
      << endl
      << "mesh" << endl
      << geo_rep<T,sequential>::_version << " " 
      << geo_rep<T,sequential>::_dimension << " " 
      << n_vert << " "
      << n_elt;
  if   (base::_version >= 2) {
    if (base::_dimension >= 3) {
      ops << " " << base::_faces.par_size();
    }
    if (base::_dimension >= 2) {
      ops << " " << base::_edges.par_size();
    }
  }
  ops << endl << endl;
  //
  // put vertices & elements
  //
  geo_rep<T,sequential>::_idx_vertex.put (ops, _point_put<T>(geo_rep<T,sequential>::_dimension));
  ops << endl;
  polymorphic_array<geo_element,sequential>::put (ops);
  ops << endl;
  //
  // put faces & edges
  //
  if   (base::_version >= 2 && base::_dimension >= 3) {
      base::_faces.put (ops); 
  }
  if   (base::_version >= 2 && base::_dimension >= 2) {
      base::_edges.put (ops); 
  }
  //
  // put domains
  //
  for (typename std::vector<domain_basic<T,sequential> >::const_iterator
        iter = base::_domains.begin(), last = base::_domains.end();
	iter != last; ++iter) {
    ops << endl;
    (*iter).put (ops);
  }
  return ops;
}
template <class T>
void
geo_seq_rep<T>::dump (std::string name)  const {
  geo_rep<T,sequential>::_idx_vertex.dump        (name + "-vert");
  polymorphic_array<geo_element,sequential>::dump(name + "-elem");
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class geo_rep<Float,sequential>;
#ifdef _RHEOLEF_HAVE_MPI
template class geo_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI
template class geo_seq_rep<Float>;
} // namespace rheolef
