///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is sequential in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/domain.h"
#include "rheolef/geo.h"
#include "rheolef/geo_connectivity.h"

namespace rheolef {

// ----------------------------------------------------------------------------
/// side and orientation
// ----------------------------------------------------------------------------
std::istream&
operator>> (std::istream& is, domain_pair_type& x)
{
    char c;
    is >> std::ws >> c;
    if (isdigit(c)) { 
      // positive orientation:
      x._pos = true;
      is.unget(); // or is.putback(c); ?
      return is >> x._isg;
    } else {
      // negative orientation:
      x._pos = false;
      return is >> x._isg;
    }
}
std::ostream&
operator<< (std::ostream& os, const domain_pair_type& x)
{
    if (! x._pos) os << '-';
    return os << x._isg;
}
// ----------------------------------------------------------------------------
/// @brief generic (seq,mpi) i/o in format version 2
// ----------------------------------------------------------------------------
#ifdef TO_CLEAN
template<class T>
iparstream&
domain_seq_rep<T>::get (iparstream& ips)
{
  //
  // get header
  //
  std::istream& is = ips.is();
  if (!scatch(is,"\ndomain")) {
    is.setstate (std::ios::badbit);
    return ips;
  }
  size_type version, n_side;
  ips >> base::_name
      >> version >> base::_dim >> n_side;
  check_macro (version == 2, "unsupported version="<<version<<" domain format");
  //
  // get data
  //
  warning_macro ("domain "<<base::_name<<" "<<n_side<<"...");
  array<domain_pair_type,sequential>::resize (n_side);
  array<domain_pair_type,sequential>::get (ips);
  warning_macro ("domain "<<base::_name<<" done");
  return ips;
}
#endif // TO_CLEAN
template<class T>
oparstream&
domain_seq_rep<T>::put (oparstream& ops)  const {
  using namespace std;
  ops << "domain" << endl
      << base::_name << endl
      << "2 " << base::_dim << " " << size() << endl;
  array<domain_pair_type,sequential>::put (ops);
  return ops;
}
// ----------------------------------------------------------------------------
/// @brief sequential i/o in format version 1 or 2 (backward compatibility)
// ----------------------------------------------------------------------------
template<class T>
iparstream&
domain_seq_rep<T>::get (
    iparstream& ips,
    const geo_basic<T,sequential>& smart_ptr_omega,
    std::vector<std::set<size_type> > ball[4])
{ 
  base::_ptr_omega = &smart_ptr_omega;
  const geo_seq_rep<T>& omega = smart_ptr_omega.data();
  
  std::istream& is = ips.is();
  if (!scatch(is,"\ndomain")) {
    is.setstate (std::ios::badbit);
    return ips;
  }
  size_type version, n_side;
  is >> base::_name >> version >> base::_dim >> n_side;
  array<domain_pair_type,sequential>::resize (n_side);
  check_macro (version <= 2, "unexpected domain format version="<<version);
  if (version == 2 || base::_dim == 0) {
    array<domain_pair_type,sequential>::get (ips);
    return ips;
  }
  check_macro (version == 1, "unexpected domain format version="<<version);
  check_macro (base::_dim <= omega.dimension(), "unexpected domain dimension="
	<<base::_dim<<" > geometry dimension = " << omega.dimension());
  //
  // old version=1 format: searh for index of sides
  //
  warning_macro ("domain "<<base::_name<<" "<<n_side<<"...");
  if (ball[base::_dim].size() == 0) {
    warning_macro ("ball["<<base::_dim<<"]...");
    ball[base::_dim].resize (omega.n_vertex());
    // check dimension and switch (edge,face,element)
    if (base::_dim == omega.dimension()) {
          build_point_to_element_sets (omega.begin(), omega.end(), ball[base::_dim].begin());
    } else {
      switch (base::_dim) {
        case 1:
          build_point_to_element_sets (omega.begin_edge(), omega.end_edge(), ball[base::_dim].begin());
          break;
        case 2:
          build_point_to_element_sets (omega.begin_face(), omega.end_face(), ball[base::_dim].begin());
          break;
        default:
          error_macro ("domain dimension=" << base::_dim << " not yet supported");
      }
    }
    warning_macro ("ball["<<base::_dim<<"] done");
  }
  polymorphic_array<geo_element,sequential> d_tmp (n_side);
  d_tmp.get (ips);
  polymorphic_array<geo_element,sequential>::const_iterator q = d_tmp.begin();
  for (iterator_ioisg p = begin_ioisg(), last = end_ioisg(); p != last; ++p, ++q) {
    const geo_element& S = *q;
    std::set<size_type> contains_S;
    build_set_that_contains_S (S, ball[base::_dim], contains_S);
    check_macro (contains_S.size() >= 1, "domain element not in mesh: " << S);
    check_macro (contains_S.size() <= 1, "problem with domain element: " << S);
    // now, choose the only one mesh (sub-)element matching domain element S
    size_type i_side = *(contains_S.begin());
    const geo_element& S_orig = omega.subgeo (base::_dim, i_side);
    geo_element::orientation_type orient;
    size_type shift;
    check_macro (S_orig.get_orientation_and_shift (S, orient, shift),
    			"problem with domain element: " << S);
    (*p).set (orient, i_side);
    warning_macro ("domain element: "
	<< S << " -> index = " << i_side << ", orient = " << orient);
  }
  warning_macro ("domain " << base::_name << " done");
  return ips;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class domain_seq_rep<Float>;

} // namespace rheolef
