///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// visualisation of geometries
//
// authors:
//	Nicolas.Roquet@imag.fr
//    	Pierre.Saramito@imag.fr
//
// date: 12 may 1997
//
// content:
//      * graphic output (with options; see geoplot(1))
//            - gnuplot
//            - plotmtv
//            - atom
//            - x3d
//            - vtk
//      * get vtk polydata file (line or surface )
//        as for a cut in a mesh
//
#include "rheolef/geo-visu-aux.h"
#include <sstream>
using namespace rheolef;
using namespace std;

namespace rheolef { 

string
python (const point& x, size_t d) 
{
    ostringstream os;
    os << "[" << x[0];
    for (size_t i = 1; i < d; i++)
      os << ", " << x[i];
    os << "]" << flush;
    string buffer = os.str();
    return buffer;
}

void 
gnuplot_point (ostream& s, const point& p, georep::size_type dim)
{
    p.put(s, dim); s << endl << endl;
}
void 
gnuplot_segment (ostream& s, const point& p, const point& q, georep::size_type dim)
{
    p.put(s, dim); s << endl;
    q.put(s, dim); s << endl << endl;
}
void 
plotmtv_point (ostream& s, const point& p,
        georep::size_type dim, 
        georep::size_type color,
        georep::size_type width)
{   
    s << "% markertype  = " << color << endl;
    s << "% markercolor = " << color << endl;
    s << "% linecolor = -1" << endl;
    s << "% dlinetype = 0" << endl;
    p.put(s, dim); s << endl;
    s << "% dlinetype = 1" << endl;
    s << endl;
}
void 
plotmtv_segment (ostream& s, const point& p, const point& q, 
	georep::size_type dim, 
	georep::size_type color, 
	georep::size_type width)
{
    if (color != 1) s << "% linecolor = " << color << endl;
    if (width != 1) s << "% linewidth = " << width << endl;
    p.put(s, dim); s << endl;
    q.put(s, dim); s << endl << endl;
}
void
atom_segment (ostream& s, const point& p, const point& q, georep::size_type dim, 
                          Float radius, georep::size_type color)
{
    switch (dim) {
    case 1: p.put(s, 1); s << " 0 0 ";
            q.put(s, 1); s << " 0 0 r=" << radius << " c=" << color << "\n";
	    break;
    case 2: p.put(s, 2); s << " 0 ";
            q.put(s, 2); s << " 0 r=" << radius << " c=" << color << "\n";
	    break;
    case 3: p.put(s, 3); s << " ";
            q.put(s, 3); s << " r=" << radius << " c=" << color << "\n";
	    break;
    }
}
void
x3d_face (ostream& s, const point& p, const point& q, const point& r, georep::size_type dim, 
                          Float scale, const point& x0, const point& dx)
{
    point p1 = scale*(p-x0)/dx;
    point q1 = scale*(q-x0)/dx;
    point r1 = scale*(r-x0)/dx;

    s << "   255 255 255\n";
    switch (dim) {
    case 2: p1.put(s, 2); s << " 0\n";
            q1.put(s, 2); s << " 0\n";
            r1.put(s, 2); s << " 0\n\n\n";
            break;
    case 3 :
    default:p1.put(s, 3); s << "\n";
            q1.put(s, 3); s << "\n";
            r1.put(s, 3); s << "\n\n\n";
            break;
    }
}
void
x3d_face (ostream& s, const point& p, const point& q, const point& r, const point& u, 
		georep::size_type dim, Float scale, const point& x0, const point& dx)
{
    point p1 = scale*(p-x0)/dx;
    point q1 = scale*(q-x0)/dx;
    point r1 = scale*(r-x0)/dx;
    point u1 = scale*(u-x0)/dx;

    s << "   255 255 255\n";
    switch (dim) {
    case 2: p1.put(s, 2); s << " 0\n";
            q1.put(s, 2); s << " 0\n";
            r1.put(s, 2); s << " 0\n";
            u1.put(s, 2); s << " 0\n\n\n";
            break;
    case 3 :
    default:p1.put(s, 3); s << "\n";
            q1.put(s, 3); s << "\n";
            r1.put(s, 3); s << "\n";
            u1.put(s, 3); s << "\n\n\n";
            break;
    }
}
void gnuplot_element(ostream& s, const geo_element& g, georep::const_iterator_node x, georep::size_type dim)
{
  switch (g.name()) {

  case 'p' : {
	    gnuplot_point (s, x[g[0]], dim);
	    break ;
  	    }
  case 'e' : {
	    gnuplot_segment (s, x[g[0]], x[g[1]], dim);
	    break ;
	    }
  case 't' : {
	    gnuplot_segment (s, x[g[0]], x[g[1]], dim);
	    gnuplot_segment (s, x[g[1]], x[g[2]], dim);
	    gnuplot_segment (s, x[g[2]], x[g[0]], dim);
	    break ;
	    }
  case 'q' : {
	    gnuplot_segment (s, x[g[0]], x[g[1]], dim);
	    gnuplot_segment (s, x[g[1]], x[g[2]], dim);
	    gnuplot_segment (s, x[g[2]], x[g[3]], dim);
	    gnuplot_segment (s, x[g[3]], x[g[0]], dim);
	    break ;
	    }
  case 'T' : {
	    gnuplot_segment (s, x[g[0]], x[g[1]], dim); s << endl;
	    gnuplot_segment (s, x[g[1]], x[g[2]], dim); s << endl;
	    gnuplot_segment (s, x[g[2]], x[g[0]], dim); s << endl;
	    gnuplot_segment (s, x[g[0]], x[g[3]], dim); s << endl;
	    gnuplot_segment (s, x[g[1]], x[g[3]], dim); s << endl;
	    gnuplot_segment (s, x[g[2]], x[g[3]], dim); s << endl;
	    break ;
	    }
  case 'P' : {
	    gnuplot_segment (s, x[g[0]], x[g[1]], dim); s << endl;
	    gnuplot_segment (s, x[g[1]], x[g[2]], dim); s << endl;
	    gnuplot_segment (s, x[g[2]], x[g[0]], dim); s << endl;
	    gnuplot_segment (s, x[g[3]], x[g[4]], dim); s << endl;
	    gnuplot_segment (s, x[g[4]], x[g[5]], dim); s << endl;
	    gnuplot_segment (s, x[g[5]], x[g[3]], dim); s << endl;
	    gnuplot_segment (s, x[g[0]], x[g[3]], dim); s << endl;
	    gnuplot_segment (s, x[g[1]], x[g[4]], dim); s << endl;
	    gnuplot_segment (s, x[g[2]], x[g[5]], dim); s << endl;
	    break ;
	    }
  case 'H' : {
	    gnuplot_segment (s, x[g[0]], x[g[1]], dim); s << endl;
	    gnuplot_segment (s, x[g[1]], x[g[2]], dim); s << endl;
	    gnuplot_segment (s, x[g[2]], x[g[3]], dim); s << endl;
	    gnuplot_segment (s, x[g[3]], x[g[0]], dim); s << endl;
	    gnuplot_segment (s, x[g[4]], x[g[5]], dim); s << endl;
	    gnuplot_segment (s, x[g[5]], x[g[6]], dim); s << endl;
	    gnuplot_segment (s, x[g[6]], x[g[7]], dim); s << endl;
	    gnuplot_segment (s, x[g[7]], x[g[4]], dim); s << endl;
	    gnuplot_segment (s, x[g[0]], x[g[4]], dim); s << endl;
	    gnuplot_segment (s, x[g[1]], x[g[5]], dim); s << endl;
	    gnuplot_segment (s, x[g[2]], x[g[6]], dim); s << endl;
	    gnuplot_segment (s, x[g[3]], x[g[7]], dim); s << endl;
	    break ;
	    }
  default  : {
	    error_macro("geo_element: unexpected type `" << int(g.type())
		 << "' and dimension "               << dim);
	    }
  }
}
void
plotmtv_element(ostream& s, const geo_element& g, georep::const_iterator_node x,
	georep::size_type dim,
	georep::size_type color,
	georep::size_type width)
{
  switch (g.name()) {

  case 'p' : {
	    plotmtv_point (s, x[g[0]], dim, color, width);
	    break ;
  	    }
  case 'e' : {
	    plotmtv_segment (s, x[g[0]], x[g[1]], dim, color, width);
	    break ;
	    }
  case 't' : {
	    plotmtv_segment (s, x[g[0]], x[g[1]], dim, color, width);
	    plotmtv_segment (s, x[g[1]], x[g[2]], dim, color, width);
	    plotmtv_segment (s, x[g[2]], x[g[0]], dim, color, width);
	    break ;
	    }
  case 'q' : {
	    plotmtv_segment (s, x[g[0]], x[g[1]], dim, color, width);
	    plotmtv_segment (s, x[g[1]], x[g[2]], dim, color, width);
	    plotmtv_segment (s, x[g[2]], x[g[3]], dim, color, width);
	    plotmtv_segment (s, x[g[3]], x[g[0]], dim, color, width);
	    break ;
	    }
  case 'T' : {
	    plotmtv_segment (s, x[g[0]], x[g[1]], dim, color, width); s << endl;
	    plotmtv_segment (s, x[g[1]], x[g[2]], dim, color, width); s << endl;
	    plotmtv_segment (s, x[g[2]], x[g[0]], dim, color, width); s << endl;
	    plotmtv_segment (s, x[g[0]], x[g[3]], dim, color, width); s << endl;
	    plotmtv_segment (s, x[g[1]], x[g[3]], dim, color, width); s << endl;
	    plotmtv_segment (s, x[g[2]], x[g[3]], dim, color, width); s << endl;
	    break ;
	    }
  default  : {
	    error_macro("geo_element: unexpected type `" << int(g.type())
		 << "' and dimension "               << dim);
	    }
  }
}
void 
atom_element(ostream& s, const geo_element& g, georep::const_iterator_node x, georep::size_type dim, 
                  Float r, georep::size_type c)
{
  switch (g.name()) {

  case 'p' : {
	    break;
	    }
  case 'e' : {
	    atom_segment (s, x[g[0]], x[g[1]], dim, r, c);
	    break ;
	    }
  case 't' : {
	    atom_segment (s, x[g[0]], x[g[1]], dim, r, c);
	    atom_segment (s, x[g[1]], x[g[2]], dim, r, c);
	    atom_segment (s, x[g[2]], x[g[0]], dim, r, c);
	    break ;
	    }
  case 'q' : {
	    atom_segment (s, x[g[0]], x[g[1]], dim, r, c);
	    atom_segment (s, x[g[1]], x[g[2]], dim, r, c);
	    atom_segment (s, x[g[2]], x[g[3]], dim, r, c);
	    atom_segment (s, x[g[3]], x[g[4]], dim, r, c);
	    break ;
	    }
  case 'T' : {
	    atom_segment (s, x[g[0]], x[g[1]], dim, r, c);
	    atom_segment (s, x[g[1]], x[g[2]], dim, r, c);
	    atom_segment (s, x[g[2]], x[g[0]], dim, r, c);
	    atom_segment (s, x[g[0]], x[g[3]], dim, r, c);
	    atom_segment (s, x[g[1]], x[g[3]], dim, r, c);
	    atom_segment (s, x[g[2]], x[g[3]], dim, r, c);
	    break ;
	    }
  default  : {
	    error_macro("atom_element: unexpected type `" << int(g.type())
		 << "' and dimension " << dim);
	    }
  }
}
void x3d_element(ostream& s, const geo_element& g, georep::const_iterator_node x, georep::size_type dim,
                  Float scale, const point& x0, const point& dx)
{
  switch (g.name()) {

  case 'p' :
  case 'e' :
	    break;

  case 't' : x3d_face (s, x[g[0]], x[g[1]], x[g[2]], dim, scale, x0, dx);
	     break ;
  case 'q' : x3d_face (s, x[g[0]], x[g[1]], x[g[2]], x[g[3]], dim, scale, x0, dx);
	     break ;
  case 'T' : x3d_face (s, x[g[0]], x[g[2]], x[g[1]], dim, scale, x0, dx);
             x3d_face (s, x[g[0]], x[g[3]], x[g[2]], dim, scale, x0, dx);
             x3d_face (s, x[g[0]], x[g[1]], x[g[3]], dim, scale, x0, dx);
             x3d_face (s, x[g[1]], x[g[2]], x[g[3]], dim, scale, x0, dx);
	     break ;
  default  : {
	    error_macro("x3d_element: unexpected type `" << int(g.type())
		 << "' and dimension " << dim);
	     }
  }
}
}// namespace rheolef
