///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// contents:
//      * vtk output: 
//
// authors:
//      Pierre.Saramito@imag.fr
//
// date: 12 may 1997  update: 6 may 2001
//

#include "rheolef/field.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h"
#include "rheolef/tiny_matvec.h"
using namespace std;
namespace rheolef { 

// from field.c, internal:
istream& load_field_data (istream& s, field& x);

void
field::put_vtk (ostream& vtk, const string& name, bool put_header) const
{
  check_macro(n_component() == 1, "unsupported " << n_component() << "-component field");
  check_macro(get_approx() == "P0" || get_approx() == "P1"
	|| get_approx() == "P1d" || get_approx() == "P2", 
	"unsupported `" << get_approx() << "' approximation");
  if (put_header) {
    if (get_approx() == "P0") {
	vtk << "CELL_DATA " << size() << endl;
    } else {
	vtk << "POINT_DATA " << size() << endl;
    }
  }
  vtk << "SCALARS " << name << " float" << endl
      << "LOOKUP_TABLE default"  << endl;
  for (size_type i = 0; i < size(); i++) vtk << at(i) << endl;
  vtk << endl;
}
//
// plot 2D P1-field as elevation in 3D
//
ostream&
field::put_vtk_2d_elevation_P1 (ostream& vtk) const
{
    // inspirated by "vtkdata/honolulu.vtk"

    if (dimension() != 2) {
       error_macro ("put_vtk_2d_elevation_P1: dimension 2 expected, " << dimension() << " founded");
    }
    if (get_approx() != "P1") {
       error_macro ("put_vtk_2d_elevation_P1: P1 approximation expected");
    }
    int digits10 = numeric_limits<Float>::digits10;
    vtk << setprecision(digits10)
        << "# vtk DataFile Version 1.0" << endl
        << "automatically generated by rheolef" << endl
        << "ASCII" << endl
        << "DATASET POLYDATA" << endl
        << "POINTS " << size() << " float" << endl;
 
    const geo& g = get_geo(); 
    geo::const_iterator_node p = g.begin_node();
    for (size_type i = 0; i < size(); i++) {
	vtk << p[i][0] << " " << p[i][1] << " " << at(i) << endl;
    }
    size_type n_line  =   g.n_triangle() +   g.n_quadrangle();
    size_type n_entry = 3*g.n_triangle() + 4*g.n_quadrangle() + n_line;
    vtk << "POLYGONS " << n_line << " " << n_entry << endl;
    for (geo::const_iterator q = g.begin(); q != g.end(); q++) {
        const geo_element& K = *q;
	switch (K.type()) {
	  case reference_element::t:
	    vtk << "3 " << K[0] << " " << K[1] << " " << K[2] << endl;
	    break;
	  case reference_element::q:
	    vtk << "4 " << K[0] << " " << K[1] << " " << K[2] << " " << K[3] << endl;
	    break;
	  default:
	    break;
        }
    }
    return vtk;
}
int
field::vtk2d_Pk (const string& basename, bool execute, bool clean, bool verbose,
				       bool grid, bool fill) const
{
  //
  // create vtk data file
  //
  bool use_colorbar = true;
  string vtk_name = basename + ".vtk";
  ofstream vtk (vtk_name.c_str());
  if (verbose) clog << "! file \"" << vtk_name << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  vtk << setprecision(digits10);
  if (get_approx() == "P1") {
     vtk << vtkdata << get_geo();
     // get_geo().put_vtk(vtk);
  } else if (get_approx() == "P2") {
     vtk << lattice << vtkdata << get_geo();
     // get_geo().put_vtk_P2_iso_P1 (vtk);
  } else {
     fatal_macro ("vtk_3d_Pk: P1 or P2 approximation expected");
  }
  put_vtk(vtk);
  vtk.close();
  //
  // output tcl script
  //
  string tcl_name = basename + ".tcl";
  ofstream tcl (tcl_name.c_str());
  if (verbose) clog << "! file \"" << tcl_name << "\" created.\n";
  tcl << setprecision(digits10);
       
  tcl << "# this is a tcl version of " << basename << ".vtk visualization\n"
      << "source \"" << _RHEOLEF_PKGDATADIR << "/vtk_interactor.tcl\"\n"
      << endl
      << "vtkUnstructuredGridReader reader;\n"
      << "    reader SetFileName \"" << basename << ".vtk\"\n"
      << "    reader Update;\n"
  ;
  string mapper;
  if (!fill) {
    tcl << "#\n"
        << "# values: iso lines\n"
        << "#\n"
        << "vtkContourFilter iso_lines;\n"
        << "    iso_lines SetInput [reader GetOutput];\n"
        << "    eval iso_lines GenerateValues 50 [[reader GetOutput] GetScalarRange];\n"
        << "    iso_lines Update;\n"
        << "vtkPolyDataMapper map_iso_lines;\n"
        << "    map_iso_lines SetInput [iso_lines GetOutput]; \n"
        << "    eval map_iso_lines SetScalarRange [[reader GetOutput] GetScalarRange];\n"
        << "vtkActor iso_lines_actor;\n"
        << "    iso_lines_actor SetMapper map_iso_lines;\n"
    ;
    mapper = "map_iso_lines";
  } else {
    tcl << "#\n"
        << "# values: fill\n"
        << "#\n"
        << "vtkDataSetMapper map_fill;\n"
        << "    map_fill SetInput [reader GetOutput];\n"
        << "    eval map_fill SetScalarRange [[reader GetOutput] GetScalarRange];\n"
        << "vtkActor fill_actor;\n"
        << "    fill_actor SetMapper map_fill;\n"
    ;
    mapper = "map_fill";
  }
  if (grid) {
    tcl << "#\n"
        << "# show grid\n"
        << "#\n"
        << "vtkDataSetMapper map_grid;\n"
        << "    map_grid SetInput [reader GetOutput];\n"
        << "    map_grid ScalarVisibilityOff;\n"
        << "vtkActor grid_actor;\n"
        << "    grid_actor SetMapper map_grid;\n"
        << "    eval [grid_actor GetProperty] SetColor 1 0.49 0.25; # flesh\n"
        << "    eval [grid_actor GetProperty] SetRepresentationToWireframe;\n"
    ;
  }
  if (use_colorbar) {
    tcl << "#\n"
        << "# the color bar\n"
        << "#\n"
        << "vtkScalarBarActor bar_actor\n"
        << "    bar_actor SetLookupTable [" << mapper << " GetLookupTable]\n"
        << "    bar_actor SetOrientationToHorizontal\n"
        << "    [bar_actor GetProperty] SetColor 0 0 0\n"
        << "    [bar_actor GetPositionCoordinate] SetCoordinateSystemToNormalizedViewport\n"
        << "    [bar_actor GetPositionCoordinate] SetValue 0.1 0.01\n"
        << "    bar_actor SetOrientationToHorizontal\n"
        << "    bar_actor SetWidth  0.8\n"
        << "    bar_actor SetHeight 0.10\n"
        << "    bar_actor SetTitle \"" << basename << "\"\n"
    ;
  }
  tcl << "#\n"
      << "# Create rendering stuff\n"
      << "#\n"
      << "vtkRenderWindow render_window;\n"
      << "vtkRenderer ren1;\n"
      << "    render_window AddRenderer ren1;\n"
      << "vtkRenderWindowInteractor iren;\n"
      << "    iren SetRenderWindow render_window;\n"
      << "\n"
  ;
  if (!fill) {
    tcl << "ren1 AddActor iso_lines_actor;\n";
  } else {
    tcl << "ren1 AddActor fill_actor;\n";
  }
  if (grid) {
    tcl << "ren1 AddActor grid_actor;\n";
  }
  if (use_colorbar) {
    tcl << "ren1 AddActor bar_actor;\n";
  }
  tcl << "\n"
      << "render_window SetSize 500 500;\n"
      << "render_window SetWindowName \"" << basename << "\";\n"
      << "ren1 SetBackground 1 1 1; \n"
      << "#\n"
      << "# render the image\n"
      << "#\n"
      << "if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {\n"
      << "    iren AddObserver UserEvent {wm deiconify .vtkInteract};\n"
      << "} else {\n"
      << "    iren SetUserMethod {wm deiconify .vtkInteract};\n"
      << "}\n"
      << "render_window Render;\n"
      << "#\n"
      << "# prevent the tk window from showing up then start the event loop\n"
      << "#\n"
      << "wm withdraw .\n"
  ;
  tcl.close();
  //
  // run vtk
  //
  int status = 0;
  char command [1000];
  if (execute) {
      sprintf (command, "vtk %s.tcl", basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status = system (command);
  }
  //
  // clear vtk data
  //
  if (clean) {
      sprintf (command, "/bin/rm -f %s.vtk %s.tcl", basename.c_str(), basename.c_str());
      if (verbose) clog << "! " << command << endl;
      status |= system (command);
  }
  return status;
}
int
field::vtk_elevation (
	const string& basename, bool execute, bool clean, bool verbose,
        bool grid, bool fill) const
{
  //
  // create vtk data file
  //
  string vtk_name = basename + ".vtk";
  ofstream vtk (vtk_name.c_str());
  if (verbose) clog << "! file \"" << vtk_name << "\" created.\n";
  put_vtk_2d_elevation_P1(vtk);
  vtk.close();
  //
  // create tcl script
  //
  string tcl_name  = basename + ".tcl";
  ofstream tcl (tcl_name.c_str());
  if (verbose) clog << "! file \"" << tcl_name << "\" created.\n";
  tcl << "#!vtk\n"
      << "# this is a tcl script for data \"" << vtk_name  << "\"\n"
      << "# that runs evelation rendering.\n"
      << "#\n"
      << "set file_name        \"" << vtk_name << "\";\n"
      << "source \"" << _RHEOLEF_PKGDATADIR << "/vtk_interactor.tcl\";\n"
      << "source \"" << _RHEOLEF_PKGDATADIR << "/vtk_elevation.tcl\";\n"
      ;
  tcl.close();
  //
  // run vtk
  //
  string command;
  int status = 0;
  if (execute) {
      command = "vtk " + tcl_name;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
      if (status != 0) 
	error_macro ("\"" << command << "\" command failed.\n");
  }
  //
  // clear tcl/vtk data
  //
  if (clean) {
      command = "/bin/rm -f " + tcl_name + " " + vtk_name;
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}
field
field::cut (
    const string& basename, 
    bool execute, bool clean, bool verbose,
    point normal, point origin
    ) const
{
  //
  // create tvk data file
  //
  string vtk_name = basename + ".vtk";
  ofstream vtk (vtk_name.c_str());
  if (verbose) clog << "! file \"" << vtk_name << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  vtk << setprecision(digits10);
  if (get_approx() == "P1") {
     vtk << vtkdata << get_geo();
  } else if (get_approx() == "P2") {
     vtk << lattice << vtkdata << get_geo();
  } else {
     fatal_macro ("vtk_3d_Pk: P1 or P2 approximation expected");
  }
  put_vtk(vtk);
  vtk.close();
  //
  // create tcl script
  //
  string tcl_name  = basename + "-do-cut.tcl";
  string poly_name = basename + "-cut.vtk";
  ofstream tcl (tcl_name.c_str());
  if (verbose) clog << "! file \"" << tcl_name << "\" created.\n";
  tcl << setprecision(digits10);
  tcl << "#!vtk\n"
      << "# this is a tcl version for field \"" << vtk_name << "\"\n"
      << "# that generates a plane cut in   \"" << poly_name << "\"\n"
      << "#\n"
      << "# load mesh and field\n"
      << "vtkUnstructuredGridReader reader;\n"
      << "    reader SetFileName \"" << basename << ".vtk\"\n"
      << "    reader Update;\n"
      << "#\n"
      << "# cut it by plane\n"
      << "#\n"
      << "vtkPlane plane;\n"
      << "    eval plane SetOrigin [[reader GetOutput] GetCenter];\n"
      << "    plane SetNormal " << normal << ";\n"
      << "    plane SetOrigin " << origin << ";\n"
      << "vtkCutter plane_cut;\n"
      << "    plane_cut SetInput [reader GetOutput];\n"
      << "    plane_cut SetCutFunction plane;\n"
      << "#\n"
      << "# merge points and force triangles\n"
      << "#\n"
      << "vtkCleanPolyData clean_plane;\n"
      << "    clean_plane SetInput [plane_cut GetOutput];\n"
      << "    clean_plane SetTolerance " << ::sqrt(numeric_limits<Float>::epsilon()) << ";\n"
      << "vtkCastToConcrete cast_filter\n"
      << "    cast_filter SetInput [clean_plane GetOutput]\n"
      << "vtkTriangleFilter triangle_filter\n"
      << "    triangle_filter SetInput [cast_filter GetPolyDataOutput]\n"
      << "#\n"
      << "# write polygons and associated\n"
      << "#\n"
      << "vtkDataSetWriter writer;\n"
      << "  writer SetInput [triangle_filter GetOutput];\n"
      << "  writer SetFileName \"" << poly_name << "\";\n"
      << "  writer Write;\n"
      << "exit;\n"
      ;
  tcl.close();
  //
  // run vtk
  //
  string command;
  // WARNING: Execute whenever, cause executing is not interactive viewing here !!
  // if (execute) {
      command = "vtk " + tcl_name;
      if (verbose) clog << "! " << command << endl;
      int status = system (command.c_str());
      if (status != 0) 
	error_macro ("\"" << command << "\" command failed.\n");
  // }
  //
  // clear tcl/vtk data
  //
  if (clean) {
      command = "/bin/rm -f " + tcl_name + " " + vtk_name;
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  //  
  // load polygonal data, build coordinates
  //
  ifstream poly(poly_name.c_str());
  if (!poly) error_macro("field: vtk polydata file \"" << poly_name << "\" not found.");
  geo g_cut; 
  if (verbose) clog << "! load `" << poly_name << "'.\n";
  poly >> vtkpolydata >> g_cut;
  g_cut.set_name (basename + "-cut");
  //
  // clear vtk polydata (cut)
  //
  if (clean) {
      command = "/bin/rm -f " + poly_name;
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  //
  // project from 2d/3d on plane in 1d/2d
  // 
  //   1D) x := dot(OM, tangent)
  //
  //   2D) x1 := dot(OM, t1)
  //       x2 := dot(OM, t2)
  bool use_projection = true;
  if (use_projection) {

    // can construct a field
    if (dimension() == 2) {

      // project in 1D space
      g_cut.set_dimension (1);
      point tangent = point(normal[1], -normal[0]);
      geo::iterator_node last_p = g_cut.end_node();  
      for (geo::iterator_node iter_p = g_cut.begin_node(); iter_p != last_p; iter_p++)
	*iter_p = point (dot(*iter_p, tangent), 0);
    } else {
      warning_macro ("3d plane cut projected in 2d...");
      g_cut.set_dimension (2);
      point t1 = point(1, 0, 0);
      if (norm(vect(t1, normal)) == Float(0)) t1 = point(0, 1, 0);
      t1 = t1 - dot(t1,normal)*normal;
      t1 = t1 / norm(t1);
      point t2 = vect(normal,t1);
      geo::iterator_node last_p = g_cut.end_node();  
      for (geo::iterator_node iter_p = g_cut.begin_node(); iter_p != last_p; iter_p++)
	*iter_p = point (dot(*iter_p - origin, t1), dot(*iter_p - origin, t2), 0);
    }
  }
  // upgrade: set edge/face connect before to call space()
  g_cut.upgrade();
  space V_cut (g_cut, "P1");
  field u_cut (V_cut);
  // polydata header:
  //     POINT_DATA <n_node>
  //     SCALARS scalars float
  //     LOOKUP_TABLE default
  scatch (poly, "POINT_DATA");
  scatch (poly, "SCALARS");
  scatch (poly, "default");
  load_field_data (poly, u_cut);
  return u_cut;
}
geo
field::iso (
    const string& basename, 
    bool execute, bool clean, bool verbose,
    const Float value
    ) const
{
  //
  // create tvk data file
  //
  string vtk_name = basename + ".vtk";
  ofstream vtk (vtk_name.c_str());
  if (verbose) clog << "! file \"" << vtk_name << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  vtk << setprecision(digits10);
  if (get_approx() == "P1") {
     vtk << vtkdata << get_geo();
  } else if (get_approx() == "P2") {
     vtk << lattice << vtkdata << get_geo();
  } else {
     fatal_macro ("iso: P1 or P2 approximation expected");
  }
  put_vtk(vtk);
  vtk.close();
  //
  // create tcl script
  //
  string tcl_name  = basename + "-do-iso.tcl";
  string poly_name = basename + "-iso.vtk";
  ofstream tcl (tcl_name.c_str());
  if (verbose) clog << "! file \"" << tcl_name << "\" created.\n";
  tcl << setprecision(digits10)
      << "#!vtk\n"
      << "# this is a tcl version for field \"" << vtk_name  << "\"\n"
      << "# that generates an isosurface in \"" << poly_name << "\"\n"
      << "#\n"
      << "set iso_value " << value << "\n"
      << "#\n"
      << "# load mesh and field\n"
      << "#\n"
      << "vtkUnstructuredGridReader reader;\n"
      << "    reader SetFileName \"" << vtk_name << "\"\n"
      << "    reader Update;\n"
      << "#\n"
      << "# compute iso lines or surfaces\n"
      << "#\n"
      << "vtkContourFilter iso_filter\n"
      << "    iso_filter SetInput [reader GetOutput]\n"
      << "    iso_filter SetValue 0 $iso_value\n"
      << "    iso_filter ComputeGradientsOff;\n"
      << "    iso_filter ComputeNormalsOff;\n"
      << "    iso_filter ComputeScalarsOff;\n"
      << "vtkPolyDataNormals normals\n"
      << "    normals SetInput [iso_filter GetOutput]\n"
      << "    normals SetFeatureAngle 45\n"
      << "#\n"
      << "# merge points and force triangles\n"
      << "#\n"
      << "vtkCleanPolyData clean_filter;\n"
      << "    clean_filter SetInput [iso_filter GetOutput];\n"
      << "    clean_filter SetTolerance " << ::sqrt(numeric_limits<Float>::epsilon()) << ";\n"
      << "vtkCastToConcrete cast_filter\n"
      << "    cast_filter SetInput [clean_filter GetOutput]\n"
      << "vtkTriangleFilter triangle_filter\n"
      << "    triangle_filter SetInput [cast_filter GetPolyDataOutput]\n"
      << "#\n"
      << "# write result\n"
      << "#\n"
      << "vtkDataSetWriter writer;\n"
      << "  writer SetInput [triangle_filter GetOutput];\n"
      << "  writer SetFileName \"" << poly_name << "\";\n"
      << "  writer Write;\n"
      << "exit;\n"
      ;
  tcl.close();
  //
  // run vtk
  //
  string command;
  int status = 0;
  if (true) {
      command = "vtk " + tcl_name;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
      if (status != 0) 
	error_macro ("\"" << command << "\" command failed.\n");
  }
  //
  // clear tcl/vtk data
  //
  if (clean) {
      command = "/bin/rm -f " + tcl_name + " " + vtk_name;
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  //  
  // load polygonal data, build coordinates
  //
  ifstream poly(poly_name.c_str());
  if (!poly) error_macro("field: vtk polydata file \"" << poly_name << "\" not found.");
  geo g_iso; 
  if (verbose) clog << "! load `" << poly_name << "'.\n";
  poly >> vtkpolydata >> g_iso;
  g_iso.set_name (basename + "-iso");
  g_iso.set_dimension (get_geo().dimension());
  //
  // clear vtk polydata (iso)
  //
  if (clean) {
      command = "/bin/rm -f " + poly_name;
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return g_iso;
}
int
field::vtk3d_Pk (
    const string& basename, 
    bool execute, bool clean, bool verbose
    ) const
{
  //
  // create vtk data file
  //
  string vtk_name = basename + ".vtk";
  ofstream vtk (vtk_name.c_str());
  if (verbose) clog << "! file \"" << vtk_name << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  vtk << setprecision(digits10);
  if (get_approx() == "P1") {
     vtk << vtkdata << get_geo();
  } else if (get_approx() == "P2") {
     vtk << vtkdata << lattice << get_geo();
  } else {
     fatal_macro ("vtk_3d_Pk: P1 or P2 approximation expected");
  }
  put_vtk(vtk);
  vtk.close();
  //
  // create tcl script
  //
  string tcl_name  = basename + ".tcl";
  ofstream tcl (tcl_name.c_str());
  if (verbose) clog << "! file \"" << tcl_name << "\" created.\n";
  tcl << setprecision(digits10)
      << "#!vtk\n"
      << "# this is a tcl script for data \"" << vtk_name  << "\"\n"
      << "# that runs interactive volume rendering.\n"
      << "#\n"
      << "set file_name        \"" << vtk_name << "\";\n"
      << "set color_wheel_file \"" << _RHEOLEF_PKGDATADIR << "/vtk_color_wheel.ppm\";\n"
      << "source \"" << _RHEOLEF_PKGDATADIR << "/vtk_interactor.tcl\";\n"
      << "source \"" << _RHEOLEF_PKGDATADIR << "/vtk_widget_object.tcl\";\n"
      << "source \"" << _RHEOLEF_PKGDATADIR << "/vtk_volume_interactor.tcl\";\n"
      << "source \"" << _RHEOLEF_PKGDATADIR << "/vtk_volume.tcl\";\n"
      ;
  tcl.close();
  //
  // run vtk
  //
  string command;
  int status = 0;
  if (execute) {
      command = "vtk " + tcl_name;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
      if (status != 0) 
	error_macro ("\"" << command << "\" command failed.\n");
  }
  //
  // clear tcl/vtk data
  //
  if (clean) {
      command = "/bin/rm -f " + tcl_name + " " + vtk_name;
      if (verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}
}// namespace rheolef
