///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "2D_D.h"
#include "rheolef/ublas_matrix_range.h"
using namespace rheolef;
using namespace std;
using namespace ublas;

//          [    u0      u1      u2   ]
//
// [ v0 ]   [ d0^T*d0 d0^T*d1 d0^T*d2 ]
// [ v1 ] = [ d1^T*d0 d1^T*d1 d1^T*d2 ]
// [ v2 ]   [ d2^T*d0 d2^T*d1 d2^T*d2 ]
//
//          + (sum_{k=0}^2 dk^T*dk)  Id
//
//        = grad^T*grad + (sum_{k=0}^2 dk^T*dk)  Id
//
//  In the axisymmetric case (r,z):
//      2D_D(0,0) += 2 ur/r vr/r r dr dz 
//                 = 2 ur vr (1/r) dr dz
//
void
_2D_D::operator() (const geo_element& K, matrix<Float>& dd) const
{
  build_gradt_grad (K, dd);
  size_type nj = get_first_basis().size(K);
  size_type ni = get_second_basis().size(K);
  switch (K.dimension()) {
    case 1:
      dd *= 2;
      break;
    case 2:
    case 3: {
      matrix<Float> trace (ni, nj);
      trace.clear();
      for (size_type k = 0; k < K.dimension(); k++) {
        matrix_range<matrix<Float> > dk (dd, range(k*ni,(k+1)*ni),range(k*nj,(k+1)*nj));
        trace += dk;
      }
      for (size_type k = 0; k < K.dimension(); k++) {
        matrix_range<matrix<Float> > dk (dd, range(k*ni,(k+1)*ni), range(k*nj,(k+1)*nj));
        dk += trace;
     }
     break;
    }
    default: {
      fatal_macro ("unexpected element `" << K.name() << "'");
      break;
    }
  }
  if (coordinate_system_type() == fem_helper::cartesian) return;
  // --------------------------------------
  // otherwise, add the (ur/r) (vr/r) r dr dz term
  //   as : ur vr (1/r) dr dz
  // i.e. using the dual `1/r' extra weight
  // --------------------------------------
  ublas::matrix<Float> m_invr;
  set_use_coordinate_system_dual_weight (true);
  build_scalar_mass (K, m_invr);
  set_use_coordinate_system_dual_weight (false);
  size_type k = 0;
  if (coordinate_system_type() == fem_helper::axisymmetric_zr) k++;
  matrix_range<matrix<Float> > dk (dd, range(k*ni,(k+1)*ni), range(k*nj,(k+1)*nj));
  dk += 2.*m_invr;
}
_2D_D::size_type
_2D_D::n_derivative() const
{
  if (coordinate_system_type() == fem_helper::cartesian) return 2;
  else return 0;
}
void
_2D_D::check_after_initialize () const
{
  size_type d = coordinate_dimension();
  check_macro (
	d == get_first_space().n_component() && 
	d == get_second_space().n_component(),
	"unsupported non-vectorial space for `2D_D' form");
}
