///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/quadrature.h"
using namespace rheolef;
using namespace std;

namespace rheolef {
static const char* static_family_name[quadrature_option_type::max_family] = {
	"gauss"           ,
	"gauss_lobatto"   ,
	"gauss_radau"	  ,
	"superconvergent" ,
	"middle_edge"
};
string
quadrature_option_type::get_family_name() const
{
  check_macro (_family >= 0 && _family < max_family,
	"unexpected quadrature family number = " << _family);
  return static_family_name[_family];
}
void
quadrature_on_geo::initialize (reference_element hat_K,  quadrature_option_type opt)
{
    resize (0);
    switch (hat_K.type()) {
      case reference_element::p:  init_point (opt); break;
      case reference_element::e:  init_edge (opt); break;
      case reference_element::t:  init_triangle (opt); break;
      case reference_element::q:  init_square (opt); break;
      case reference_element::T:  init_tetrahedron (opt); break;
      case reference_element::P:  init_prism (opt); break;
      case reference_element::H:  init_hexahedron (opt); break;
      default:
	fatal_macro ("quadrature formula on element type `"
	  << hat_K.name() << "' are not yet implemented.");
    }
}
void 
quadrature::_initialize (reference_element hat_K) const
{
    reference_element::enum_type K_type = hat_K.type();
    _quad[K_type].initialize (hat_K, _options);
    _initialized [K_type] = true;
}
quadrature::const_iterator 
quadrature::begin (reference_element hat_K) const
{
    reference_element::enum_type K_type = hat_K.type();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _quad[K_type].begin();
}
quadrature::const_iterator 
quadrature::end (reference_element hat_K) const
{
    reference_element::enum_type K_type = hat_K.type();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _quad[K_type].end();
}
quadrature::size_type 
quadrature::size (reference_element hat_K) const
{
    reference_element::enum_type K_type = hat_K.type();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _quad[K_type].size();
}
ostream&
operator<< (ostream& out, const quadrature_on_geo& x)
{
  out << setprecision (numeric_limits<Float>::digits10)
      << x.size() << endl;
  for (size_t r = 0; r < x.size(); r++)
    out << x[r].x << "\t" << x[r].w << endl;
  return out;
}
ostream&
operator<< (ostream& out, const quadrature& x)
{
  out << "quadrature" << endl
      << x._options.get_family_name() << " " << x._options.get_order()  << endl;
  for (size_t i = 0; i != size_t(reference_element::max_size); i++) {
    reference_element::enum_type K_type = reference_element::enum_type(i);
    if (! x._initialized [K_type]) continue;
    reference_element hat_K (K_type);
    out << "reference_element " << hat_K.name() << endl
        << x._quad[K_type];
  }
  out << "end_quadrature" << endl;
  return out;
}
} // namespace rheolef
