/***************************************************************************
 *   Copyright (C) 2009 by Resara LLC   *
 *   brendan@resara.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "rdsldapsessiontester.h"

REGISTER_TEST(RdsLdapSessionTester);

RdsLdapSessionTester::RdsLdapSessionTester(QObject *parent)
		: QObject(parent), sesh()
{
}


RdsLdapSessionTester::~RdsLdapSessionTester()
{
}

bool RdsLdapSessionTester::objContains(const QString &dn, const QString &name, const QString &value)
{
	ReturnValue ret = sesh.globalSession()->read(dn, QStringList() << name);
	if (ret.isError()) return false;
	LdapResult list = ret.value<LdapResult>();
	for (int i = 0; i < list[name].size(); ++i)
	{
		if (list[name][i] == value) return true;
	}
	return false;
}

void RdsLdapSessionTester::initTestCase()
{
	sesh.globalSession()->remove("CN=depthdude,ou=depthou,ou=testarea,dc=resara,dc=local");
	
	sesh.globalSession()->remove("ou=depthou,ou=testarea,dc=resara,dc=local");

	sesh.globalSession()->remove("cn=normal,ou=testarea,dc=resara,dc=local");

	sesh.globalSession()->remove("cn=freak,ou=testarea,dc=resara,dc=local");

	sesh.globalSession()->remove("ou=newname,dc=resara,dc=local");// sometimes sneaks in if a verify fails
	sesh.globalSession()->remove("ou=newou,dc=resara,dc=local");// sometimes sneaks in if a verify fails

	sesh.globalSession()->remove("ou=testarea,dc=resara,dc=local");

	RdsLdapActions action;
	action.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	action.add(RdsLdapActions::Add, "ou", "testarea");
	action.add(RdsLdapActions::Add, "description", "test area");
	TEST_FUNCTION(sesh.globalSession()->add("ou=testarea,dc=resara,dc=local", action));

	action.clear();
	action.add(RdsLdapActions::Add, "objectclass", "person");
	action.add(RdsLdapActions::Add, "cn", "freak");
	action.add(RdsLdapActions::Add, "sn", "freak");
	TEST_FUNCTION(sesh.globalSession()->add("CN=freak,OU=testarea,DC=resara,DC=local", action));

	action.clear();
	action.add(RdsLdapActions::Add, "objectclass", "person");
	action.add(RdsLdapActions::Add, "cn", "normal");
	action.add(RdsLdapActions::Add, "sn", "normal");
	TEST_FUNCTION(sesh.globalSession()->add("CN=normal,OU=testarea,DC=resara,DC=local", action));

	action.clear();
	action.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	action.add(RdsLdapActions::Add, "ou", "depthou");
	TEST_FUNCTION(sesh.globalSession()->add("OU=depthou,OU=testarea,DC=resara,DC=local", action));

	action.clear();
	action.add(RdsLdapActions::Add, "objectclass", "person");
	action.add(RdsLdapActions::Add, "cn", "depthdude");
	action.add(RdsLdapActions::Add, "sn", "depthdude");
	TEST_FUNCTION(sesh.globalSession()->add("CN=depthdude,OU=depthou,OU=testarea,DC=resara,DC=local", action));
}

void RdsLdapSessionTester::init()
{
	m_basedn = "OU=testarea,dc=resara,dc=local";
}

void RdsLdapSessionTester::cleanupTestCase()
{
	sesh.globalSession()->remove("CN=depthdude,ou=depthou,ou=testarea,dc=resara,dc=local");
	
	sesh.globalSession()->remove("ou=depthou,ou=testarea,dc=resara,dc=local");

	sesh.globalSession()->remove("cn=normal,ou=testarea,dc=resara,dc=local");

	sesh.globalSession()->remove("cn=freak,ou=testarea,dc=resara,dc=local");

	sesh.globalSession()->remove("ou=newname,dc=resara,dc=local");// sometimes sneaks in if a verify fails
	sesh.globalSession()->remove("ou=newou,dc=resara,dc=local");// sometimes sneaks in if a verify fails

	sesh.globalSession()->remove("ou=testarea,dc=resara,dc=local");
}

#ifdef __RDS_SERVER
void RdsLdapSessionTester::bindTest()
{
	QString dn("ou=sessiontester,dc=resara,dc=local");
	sesh.globalSession()->remove(dn);
	RdsLdapActions act, act2;
	act.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	act.add(RdsLdapActions::Add, "ou", "sessiontester");
	act.add(RdsLdapActions::Add, "description", "too easy");
	RdsLdapSession session;
	ReturnValue ret = session.anonymousBind();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	//test to make sure it cant add or modify anything...
	ret = session.add(dn, act);
	QVERIFY(ret.isError());
	act2.add(RdsLdapActions::Replace, "cn", "silly");
	ret = session.modify(m_basedn, act2);
	QVERIFY(ret.isError());
	ret = session.read(m_basedn);
	QVERIFY(!ret.isError());


	ret = session.bind(QString("user"), QString("passwd"));
	QVERIFY(ret.isError());
	ret = session.bind(QString("administrator"), QString("muahahaha"));
	QVERIFY(ret.isError());
	ret = session.bind(QString("Guest"), QString(""));
	QVERIFY(ret.isError());
	ret = session.bind(QString("administrator"), QString("Resara123"));
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	ret = session.add(dn, act);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	act.clear();
	act.add(RdsLdapActions::Replace, "description", "awesome");
	ret = session.modify(dn, act);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	ret = session.read(dn);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	ret = session.remove(dn);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));

	//test handle()
	LDAP* handle = session.handle();
	QVERIFY(handle != NULL);
	handle = sesh.globalSession()->handle();
	QVERIFY(handle != NULL);
	handle = sesh.handle();
	QVERIFY(handle == NULL);
}
#endif

void RdsLdapSessionTester::searchTest()
{
	ReturnValue ret = sesh.globalSession()->search(QString("OU=testarea,dc=resara,dc=local"), QString("(|(cn=depthdude)(cn=normal))"));
	LdapResults sResults = ret.value<LdapResults>();
	QList<QString> ldapresultskeys = sResults.keys();
	QVERIFY2(ldapresultskeys.size() == 2, "wrong number of search results found");

	ret = sesh.globalSession()->search(QString("OU=testarea,dc=resara,dc=local"), QString("(|(cn=depthdude)(cn=normal))"), false);
	sResults = ret.value<LdapResults>();
	ldapresultskeys = sResults.keys();
	QVERIFY2(ldapresultskeys.size() == 1, "wrong number of search results found");

}

void RdsLdapSessionTester::readTest()
{
	QStringList attrs;
	attrs << "ou";

	ReturnValue ret = sesh.globalSession()->read(m_basedn);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	LdapResult hash = ret.value<LdapResult>();
	QList<QString> keys = hash.keys();
	int attrvalues = 0;
	for (int i = 0; i < keys.size(); ++i)
	{
		QList<QByteArray> list = hash[keys[i]];
		for (int j = 0; j < list.size(); ++j)
		{
			++attrvalues;
		}
	}
	QCOMPARE(keys.size(), 12);
	QCOMPARE(attrvalues, 13);

	ret = sesh.globalSession()->read(m_basedn, attrs);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	hash = ret.value<LdapResult>();
	LdapValues list = hash["ou"];
	QVERIFY(list.size() == 1);
	QCOMPARE(list[0], QByteArray("testarea"));

	//BREAK IT!!
	ret = sesh.globalSession()->read("ou:testarea,dc=resara,dc=local");
	QVERIFY(ret.isError());
	ret = sesh.globalSession()->read("ou=testarea,dc:resara,dc=local");
	QVERIFY(ret.isError());
	ret = sesh.globalSession()->read("");
	QVERIFY(ret.isError());
	ret = sesh.globalSession()->read("ziltoid");
	QVERIFY(ret.isError());
}

void RdsLdapSessionTester::listTest()
{
	ReturnValue ret;

	QVERIFY2(!(ret = sesh.globalSession()->list(m_basedn)).isError(), qPrintable(ret.errString()));
	QStringList list = ret.toStringList();
	QCOMPARE(list.size(), 5);
	QVERIFY(list.contains("CN=depthdude,OU=depthou,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("OU=depthou,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=normal,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=freak,OU=testarea,DC=resara,DC=local"));

	QVERIFY2(!(ret =  sesh.globalSession()->list(m_basedn, "(cn=*)")).isError(), qPrintable(ret.errString()));
	list = ret.toStringList();
	QCOMPARE(list.size(), 3);
	QVERIFY(list.contains("CN=depthdude,OU=depthou,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=normal,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=freak,OU=testarea,DC=resara,DC=local"));

	QVERIFY2(!(ret =  sesh.globalSession()->list(m_basedn, "(cn=*)", false)).isError(), qPrintable(ret.errString()));
	list = ret.toStringList();
	QCOMPARE(list.size(), 2);
	QVERIFY(list.contains("CN=freak,OU=testarea,DC=resara,DC=local"));
	QVERIFY(list.contains("CN=normal,OU=testarea,DC=resara,DC=local"));

}

void RdsLdapSessionTester::addModifyTest()
{	
	//add things that should work
	RdsLdapActions actions;
	QString dn("cn=testdude,ou=testarea,dc=resara,dc=local");
	sesh.globalSession()->remove(dn).isError();
	actions.add(RdsLdapActions::Add, "cn", "testdude");
	actions.add(RdsLdapActions::Add, "sn", "testdude");
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "description", "this is a test person");
	ReturnValue ret = sesh.globalSession()->add(dn, actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(dn, "cn", "testdude"));
	QVERIFY(objContains(dn, "sn", "testdude"));
	QVERIFY(objContains(dn, "objectclass", "person"));
	QVERIFY(objContains(dn, "description", "this is a test person"));
	QVERIFY2(!(ret = sesh.globalSession()->remove(dn)).isError(), qPrintable(ret.errString()));

	QList<QString> list;
	list << "description1" << "description2" << "description3";
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "sn", "dude");
	actions.add(RdsLdapActions::Add, "cn", "testdude");
	actions.add(RdsLdapActions::Add, "description", "this is a test person");
	actions.add(RdsLdapActions::Add, "description", "still a test person");
	actions.add(RdsLdapActions::Add, "description", list);
	ret = sesh.globalSession()->add(dn, actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(dn, "cn", "testdude"));
	QVERIFY(objContains(dn, "sn", "dude"));
	QVERIFY(objContains(dn, "objectclass", "person"));
	QVERIFY(objContains(dn, "description", "this is a test person"));
	QVERIFY(objContains(dn, "description", "still a test person"));
	QVERIFY(objContains(dn, "description", "description1"));
	QVERIFY(objContains(dn, "description", "description2"));
	QVERIFY(objContains(dn, "description", "description3"));

	actions.clear();
	actions.add(RdsLdapActions::Add, "description", "person");
	QVERIFY(sesh.globalSession()->add(dn, actions).isError()); //can't add() to an object that already exists
	QVERIFY2(!(ret = sesh.globalSession()->remove(dn)).isError(), qPrintable(ret.errString()));

	//add things that should break
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "", "a description");
	ret = sesh.globalSession()->add(dn, actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "description", "");
	ret = sesh.globalSession()->add(dn, actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "blorg", "broked");
	ret = sesh.globalSession()->add(dn, actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "cn", "newou");
	ret = sesh.globalSession()->add(dn, actions);
	QVERIFY(ret.isError());

	dn = "cn:brokendude,ou=testarea,dc=resara,dc=local";
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	QVERIFY(sesh.globalSession()->add(dn, actions).isError());

	//modify things that should work
	dn = "cn=testdude,ou=testarea,dc=resara,dc=local";
	sesh.globalSession()->remove(dn);
	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "person");
	actions.add(RdsLdapActions::Add, "description", list);
	actions.add(RdsLdapActions::Add, "sn", "dude");
	actions.add(RdsLdapActions::Add, "cn", "testdude");
	ret = sesh.globalSession()->add(dn, actions);//modify from nothing (aka add)
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(dn, "cn", "testdude"));
	QVERIFY(objContains(dn, "sn", "dude"));
	QVERIFY(objContains(dn, "objectclass", "person"));
	QVERIFY(objContains(dn, "description", "description1"));
	QVERIFY(objContains(dn, "description", "description2"));
	QVERIFY(objContains(dn, "description", "description3"));
	QVERIFY2(!(ret = sesh.globalSession()->remove(dn)).isError(), qPrintable(ret.errString()));

	actions.clear();
	actions.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	actions.add(RdsLdapActions::Add, "ou", "newou");
	actions.add(RdsLdapActions::Add, "l", "testlocation");
	actions.add(RdsLdapActions::Add, "st", "state");
	actions.add(RdsLdapActions::Add, "description", "test unit");
	dn = ("ou=newou,dc=resara,dc=local");
	TEST_FUNCTION(sesh.globalSession()->add(dn, actions));

	//back to member obj, done with obj1
	actions.clear();
	actions.add(RdsLdapActions::Add, "description", list);
	actions.add(RdsLdapActions::Remove, "description", "test unit");
	actions.add(RdsLdapActions::Remove, "st", "");
	actions.add(RdsLdapActions::Replace, "l", "dude");
	ret = sesh.globalSession()->modify(dn, actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(dn, "l", "dude"));
	QVERIFY(!objContains(dn, "l", "testlocation"));
	QVERIFY(!objContains(dn, "st", "state"));
	QVERIFY(objContains(dn, "ou", "newou"));
	QVERIFY(objContains(dn, "objectclass", "organizationalUnit"));
	QVERIFY(!objContains(dn, "description", "test unit"));
	QVERIFY(objContains(dn, "description", "description1"));
	QVERIFY(objContains(dn, "description", "description2"));
	QVERIFY(objContains(dn, "description", "description3"));

	actions.clear();
	actions.add(RdsLdapActions::Replace, "description", "test unit");
	ret = sesh.globalSession()->modify(dn, actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(dn, "l", "dude"));
	QVERIFY(objContains(dn, "ou", "newou"));
	QVERIFY(objContains(dn, "objectclass", "organizationalUnit"));
	QVERIFY(objContains(dn, "description", "test unit"));
	QVERIFY(!objContains(dn, "description", "description1"));
	QVERIFY(!objContains(dn, "description", "description2"));
	QVERIFY(!objContains(dn, "description", "description3"));

	actions.clear();
	actions.add(RdsLdapActions::Add, "description", list);
	actions.add(RdsLdapActions::Remove, "description", "");
	ret = sesh.globalSession()->modify(dn, actions);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(objContains(dn, "l", "dude"));
	QVERIFY(objContains(dn, "ou", "newou"));
	QVERIFY(objContains(dn, "objectclass", "organizationalUnit"));
	QVERIFY(!objContains(dn, "description", "test unit"));
	QVERIFY(!objContains(dn, "description", "description1"));
	QVERIFY(!objContains(dn, "description", "description2"));
	QVERIFY(!objContains(dn, "description", "description3"));

	//modify things that should break
	actions.clear();
	actions.add(RdsLdapActions::Remove, "description", "");
	ret = sesh.globalSession()->modify(dn, actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Remove, "", "dude");
	ret = sesh.globalSession()->modify(dn, actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Add, "l", "new");
	ret = sesh.globalSession()->modify(dn, actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Replace, "ou", "testou");
	ret = sesh.globalSession()->modify(dn, actions);
	QVERIFY(ret.isError());
	actions.clear();
	actions.add(RdsLdapActions::Remove, "blorg", "broked");
	ret = sesh.globalSession()->modify(dn, actions);
	QVERIFY(ret.isError());
	sesh.globalSession()->remove(dn);
}

void RdsLdapSessionTester::RemoveTest()
{
	QString obj2("ou=nonexistant,dc=resara,dc=local");
	QString obj3("ou:nonexistant,dc=resara,dc=local");
	QString obj4, obj5("ou=existant,dc=resara,dc=local");
	RdsLdapActions act;
	act.add(RdsLdapActions::Add, "ou", "existant");
	act.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	QVERIFY(!sesh.globalSession()->add(obj5, act).isError());
	QVERIFY(sesh.globalSession()->remove(obj2).isError());
	QVERIFY(sesh.globalSession()->remove(obj3).isError());
	QVERIFY(sesh.globalSession()->remove(obj4).isError());
	QVERIFY(!sesh.globalSession()->remove(obj5).isError());
	QVERIFY(sesh.globalSession()->remove(obj5).isError());
	QCOMPARE(obj5, QString("ou=existant,dc=resara,dc=local"));
}

void RdsLdapSessionTester::RenameTest()
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	actions.add(RdsLdapActions::Add, "ou", "newou");
	actions.add(RdsLdapActions::Add, "l", "testlocation");
	actions.add(RdsLdapActions::Add, "st", "state");
	actions.add(RdsLdapActions::Add, "description", "test unit");
	QString dn = ("ou=newou,dc=resara,dc=local");
	ReturnValue ret = TEST_FUNCTION(sesh.globalSession()->add(dn, actions));

	//valid re-names
	ret = sesh.globalSession()->rename(dn, "ou=newname");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	sesh.globalSession()->remove(dn);
	dn = "ou=newname,dc=resara,dc=local";

	//invalid re-names
	QVERIFY(sesh.globalSession()->rename(dn, "ou:orgunit").isError());
	QVERIFY(sesh.globalSession()->rename(dn, "dc=funky").isError());
	QVERIFY(sesh.globalSession()->rename(dn, "cn=COMMONNAME").isError());
	QVERIFY(sesh.globalSession()->rename(dn, "IWANTTHISNAME").isError());
	QVERIFY(sesh.globalSession()->rename(dn, "").isError());
	QVERIFY(sesh.globalSession()->rename(dn, "tard=ugly").isError());
	QVERIFY(sesh.globalSession()->rename(dn, "ou=newname,ou=testarea").isError());

	sesh.globalSession()->remove(dn);
}

void RdsLdapSessionTester::moveTest()
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "objectclass", "organizationalUnit");
	actions.add(RdsLdapActions::Add, "ou", "newou");
	actions.add(RdsLdapActions::Add, "l", "testlocation");
	actions.add(RdsLdapActions::Add, "st", "state");
	actions.add(RdsLdapActions::Add, "description", "test unit");
	QString dn = ("ou=newou,dc=resara,dc=local");
	ReturnValue ret = TEST_FUNCTION(sesh.globalSession()->add(dn, actions));

	ret = sesh.globalSession()->move(dn, "ou:testarea,dc=resara,dc=local");
	QVERIFY(ret.isError());
	ret = sesh.globalSession()->move(dn, "dc=resara,dc=local");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	dn = "ou=newou,dc=resara,dc=local";
	ret = sesh.globalSession()->move(dn, "ou=testarea,dc=resara,dc=local");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	dn = "ou=newou,ou=testarea,dc=resara,dc=local";

	sesh.globalSession()->remove(dn);
}
