/***************************************************************************
 *   Copyright (C) 2009 by Resara LLC   *
 *   brendan@resara.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "rdsdhcptester.h"
#include <QHostAddress>

REGISTER_TEST(RdsDhcpTester);

typedef QPair<QHostAddress, QHostAddress> RangePair;
QTRPC_REGISTER_METATYPE(QHostAddress);
QTRPC_REGISTER_METATYPE(QList<RangePair>);

RdsDhcpTester::RdsDhcpTester(QObject *parent)
		: QObject(parent)
{
}


RdsDhcpTester::~RdsDhcpTester()
{
}

void RdsDhcpTester::init()
{
	QFile("/tmp/testdhcpd.conf").setPermissions(QFile::ReadOwner | QFile::WriteOwner | QFile::ReadGroup | QFile::ReadOther);
	QProcess proc;
	proc.setProcessChannelMode(QProcess::ForwardedChannels);
	proc.start("dhcpd3", QStringList() << "-t" << "-cf" << "/tmp/testdhcpd.conf");
	proc.waitForFinished();
	if (proc.exitCode() != 0)
	{
		QFAIL("Invalid configuration file, not reloading.");
	}
	QFile file("/tmp/testdhcpd.conf");
	file.open(QFile::ReadOnly);
	QVERIFY2(manager.parse(file.readAll()), "Failed to parse the testdhcpd.conf file");
}

void RdsDhcpTester::cleanup()
{
	qDebug() << "\n\nNEXT TEST\n\n";
}

void RdsDhcpTester::constructorTest()
{
	RdsDhcpManager manager1 = manager, manager2(manager);

	QString conf = manager.toString();
//	qDebug() << "CONFIGURATION :::::" << conf;
	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reloadConfig());

	QCOMPARE(manager.toString(), conf);
//	qDebug() << "CONFIGURATION :::::" << manager.toString();
}


void RdsDhcpTester::valuesTest()
{
	RdsDhcpValues values = TEST_FUNCTION(manager.values());

	//read values
	ReturnValue ret = values.values();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVariantMap Values = ret.toMap();
	QStringList list = Values.keys();
	for (int i = 0; i < list.size(); ++i)
	{
		ret = values.value(list[i]);
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		QCOMPARE(ret.toString(), Values.value(list[i]).toString());

		//remove them to add them later
		ret = values.removeValue(list[i]);
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		ret = values.value(list[i]);
		QCOMPARE(ret.toString(), QString());
	}

	//set with setValues()
	TEST_FUNCTION(values.setValues(Values));

	ret = TEST_FUNCTION(values.values());
	QVariantMap Values2 = ret.toMap();
	QCOMPARE(Values, Values2);
	list = Values.keys();
	for (int i = 0; i < list.size(); ++i)
	{
		ret = TEST_FUNCTION(values.value(list[i]));
		QCOMPARE(ret.toString(), Values.value(list[i]).toString());

		//remove them to add them later
		ret = TEST_FUNCTION(values.removeValue(list[i]));
		ret = values.value(list[i]);
		QCOMPARE(ret.toString(), QString());
	}

	//set with setValue()
	for (int i = 0; i < list.size(); ++i)
	{
		ret = values.setValue(list[i], Values.value(list[i]).toString());
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		QVERIFY(ret.toBool());
	}
	//verify they are there
	ret = TEST_FUNCTION(values.values());
	Values2 = ret.toMap();
	QCOMPARE(Values, Values2);
	list = Values.keys();
	for (int i = 0; i < list.size(); ++i)
	{
		ret = values.value(list[i]);
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		QCOMPARE(ret.toString(), Values.value(list[i]).toString());
	}

	// this was a bug, so we're testing it now...
	TEST_FUNCTION_ERROR(values.setValue("option", "brokenoption"));
	TEST_FUNCTION(values.setValue("option", "brokenoption brokenvalue"));
	ret = TEST_FUNCTION(values.option("brokenoption"));
	QCOMPARE(ret.toString(), QString("brokenvalue"));
	TEST_FUNCTION(values.removeOption("brokenoption"));
	ret = TEST_FUNCTION(values.option("brokenoption"));
	QCOMPARE(ret.toString(), QString());

	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reload());

	values = TEST_FUNCTION(manager.values());
	ret = TEST_FUNCTION(values.values());
	Values2 = ret.toMap();
	QCOMPARE(Values, Values2);

	ret = TEST_FUNCTION(values.option("brokenoption"));
	QCOMPARE(ret.toString(), QString());
}

void RdsDhcpTester::valueOptionsTest()
{
	RdsDhcpValues values = TEST_FUNCTION(manager.values());

	ReturnValue ret = TEST_FUNCTION(values.options());
	QVariantMap options = ret.toMap();
	QStringList list = options.keys();
	for (int i = 0; i < list.size(); ++i)
	{
		ret = values.option(list[i]);
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		QCOMPARE(ret.toString(), options.value(list[i]).toString());

		//remove them to add them later
		ret = values.removeOption(list[i]);
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		ret = values.option(list[i]);
		QCOMPARE(ret.toString(), QString());
	}
	QCOMPARE(values.options().toMap().size(), 0);

	//set with setOptions()
	TEST_FUNCTION(values.setOptions(options));

	ret = TEST_FUNCTION(values.options());
	QVariantMap options2 = ret.toMap();
	QCOMPARE(options, options2);
	list = options.keys();
	for (int i = 0; i < list.size(); ++i)
	{
		ret = values.option(list[i]);
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		QCOMPARE(ret.toString(), options.value(list[i]).toString());

		//remove them to add them later
		ret = values.removeOption(list[i]);
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		ret = values.option(list[i]);
		QCOMPARE(ret.toString(), QString());
	}
	QCOMPARE(values.options().toMap().size(), 0);

	//set with setOption()
	for (int i = 0; i < list.size(); ++i)
	{
		ret = values.setOption(list[i], options.value(list[i]).toString());
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		QVERIFY(ret.toBool());
	}

	TEST_FUNCTION(values.setOptions(options));
	ret = TEST_FUNCTION(values.options());
	options2 = ret.toMap();
	QCOMPARE(options, options2);
	list = options.keys();
	for (int i = 0; i < list.size(); ++i)
	{
		ret = values.option(list[i]);
		QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
		QCOMPARE(ret.toString(), options.value(list[i]).toString());
	}
	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reload());

	values = TEST_FUNCTION(manager.values());
	ret = TEST_FUNCTION(values.options());
	options2 = ret.toMap();
	QCOMPARE(options, options2);
}

void RdsDhcpTester::valueGroupsTest()
{
	RdsDhcpValues values = TEST_FUNCTION(manager.values());

	ReturnValue ret = TEST_FUNCTION(values.groups());
	QStringList list = ret.toStringList();
	QVERIFY2(list.size() > 0, "NO GROUPS IN testdhcpd.conf!  this test needs groups!");
	QVERIFY2(list.contains("Group"), "config does not have a group named \"Group\")");

	TEST_FUNCTION_ERROR(values.group("grewpy"));
	RdsDhcpGroup group = TEST_FUNCTION(values.group("Group"));

	//do some RdsDhcpGroup Tests
	ret = TEST_FUNCTION(group.name());
	QCOMPARE(ret.toString(), QString("Group"));
	TEST_FUNCTION(group.setName("bleh"));
	ret = TEST_FUNCTION(group.name());
	QCOMPARE(ret.toString(), QString("bleh"));

	group = TEST_FUNCTION(values.group("bleh"));
	TEST_FUNCTION(group.setName("Group"));
	//done with that

	group = TEST_FUNCTION(values.addGroup("testgroup"));
	QCOMPARE(group.name().toString(), QString("testgroup"));
	TEST_FUNCTION(values.group("testgroup"));
	ret = TEST_FUNCTION(values.removeGroup("testgroup"));
	QVERIFY(ret.toBool());
	TEST_FUNCTION_ERROR(values.group("testgroup"));

	group = TEST_FUNCTION(values.addGroup("emptygroup"));
	QCOMPARE(group.name().toString(), QString("emptygroup"));

	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reload());

	values = TEST_FUNCTION(manager.values());
	ret = TEST_FUNCTION(values.groups());
	list = ret.toStringList();
	QVERIFY2(list.size() == 2, "DON'T HAVE 2 GROUPS AFTER RELOAD");
	group = TEST_FUNCTION(values.group("Group"));
	QCOMPARE(group.name().toString(), QString("Group"));
	ret = TEST_FUNCTION(group.values());
	QVariantMap map = ret.toMap();
	QVERIFY2(map.size() == 1, "lost group values after reload");

	group = TEST_FUNCTION(values.group("emptygroup"));
	QCOMPARE(group.name().toString(), QString("emptygroup"));

}

void RdsDhcpTester::valueSubnetTest()
{
	RdsDhcpValues values = TEST_FUNCTION(manager.values());

	RdsDhcpSharedNetwork sharednetwork = TEST_FUNCTION(values.sharedNetwork("WORKSTATIONS"));
	QStringList list = sharednetwork.subnets().toStringList();
	QVERIFY2(list.contains("Subnet"), "GROUP 'WORKSTATIONS' DOES NOT CONTAIN SUBNET 'Subnet'");
//-----------Depth------------
	RdsDhcpSubnet subnet = TEST_FUNCTION(sharednetwork.subnet("Subnet"));
	QCOMPARE(subnet.name().toString(), QString("Subnet"));
	QHostAddress address, netmask;
	address = subnet.address().value<QHostAddress>();
	netmask = subnet.netmask().value<QHostAddress>();
	subnet.setAddress(QHostAddress("192.168.1.1"));
	subnet.setNetmask(QHostAddress("255.255.255.0"));
	QCOMPARE(subnet.address().value<QHostAddress>(), QHostAddress("192.168.1.1"));
	QCOMPARE(subnet.netmask().value<QHostAddress>(), QHostAddress("255.255.255.0"));
	subnet.setAddress(address);
	subnet.setNetmask(netmask);

	QList<RangePair> rlist = subnet.ranges().value<QList<RangePair> >();
	QVERIFY2(rlist.size() == 1, "WHERE DID THESE OTHER RANGES COME FROM?");
	RangePair rpair = rlist[0];//easy verification later
	
	//make sure parents are updated on name changes
	TEST_FUNCTION(subnet.setName("TitsMcGee"));
	subnet = TEST_FUNCTION(sharednetwork.subnet("TitsMcGee"));
	QCOMPARE(subnet.name().toString(), QString("TitsMcGee"));
	TEST_FUNCTION(subnet.setName("Subnet"));

	ReturnValue ret = TEST_FUNCTION(subnet.ranges());
//----------------------------

	ret = TEST_FUNCTION(values.subnets());
	list = ret.toStringList();
	QVERIFY2(list.size() == 0, "WHERE DID THESE SUBNETS COME FROM?");

	subnet = TEST_FUNCTION(values.addSubnet("aild"));

	QCOMPARE(subnet.name().toString(), QString("aild"));
	ret = subnet.setName("btbam");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(ret.toBool());
	QCOMPARE(subnet.name().toString(), QString("btbam"));

	//check parent was updated
	subnet = TEST_FUNCTION(values.subnet("btbam"));

	ret = subnet.setAddress(QHostAddress("172.16.0.0"));
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(ret.toBool());
	ret = subnet.setNetmask(QHostAddress("255.255.0.0"));
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(ret.toBool());
	QCOMPARE(subnet.address().value<QHostAddress>(), QHostAddress("172.16.0.0"));
	QCOMPARE(subnet.netmask().value<QHostAddress>(), QHostAddress("255.255.0.0"));

	rlist.clear();
	rlist.append(qMakePair(QHostAddress("172.16.8.1"), QHostAddress("172.16.8.63")));
	//rlist.append(qMakePair(QHostAddress("172.16.2.64"), QHostAddress("172.16.2.127")));
	rlist.append(qMakePair(QHostAddress("172.16.8.128"), QHostAddress("172.16.8.191")));
	rlist.append(qMakePair(QHostAddress("172.16.8.192"), QHostAddress("172.16.8.255")));
	ret = TEST_FUNCTION(subnet.setRanges(rlist));
	QVERIFY(ret.toBool());

	ret = subnet.ranges();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	rlist = ret.value<QList<RangePair> >();
	QVERIFY2(rlist.size() > 0, "NO RANGES");
// 	qDebug() << "rlist.size()" << rlist.size();
//  	foreach(RangePair range, rlist)
// 		qDebug() << range.first << range.second;

	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reload());
	values = TEST_FUNCTION(manager.values());

	ret = TEST_FUNCTION(values.subnets());
	list = ret.toStringList();
	QVERIFY2(list.size() > 0, "NO SUBNETS ON RELOAD");

	subnet = TEST_FUNCTION(values.subnet("btbam"));
	ret = TEST_FUNCTION(subnet.ranges());
	QList<RangePair> rlist2 = ret.value<QList<RangePair> >();
	QCOMPARE(rlist, rlist2);

	sharednetwork = TEST_FUNCTION(values.sharedNetwork("WORKSTATIONS"));
	list = sharednetwork.subnets().toStringList();
	QVERIFY2(list.contains("Subnet"), "GROUP 'WORKSTATIONS' DOES NOT CONTAIN SUBNET 'Subnet' AFTER RELOAD");
	subnet = TEST_FUNCTION(sharednetwork.subnet("Subnet"));
	QCOMPARE(subnet.address().value<QHostAddress>(), address);
	QCOMPARE(subnet.netmask().value<QHostAddress>(), netmask);

	rlist = subnet.ranges().value<QList<RangePair> >();
	QVERIFY2(rlist.size() == 1, "BONUS RANGES!!");
	QCOMPARE(rpair, rlist[0]);
	
}

void RdsDhcpTester::valueSharedNetworkTest()
{
	RdsDhcpValues values = TEST_FUNCTION(manager.values());

	ReturnValue ret = values.sharedNetworks();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QStringList list = ret.toStringList();
	QVERIFY2(list.size() > 0, "NO SHAREDNETWORKS IN dhcpd.conf!  this test needs sharednetworks!");
	//for (int i = 0; i < list.size(); ++i)
	//	qDebug() << list[i];
	QVERIFY2(list.contains("WORKSTATIONS"), "NO SHAREDNETWORK NAMED 'WORKSTATIONS'. This sharednetwork is required");

	TEST_FUNCTION_ERROR(values.sharedNetwork("netty"));
	ret = values.sharedNetwork("WORKSTATIONS");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));

	//do some RdsDhcpSharedNetwork Tests
	RdsDhcpSharedNetwork sharednetwork = ret;
	ret = sharednetwork.name();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QCOMPARE(ret.toString(), QString("WORKSTATIONS"));
	ret = sharednetwork.setName("bleh");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	ret = sharednetwork.name();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QCOMPARE(ret.toString(), QString("bleh"));
	ret = values.sharedNetwork("bleh");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	sharednetwork = ret;
	sharednetwork.setName("WORKSTATIONS");
	//done with that

	sharednetwork = TEST_FUNCTION(values.addSharedNetwork("testnetwork"));
	QCOMPARE(sharednetwork.name().toString(), QString("testnetwork"));

	ret = TEST_FUNCTION(values.removeSharedNetwork("testnetwork"));
	QVERIFY(ret.toBool());
	TEST_FUNCTION_ERROR(values.sharedNetwork("testnetwork"));

	TEST_FUNCTION(values.addSharedNetwork("emptynetwork"));

	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reload());

	values = TEST_FUNCTION(manager.values());

	ret = TEST_FUNCTION(values.sharedNetworks());
	list = ret.toStringList();
	QVERIFY2(list.size() > 0, "NO SHARED NETWORKS ON RELOAD");

	TEST_FUNCTION(values.sharedNetwork("emptynetwork"));
	TEST_FUNCTION(values.sharedNetwork("WORKSTATIONS"));
}

void RdsDhcpTester::valueHostTest()//done
{
	RdsDhcpValues values = TEST_FUNCTION(manager.values());

	ReturnValue ret = values.hosts();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QStringList list = ret.toStringList();
	QVERIFY(list.size() == 0);
	//qDebug() << list.size();
	//for (int i = 0; i < list.size(); ++i)
	//	qDebug() << list[i];

	//set values using setValue()
	ret = values.addHost("testhost");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	RdsDhcpHost host = ret;

	//these maps are useful later
	QVariantMap vmap, omap;
	vmap["fixed-address"] = "10.0.4.243";
	vmap["next-server"] = "10.0.4.1";
	omap["subnet-mask"] = "255.255.0.0";
	omap["routers"] = "10.0.0.1";
	omap["domain-name-servers"] = "10.0.0.1";

	TEST_FUNCTION(host.setValue("fixed-address", "10.0.4.243"));
	TEST_FUNCTION(host.setValue("next-server", "10.0.4.1"));
	TEST_FUNCTION(host.setValue("option", "subnet-mask 255.255.0.0"));
	TEST_FUNCTION(host.setValue("option", "routers 10.0.0.1"));
	TEST_FUNCTION(host.setValue("option", "domain-name-servers 10.0.0.1"));

	TEST_FUNCTION(values.host("testhost"));

	//set values using SetValues()
	host = TEST_FUNCTION(values.addHost("testhost2"));
	QVariantMap map;
	map["fixed-address"] = "10.0.4.243";
	map["next-server"] = "10.0.4.1";
	map.insertMulti("option", "subnet-mask 255.255.0.0");
	map.insertMulti("option", "routers 10.0.0.1");
	map.insertMulti("option", "domain-name-servers 10.0.0.1");
	ret = host.setValues(map);
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));


	host = TEST_FUNCTION(values.host("testhost2"));
	QCOMPARE(host.name().toString(), QString("testhost2"));
	ret = host.setName("poopypants");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QCOMPARE(host.name().toString(), QString("poopypants"));

	//make sure parent is updated
	TEST_FUNCTION(values.host("poopypants"));

	ret = values.removeHost("poopypants");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QVERIFY(values.host("poopypants").isError());

	ret = values.hosts();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	list = ret.toStringList();
	QVERIFY2(list.size() > 0, "NO HOSTS");
	//for (int i = 0; i < list.size(); ++i)
	//	qDebug() << list[i];


	ret = values.groups();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	list = ret.toStringList();
//-----------------test hosts in a child calss------------
	QVERIFY2(list.contains("Group"), "NO GROUP NAMED 'Group'.  This test requires a group named 'Group'");

	//do some RdsDhcpHost Tests
	RdsDhcpGroup group = TEST_FUNCTION(values.group("Group"));

	host = TEST_FUNCTION(group.addHost("hostile"));
	TEST_FUNCTION(host.setValues(map));
	ret = host.name();
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	QCOMPARE(ret.toString(), QString("hostile"));

	ret = group.host("hostile");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	host = ret;
	ret = host.setName("toberemoved");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));

	//check that the parent was updated
	ret = group.host("toberemoved");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	ret = group.removeHost("toberemoved");
	QVERIFY2(!ret.isError(), qPrintable(ret.errString()));
	TEST_FUNCTION_ERROR(group.host("toberemoved"));

	//add a host to make sure it is still there after reload
	host = TEST_FUNCTION(group.addHost("childhost"));
	TEST_FUNCTION(host.setValues(map));
	ret = TEST_FUNCTION(group.hosts());
	QStringList hostlist = ret.toStringList();
//----------------------------------------------------------

	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reload());
	//make sure every thing is still there
	values = TEST_FUNCTION(manager.values());

	list = values.hosts().toStringList();
	QVERIFY2(list.size() > 0, "NO HOSTS ON RELOAD");

	host = TEST_FUNCTION(values.host("testhost"));
	ret = TEST_FUNCTION(host.values());
	QCOMPARE(vmap, ret.toMap());
	ret = TEST_FUNCTION(host.options());
	QCOMPARE(omap, ret.toMap());

	TEST_FUNCTION_ERROR(values.host("poopypants"));

	group = TEST_FUNCTION(values.group("Group"));
	TEST_FUNCTION_ERROR(group.host("toberemoved"));
	host = TEST_FUNCTION(group.host("childhost"));
	ret = TEST_FUNCTION(host.values());
	QCOMPARE(vmap, ret.toMap());
	ret = TEST_FUNCTION(host.options());
	QCOMPARE(omap, ret.toMap());

	ret = TEST_FUNCTION(group.hosts());
	list = ret.toStringList();
	QCOMPARE(list, hostlist);
}
