/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsuser.h"
#include "rdsuser_p.h"
#include <RdsUtils>
#include <RdsSid>
#include <QDebug>
#include <QRegExp>
#include  <RdsNtSecurityDescriptor>
#include <RdsGuid>

#define ACCOUNTDISABLE 				0x0002
#define LOCKOUT 						0x0010
#define PASSWD_CANT_CHANGE 			0x0040
#define ENCRYPTED_TEXT_PWD_ALLOWED 	0x0080
#define NORMAL_ACCOUNT 				0x0200
#define DONT_EXPIRE_PASSWORD			0x10000
#define SMARTCARD_REQUIRED			0x40000
#define NOT_DELEGATED					0x100000
#define USE_DES_KEY_ONLY				0x200000
#define DONT_REQ_PREAUTH				0x400000

QTRPC_REGISTER_METATYPE(RdsUser::UserStates);

RdsUser::RdsUser()
		: RdsAdObject()
{
	QXT_INIT_PRIVATE(RdsUser);
}

RdsUser::RdsUser(const QString &dn)
		: RdsAdObject(dn)
{
	QXT_INIT_PRIVATE(RdsUser);
}

RdsUser::RdsUser(const RdsUser &other)
		: RdsAdObject(other)
{
	QXT_INIT_PRIVATE(RdsUser);
}

RdsUser::~RdsUser()
{
}

RdsUser& RdsUser::operator=(const RdsUser & other)
{
	RdsAdObject::operator=(other);
	return *this;
}

ReturnValue RdsUser::createUser(const QString &dn, const QString &account)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "objectClass", "user");
	actions.add(RdsLdapActions::Add, "samAccountName", account);
	actions.add(RdsLdapActions::Add, "userPrincipalName", account + "@" + RdsUtils::realm());
	actions.add(RdsLdapActions::Add, "userAccountControl", "514");
	actions.add(RdsLdapActions::Add, "scriptPath", "launchapp.bat");
	ReturnValue ret = rdsLdapSession()->add(dn, actions);
	if (ret.isError()) return(ret);
	else return(dn);
}

ReturnValue RdsUser::setPassword(const QString &newpw)
{
	RdsLdapActions actions;
	QString pwd = QString("\"" + newpw + "\"");
	actions.addUnicode(RdsLdapActions::Replace, "unicodePwd", pwd);
	return(modify(actions));
}

ReturnValue RdsUser::userName() const
{
	ReturnValue ret = read();
	if (ret.isError()) return(ret);
	LdapResult entry = ret.value<LdapResult>();

	if (!entry.contains("samaccountname") || entry["samaccountname"].count() == 0) return("");
	else return(entry["samaccountname"][0]);
}

ReturnValue RdsUser::setUserName(const QString &username)
{
	RdsLdapActions actions;
	if (username == "")
		actions.add(RdsLdapActions::Remove, "samaccountName", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "samaccountName", username);
	return(modify(actions));
}

ReturnValue RdsUser::description() const
{
	ReturnValue ret = read();
	if (ret.isError()) return(ret);
	LdapResult entry = ret.value<LdapResult>();

	if (!entry.contains("description") || entry["description"].count() == 0) return("");
	else return(entry["description"][0]);
}

ReturnValue RdsUser::setDescription(const QString &desc)
{
	RdsLdapActions actions;
	if (desc == "")
		actions.add(RdsLdapActions::Remove, "description", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "description", desc);
	return(modify(actions));
}

#define CHECK_ACCOUNT_FLAG(flag,acctflag) if ((acctflags & acctflag) == acctflag) 	flags |= flag;

ReturnValue RdsUser::flags() const
{
	UserStates flags = NoState;
	ReturnValue ret = read();
	if (ret.isError()) return(ret);
	LdapResult result = ret.value<LdapResult>();

	if (!result.contains("useraccountcontrol") || (result["useraccountcontrol"].size() == 0))
	{
		return(ReturnValue(1, "useraccountcontrol attribute does not exist"));
	}

	int acctflags = result["useraccountcontrol"][0].toInt();

	//MustChangePassword
	if (result.contains("pwdlastset") && (result["pwdlastset"].size() != 0) && (result["pwdlastset"][0] == "0"))
		flags |= MustChangePassword;

	ret = securityDescriptor();
	if (!ret.isError())
	{
		bool founda = false, foundb = false;

		RdsNtSecurityDescriptor desc = ret.value<RdsNtSecurityDescriptor>();
		foreach(RdsAce ace, desc.dacl())
		{
			if ((ace.type() == RdsAce::ObjectDeny) && (ace.object() == "AB721A53-1E2F-11D0-9819-00AA0040529B"))
			{
				if (ace.sid().toShortString() == "PS") founda = true;
				if (ace.sid().toShortString() == "WD") foundb = true;
			}
		}

		if (founda && foundb)
			flags |= CannotChangePassword;
	}
	else
	{
		qWarning() << "Failed to get security descriptor:" << ret.errString();
	}

	CHECK_ACCOUNT_FLAG(Locked, LOCKOUT);
	CHECK_ACCOUNT_FLAG(Disabled, ACCOUNTDISABLE);
	CHECK_ACCOUNT_FLAG(PasswordNeverExpires, DONT_EXPIRE_PASSWORD);
	CHECK_ACCOUNT_FLAG(UseReversibleEncryption, ENCRYPTED_TEXT_PWD_ALLOWED);
	CHECK_ACCOUNT_FLAG(SmartCardRequired, SMARTCARD_REQUIRED);
	CHECK_ACCOUNT_FLAG(UseDesPassword, USE_DES_KEY_ONLY);
	CHECK_ACCOUNT_FLAG(NotDelegated, NOT_DELEGATED);
	CHECK_ACCOUNT_FLAG(NoPreAuth, DONT_REQ_PREAUTH);

	return(QVariant::fromValue(flags));
}

#define DO_FLAG(a) 	if((changedflags & a) == a) \
	{ \
		if((flags & a) == a) \
			setFlag(a); \
		else \
			unsetFlag(a); \
	}

ReturnValue RdsUser::setFlags(const UserStates &flags)
{
	UserStates curflags;
	UserStates changedflags;
	ReturnValue ret = RdsUser::flags();
	if (ret.isError()) return(ret);

	curflags = ret.value<UserStates>();
	changedflags = flags ^ curflags;

	DO_FLAG(Locked);
	DO_FLAG(Disabled);
	DO_FLAG(MustChangePassword);
	DO_FLAG(CannotChangePassword);
	DO_FLAG(PasswordNeverExpires);
	DO_FLAG(UseReversibleEncryption);
	DO_FLAG(SmartCardRequired);
	DO_FLAG(UseDesPassword);
	DO_FLAG(NotDelegated);
	DO_FLAG(NoPreAuth);

	return(true);
}

#define SET_ACCT_FLAG(flag,acctflag) case flag: \
	actions.add(RdsLdapActions::Replace, "userAccountControl", QString("%1").arg(accountflags | acctflag)); \
	break;

ReturnValue RdsUser::setFlag(UserState flag)
{
	UserStates flags = NoState;
	ReturnValue ret = read();
	if (ret.isError()) return(ret);
	LdapResult result = ret.value<LdapResult>();

	int accountflags;
	if (result.contains("useraccountcontrol") && (result["useraccountcontrol"].size() != 0))
		accountflags = result["useraccountcontrol"][0].toInt();
	else
		accountflags = 514;


	RdsLdapActions actions;

	switch (flag)
	{
		case Locked:
			return(true);
			break;
		case MustChangePassword:
			actions.add(RdsLdapActions::Replace, "pwdLastSet", "0");
			break;
		case CannotChangePassword:
		{
			ret = securityDescriptor();
			if (ret.isError()) return(ret);
			RdsNtSecurityDescriptor desc = ret.value<RdsNtSecurityDescriptor>();
			QList<RdsAce> list = desc.dacl();
			RdsAce *a = NULL, *b = NULL;

			for (int i = 0; i < list.count(); i ++)
			{
				RdsAce ace = list[i];

				if (ace.object() == "AB721A53-1E2F-11D0-9819-00AA0040529B")
				{
					if (ace.sid().toShortString() == "PS") a = &list[i];
					if (ace.sid().toShortString() == "WD") b = &list[i];
				}
			}

			if (a != NULL) list.removeAll(*a);
			if (b != NULL) list.removeAll(*b);

			list << RdsAce(QString("OD;;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;;PS"));
			list << RdsAce(QString("OD;;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;;WD"));

			desc.setDacl(list);
			return(setSecurityDescriptor(desc));

		}
		break;
		SET_ACCT_FLAG(Disabled, ACCOUNTDISABLE);
		SET_ACCT_FLAG(PasswordNeverExpires, DONT_EXPIRE_PASSWORD);
		SET_ACCT_FLAG(UseReversibleEncryption, ENCRYPTED_TEXT_PWD_ALLOWED);
		SET_ACCT_FLAG(SmartCardRequired, SMARTCARD_REQUIRED);
		SET_ACCT_FLAG(UseDesPassword, USE_DES_KEY_ONLY);
		SET_ACCT_FLAG(NotDelegated, NOT_DELEGATED);
		SET_ACCT_FLAG(NoPreAuth, DONT_REQ_PREAUTH);
		default:
			return(ReturnValue(1, "Unknown Flag: " + flag));
	}

	return(modify(actions));
}

#define UNSET_ACCT_FLAG(flag,acctflag) case flag: \
	actions.add(RdsLdapActions::Replace, "userAccountControl", QString("%1").arg(accountflags & (~acctflag))); \
	break;

ReturnValue RdsUser::unsetFlag(UserState flag)
{
	UserStates flags = NoState;
	ReturnValue ret = read();
	if (ret.isError()) return(ret);
	LdapResult result = ret.value<LdapResult>();

	int accountflags;
	if (result.contains("useraccountcontrol") && (result["useraccountcontrol"].size() != 0))
		accountflags = result["useraccountcontrol"][0].toInt();
	else
		accountflags = 514;

	RdsLdapActions actions;

	switch (flag)
	{
		case Locked:
			actions.add(RdsLdapActions::Replace, "lockoutTime", "0");
			break;
		case MustChangePassword:
			actions.add(RdsLdapActions::Replace, "pwdLastSet", "-1");
			break;
		case CannotChangePassword:
		{
			ret = securityDescriptor();
			if (ret.isError()) return(ret);
			RdsNtSecurityDescriptor desc = ret.value<RdsNtSecurityDescriptor>();
			QList<RdsAce> list = desc.dacl();
			RdsAce *a = NULL, *b = NULL;

			for (int i = 0; i < list.count(); i ++)
			{
				RdsAce ace = list[i];

				if ((ace.type() == RdsAce::ObjectDeny) && (ace.object() == "AB721A53-1E2F-11D0-9819-00AA0040529B"))
				{
					if (ace.sid().toShortString() == "PS") a = &list[i];
					if (ace.sid().toShortString() == "WD") b = &list[i];
				}
			}

			if (a != NULL) list.removeAll(*a);
			if (b != NULL) list.removeAll(*b);
			desc.setDacl(list);
			return(setSecurityDescriptor(desc));

		}
		break;
		UNSET_ACCT_FLAG(Disabled, ACCOUNTDISABLE);
		UNSET_ACCT_FLAG(PasswordNeverExpires, DONT_EXPIRE_PASSWORD);
		UNSET_ACCT_FLAG(UseReversibleEncryption, ENCRYPTED_TEXT_PWD_ALLOWED);
		UNSET_ACCT_FLAG(SmartCardRequired, SMARTCARD_REQUIRED);
		UNSET_ACCT_FLAG(UseDesPassword, USE_DES_KEY_ONLY);
		UNSET_ACCT_FLAG(NotDelegated, NOT_DELEGATED);
		UNSET_ACCT_FLAG(NoPreAuth, DONT_REQ_PREAUTH);
		default:
			return(true);
	}

	return(modify(actions));
}

ReturnValue RdsUser::primaryGroup() const
{
	ReturnValue ret = sid();
	if (ret.isError()) return(ret);
	RdsSid sid = ret.value<RdsSid>();
	ret = readAttribute("primarygroupid");
	if (ret.isError()) return(ret);

	LdapValues attr = ret.value<LdapValues>();
	sid.setRid(attr[0].toInt());
	return(RdsUtils::getObjectBySid(sid));
}

ReturnValue RdsUser::setPrimaryGroup(const QString &groupdn)
{
	ReturnValue ret = rdsLdapSession()->read(groupdn, QStringList() << "objectSid");
	if (ret.isError()) return(ret);

	LdapResult result = ret.value<LdapResult>();
	if (!result.contains("objectsid") || (result["objectsid"].count() == 0))
	{
		return(ReturnValue(1, "objectSid attribute does not exist"));
	}

	RdsSid sid(result["objectsid"][0]);

	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "member", dn());
	ret = rdsLdapSession()->modify(groupdn, actions);
	if (ret.isError() && (ret.errNumber() != 68) && (ret.errNumber() != 20)) return(ret);

	RdsLdapActions actions2;
	actions2.add(RdsLdapActions::Replace, "primarygroupid", QString("%1").arg(sid.rid()));
	ret = modify(actions2);
	return(ret);
}

ReturnValue RdsUser::cn() const
{
	ReturnValue ret = readAttribute("cn");
	if (ret.isError()) return(ret);
	LdapValues values = ret.value<LdapValues>();
	return(QString(values[0]));
}

ReturnValue RdsUser::setCn(const QString &cn)
{
	return(rename("cn=" + cn));
}

ReturnValue RdsUser::groups() const
{
	QStringList list;
	ReturnValue ret = primaryGroup();
	if (!ret.isError()) list << ret.toString();

	ret = read(QStringList() << "memberof");
	if (ret.isError()) return(ret);

	LdapResult result = ret.value<LdapResult>();
	if (!result.contains("memberof")) return(list);

	foreach(QByteArray value, result["memberof"])
	{
		list << value;
	}

	return(list);
}

ReturnValue RdsUser::joinGroup(const QString &groupdn)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "member", dn());
	return(rdsLdapSession()->modify(groupdn, actions));

}

ReturnValue RdsUser::leaveGroup(const QString &groupdn)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Remove, "member", dn());
	return(rdsLdapSession()->modify(groupdn, actions));
}

ReturnValue RdsUser::firstName() const
{
	ReturnValue ret = readAttribute("givenname");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setFirstName(const QString &name)
{
	RdsLdapActions actions;
	if (name == "")
		actions.add(RdsLdapActions::Remove, "givenname", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "givenname", name);

	return(modify(actions));
}

ReturnValue RdsUser::initials() const
{
	ReturnValue ret = readAttribute("initials");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setInitials(const QString &initial)
{
	RdsLdapActions actions;
	if (initial == "")
		actions.add(RdsLdapActions::Remove, "initials", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "initials", initial);

	return(modify(actions));
}

ReturnValue RdsUser::lastName() const
{
	ReturnValue ret = readAttribute("sn");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setLastName(const QString &name)
{
	RdsLdapActions actions;
	if (name == "")
		actions.add(RdsLdapActions::Remove, "sn", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "sn", name);

	return(modify(actions));
}

ReturnValue RdsUser::displayName() const
{
	ReturnValue ret = readAttribute("displayname");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setDisplayName(const QString &name)
{
	RdsLdapActions actions;
	if (name == "")
		actions.add(RdsLdapActions::Remove, "displayName", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "displayName", name);

	return(modify(actions));
}

ReturnValue RdsUser::office() const
{
	ReturnValue ret = readAttribute("physicalDeliveryOfficeName");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setOffice(const QString &office)
{
	RdsLdapActions actions;
	if (office == "")
		actions.add(RdsLdapActions::Remove, "physicalDeliveryOfficeName", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "physicalDeliveryOfficeName", office);

	return(modify(actions));
}

ReturnValue RdsUser::phoneNumber() const
{
	ReturnValue ret = readAttribute("telephoneNumber");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setPhoneNumber(const QString &number)
{
	RdsLdapActions actions;
	if (number == "")
		actions.add(RdsLdapActions::Remove, "telephoneNumber", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "telephoneNumber", number);

	return(modify(actions));
}

ReturnValue RdsUser::email() const
{
	ReturnValue ret = readAttribute("mail");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setEmail(const QString &mail)
{
	RdsLdapActions actions;
	if (mail == "")
		actions.add(RdsLdapActions::Remove, "mail", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "mail", mail);

	return(modify(actions));
}

ReturnValue RdsUser::webPage() const
{
	ReturnValue ret = readAttribute("wWWHomePage");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setWebPage(const QString &url)
{
	RdsLdapActions actions;
	if (url == "")
		actions.add(RdsLdapActions::Remove, "wWWHomePage", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "wWWHomePage", url);

	return(modify(actions));
}

ReturnValue RdsUser::street() const
{
	ReturnValue ret = readAttribute("streetAddress");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setStreet(const QString &street)
{
	RdsLdapActions actions;
	if (street == "")
		actions.add(RdsLdapActions::Remove, "streetAddress", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "streetAddress", street);

	return(modify(actions));
}

ReturnValue RdsUser::poBox() const
{
	ReturnValue ret = readAttribute("postOfficeBox");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setPoBox(const QString &po)
{
	RdsLdapActions actions;
	if (po == "")
		actions.add(RdsLdapActions::Remove, "postOfficeBox", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "postOfficeBox", po);

	return(modify(actions));
}

ReturnValue RdsUser::city() const
{
	ReturnValue ret = readAttribute("l");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setCity(const QString &city)
{
	RdsLdapActions actions;
	if (city == "")
		actions.add(RdsLdapActions::Replace, "l", city);
	else
		actions.add(RdsLdapActions::Replace, "l", city);

	return(modify(actions));
}

ReturnValue RdsUser::state() const
{
	ReturnValue ret = readAttribute("st");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setState(const QString &state)
{
	RdsLdapActions actions;

	if (state == "")
		actions.add(RdsLdapActions::Remove, "st", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "st", state);

	return(modify(actions));
}

ReturnValue RdsUser::country() const
{
	ReturnValue ret = readAttribute("c");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setCountry(const QString &country)
{
	RdsLdapActions actions;
	if (country == "")
		actions.add(RdsLdapActions::Remove, "c", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "c", country);

	return(modify(actions));
}

ReturnValue RdsUser::postalCode() const
{
	ReturnValue ret = readAttribute("postalCode");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setPostalCode(const QString &code)
{
	RdsLdapActions actions;
	if (code == "")
		actions.add(RdsLdapActions::Remove, "postalCode", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "postalCode", code);

	return(modify(actions));
}

ReturnValue RdsUser::homeDrive() const
{
	ReturnValue ret = readAttribute("homeDrive");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setHomeDrive(const QString &drive)
{
	RdsLdapActions actions;
	if (drive == "")
	{
		actions.add(RdsLdapActions::Remove, "homeDrive", QStringList());
	}
	else
		actions.add(RdsLdapActions::Replace, "homeDrive", drive);

	return(modify(actions));
}

ReturnValue RdsUser::homePath() const
{
	ReturnValue ret = readAttribute("homeDirectory");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setHomePath(const QString &path)
{
	RdsLdapActions actions;
	if (path == "")
		actions.add(RdsLdapActions::Remove, "homeDirectory", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "homeDirectory", path);

	return(modify(actions));
}

ReturnValue RdsUser::profilePath() const
{
	ReturnValue ret = readAttribute("profilePath");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setProfilePath(const QString &path)
{
	RdsLdapActions actions;

	if (path == "")
		actions.add(RdsLdapActions::Remove, "profilePath", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "profilePath", path);

	return(modify(actions));
}

ReturnValue RdsUser::logonScript() const
{
	ReturnValue ret = readAttribute("scriptPath");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setLogonScript(const QString &path)
{
	RdsLdapActions actions;

	if (path == "")
		actions.add(RdsLdapActions::Remove, "scriptPath", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "scriptPath", path);

	return(modify(actions));
}

ReturnValue RdsUser::pagerNumber() const
{
	ReturnValue ret = readAttribute("pager");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setPagerNumber(const QString &pager)
{
	RdsLdapActions actions;

	if (pager == "")
		actions.add(RdsLdapActions::Remove, "pager", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "pager", pager);

	return(modify(actions));
}

ReturnValue RdsUser::mobileNumber() const
{
	ReturnValue ret = readAttribute("mobile");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setMobileNumber(const QString &number)
{
	RdsLdapActions actions;

	if (number == "")
		actions.add(RdsLdapActions::Remove, "mobile", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "mobile", number);

	return(modify(actions));
}

ReturnValue RdsUser::faxNumber() const
{
	ReturnValue ret = readAttribute("facsimileTelephoneNumber");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setFaxNumber(const QString &number)
{
	RdsLdapActions actions;

	if (number == "")
		actions.add(RdsLdapActions::Remove, "facsimileTelephoneNumber", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "facsimileTelephoneNumber", number);

	return(modify(actions));
}

ReturnValue RdsUser::homeNumber() const
{
	ReturnValue ret = readAttribute("homePhone");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setHomeNumber(const QString &number)
{
	RdsLdapActions actions;

	if (number == "")
		actions.add(RdsLdapActions::Remove, "homePhone", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "homePhone", number);

	return(modify(actions));
}

ReturnValue RdsUser::notes() const
{
	ReturnValue ret = readAttribute("info");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setNotes(const QString &notes)
{
	RdsLdapActions actions;

	if (notes == "")
		actions.add(RdsLdapActions::Remove, "info", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "info", notes);

	return(modify(actions));
}

ReturnValue RdsUser::title() const
{
	ReturnValue ret = readAttribute("title");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setTitle(const QString &title)
{
	RdsLdapActions actions;

	if (title == "")
		actions.add(RdsLdapActions::Remove, "title", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "title", title);

	return(modify(actions));
}

ReturnValue RdsUser::department() const
{
	ReturnValue ret = readAttribute("department");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setDepartment(const QString &department)
{
	RdsLdapActions actions;

	if (department == "")
		actions.add(RdsLdapActions::Remove, "department", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "department", department);

	return(modify(actions));
}

ReturnValue RdsUser::company() const
{
	ReturnValue ret = readAttribute("company");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsUser::setCompany(const QString &company)
{
	RdsLdapActions actions;

	if (company == "")
		actions.add(RdsLdapActions::Remove, "company", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "company", company);

	return(modify(actions));
}

ReturnValue RdsUser::emailAliases()
{
	QStringList list;
	ReturnValue ret = read(QStringList() << "proxyaddresses");
	if (ret.isError()) return(ret);
	
	LdapResult result = ret.value<LdapResult>();
	if (!result.contains("proxyaddresses")) return(list);
	
	foreach(QByteArray value, result["proxyaddresses"])
	{
		list << value;
	}
	
	return(list);
}

ReturnValue RdsUser::setEmailAliases(QStringList aliases)
{
	RdsLdapActions actions;
	if (aliases.size() == 0)
		actions.add(RdsLdapActions::Remove, "proxyaddresses", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "proxyaddresses", aliases);
	return(modify(actions));
}


QDataStream& operator<<(QDataStream& d, const RdsUser& object)
{
	d << object.dn();
	return(d);
}

QDataStream& operator>>(QDataStream& d, RdsUser& object)
{
	QString dn;
	d >> dn;
	object.setDn(dn);
	return(d);
}

QDataStream& operator<<(QDataStream& d, const RdsUser::UserStates& object)
{
	d << (int)object;
	return(d);
}

QDataStream& operator>>(QDataStream& d, RdsUser::UserStates& object)
{
	int tmp;
	d >> tmp;
	object = RdsUser::UserStates(tmp);
	return(d);
}

