/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsguid.h"
#include "rdsguid_p.h"

#include <QByteArray>
#include <QString>
#include <QtEndian>
#include <QDebug>

#include "rdsglobal.h"

RdsGuid::RdsGuid()
{
	qxt_d().data = new RdsGuidData();
}

RdsGuid::RdsGuid(const char *str)
{
	qxt_d().data = new RdsGuidData();
	qxt_d().parseString(str);
}

RdsGuid::RdsGuid(const QString &str)
{
	qxt_d().data = new RdsGuidData();
	qxt_d().parseString(str);
}

RdsGuid::RdsGuid(const QByteArray &data)
{
	qxt_d().data = new RdsGuidData();
	qxt_d().parseByteArray(data);
}

RdsGuid::RdsGuid(const RdsGuid &other)
{
	qxt_d().data = other.qxt_d().data;
}

RdsGuid& RdsGuid::operator=(const RdsGuid & other)
{
	qxt_d().data = other.qxt_d().data;
	return *this;
}

RdsGuid& RdsGuid::operator=(const QString & str)
{
	qxt_d().parseString(str);
	return *this;
}

RdsGuid& RdsGuid::operator=(const QByteArray & str)
{
	qxt_d().parseByteArray(str);
	return *this;
}

RdsGuid& RdsGuid::operator=(const char * str)
{
	qxt_d().parseString(str);
	return *this;
}

RdsGuid::~RdsGuid()
{
}

quint32 RdsGuid::timeLow() const
{
	return qxt_d().data->timeLow;
}

ReturnValue RdsGuid::setTimeLow(quint32 timeLow)
{
	qxt_d().data->isNull = false;
	qxt_d().data->timeLow = timeLow;
	return true;
}

quint16 RdsGuid::timeMiddle() const
{
	return qxt_d().data->timeMiddle;
}

ReturnValue RdsGuid::setTimeMiddle(quint16 timeMiddle)
{
	qxt_d().data->isNull = false;
	qxt_d().data->timeMiddle = timeMiddle;
	return true;
}

quint16 RdsGuid::timeHiAndVersion() const
{
	return qxt_d().data->timeHiAndVersion;
}

ReturnValue RdsGuid::setTimeHiAndVersion(quint16 timeHiAndVersion)
{
	qxt_d().data->isNull = false;
	qxt_d().data->timeHiAndVersion = timeHiAndVersion;
	return true;
}

quint16 RdsGuid::clockSequence() const
{
	return qxt_d().data->clockSequence;
}

ReturnValue RdsGuid::setClockSequence(const quint16 clockSequence)
{
	qxt_d().data->isNull = false;
	qxt_d().data->clockSequence = clockSequence;
	return true;
}

quint64 RdsGuid::node() const
{
	return qxt_d().data->node;
}

ReturnValue RdsGuid::setNode(const quint64 node)
{
	qxt_d().data->isNull = false;
	qxt_d().data->node = node;
	return true;
}

bool RdsGuid::isNull() const
{
	return qxt_d().data->isNull;
}

QString RdsGuid::toString() const
{
	if (qxt_d().data->isNull)
		return "";
	return
	    INT_AS_HEX(qxt_d().data->timeLow)
	    + "-"
	    + INT_AS_HEX(qxt_d().data->timeMiddle)
	    + "-"
	    + INT_AS_HEX(qxt_d().data->timeHiAndVersion)
	    + "-"
	    + INT_AS_HEX(qxt_d().data->clockSequence)
	    + "-"
	    + INT_AS_HEX_SIZE(qxt_d().data->node, 6);
}

QByteArray RdsGuid::toBinary() const
{
// 	GUID Format: (128 bit total)
//		uint32 time_low
// 		uint16 time_mid
// 		uint16 time_hi_and_version
// 		uint8[2] clock_seq
// 		uint8[6] node

	QByteArray ar;
	
	quint16 clockSequence = qToBigEndian(qxt_d().data->clockSequence);
	quint64 node = qToBigEndian(qxt_d().data->node << 16); // 8 * 2, 2 being the difference between 6 and 8

	ar.append(QByteArray::fromRawData((char*)&qxt_d().data->timeLow, sizeof(quint32)));
	ar.append(QByteArray::fromRawData((char*)&qxt_d().data->timeMiddle, sizeof(quint16)));
	ar.append(QByteArray::fromRawData((char*)&qxt_d().data->timeHiAndVersion, sizeof(quint16)));
	ar.append(QByteArray::fromRawData((char*)&clockSequence, sizeof(quint16)));
	ar.append(QByteArray::fromRawData((char*)&node, 6));

	return ar;
}

bool RdsGuid::operator==(const RdsGuid &other) const
{
	VERIFY_VALUES(isNull);
	VERIFY_VALUES(timeLow);
	VERIFY_VALUES(timeMiddle);
	VERIFY_VALUES(timeHiAndVersion);
	VERIFY_VALUES(clockSequence);
	VERIFY_VALUES(node);
	return true;
}

bool RdsGuid::operator!=(const RdsGuid &other) const
{
	return !operator==(other);
}

bool RdsGuid::operator==(const QString &str) const
{
	return operator==(RdsGuid(str));
}

bool RdsGuid::operator!=(const QString &str) const
{
	return !operator==(RdsGuid(str));
}

bool RdsGuid::operator==(const QByteArray &str) const
{
	return operator==(RdsGuid(str));
}

bool RdsGuid::operator!=(const QByteArray &str) const
{
	return !operator==(RdsGuid(str));
}

bool RdsGuid::operator==(const char *str) const
{
	return operator==(RdsGuid(str));
}

bool RdsGuid::operator!=(const char *str) const
{
	return !operator==(RdsGuid(str));
}

ReturnValue RdsGuidPrivate::parseString(const QString &str)
{
	this->data->isNull = false;
	int offset = 0;
	{
		bool ok = true;
		do
		{
			if ((str.count() - offset) < 36)
			{
				this->data->isNull = true;
				return false;
			}
			QString(str.at(offset)).toInt(&ok, 16);
		}
		while (!ok && ++offset);
	}
	READ_INT_FROM_STRING(this->data->timeLow, str);
	offset++; //hyphen
	READ_INT_FROM_STRING(this->data->timeMiddle, str);
	offset++; //hyphen
	READ_INT_FROM_STRING(this->data->timeHiAndVersion, str);
	offset++; //hyphen
	READ_INT_FROM_STRING(this->data->clockSequence, str);
	offset++; //hyphen
	READ_INT_FROM_STRING_SIZE(this->data->node, str, 6);

	return true;
}

ReturnValue RdsGuidPrivate::parseByteArray(const QByteArray &data)
{
// 	GUID Format: (128 bit total)
//		uint32 time_low
// 		uint16 time_mid
// 		uint16 time_hi_and_version
// 		uint8[2] clock_seq
// 		uint8[6] node

	if (data.count() < 16)
	{
		this->data->isNull = true;
		return false;
	}
	this->data->isNull = false;

	const char* dataPtr = data.constData();
	const char* offset = dataPtr;

	READ_INT_VALUE(this->data->timeLow, offset);
	READ_INT_VALUE(this->data->timeMiddle, offset);
	READ_INT_VALUE(this->data->timeHiAndVersion, offset);
	READ_INT_VALUE_BIGENDIAN(this->data->clockSequence, offset);
	READ_INT_VALUE_SIZE_BIGENDIAN(this->data->node, offset, 6);

	return true;
}

QDebug operator<<(QDebug dbg, const RdsGuid& guid)
{
	dbg.nospace() << "RdsGuid(" << guid.toString() << ")";
	return dbg.space();
}

QDataStream& operator<<(QDataStream& d, const RdsGuid& guid)
{
	d << guid.qxt_d().data->isNull;
	d << guid.qxt_d().data->timeLow;
	d << guid.qxt_d().data->timeMiddle;
	d << guid.qxt_d().data->timeHiAndVersion;
	d << guid.qxt_d().data->clockSequence;
	d << guid.qxt_d().data->node;
	return d;
}

QDataStream& operator>>(QDataStream& d, RdsGuid& guid)
{
	d >> guid.qxt_d().data->isNull;
	d >> guid.qxt_d().data->timeLow;
	d >> guid.qxt_d().data->timeMiddle;
	d >> guid.qxt_d().data->timeHiAndVersion;
	d >> guid.qxt_d().data->clockSequence;
	d >> guid.qxt_d().data->node;
	return d;
}



