/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsgroup.h"
#include "rdsgroup_p.h"
#include <RdsSid>
#include <RdsUtils>
#include <QDebug>

RdsGroup::RdsGroup()
		: RdsAdObject()
{
	QXT_INIT_PRIVATE(RdsGroup);
}

RdsGroup::RdsGroup(const RdsGroup &group)
		: RdsAdObject(group.dn())
{
	QXT_INIT_PRIVATE(RdsGroup);
}

RdsGroup::RdsGroup(const QString &dn)
		: RdsAdObject(dn)
{
	QXT_INIT_PRIVATE(RdsGroup);
}


RdsGroup::~RdsGroup()
{
}

ReturnValue RdsGroup::createGroup(const QString &dn)
{
	QString name = dn;
	name.replace("cn=","",Qt::CaseInsensitive);
	QStringList parts = name.split(",");
	if(parts.size() == 0) return("Malformed group DN: " + dn);
	
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "objectClass", "group");
	actions.add(RdsLdapActions::Add, "sAMAccountName", parts[0]);
	ReturnValue ret = rdsLdapSession()->add(dn, actions);
	if (ret.isError()) return(ret);
	else return(dn);
}

ReturnValue RdsGroup::cn() const
{
	ReturnValue ret = readAttribute("cn");
	if (ret.isError()) return(ret);
	LdapValues values = ret.value<LdapValues>();
	return(QString(values[0]));
}

ReturnValue RdsGroup::setCn(const QString &cn)
{
	return(rename("CN=" + cn));
}

ReturnValue RdsGroup::members() const
{
	ReturnValue ret = read(QStringList() << "member");
	if (ret.isError()) return(ret);

	QStringList list;

	LdapResult result = ret.value<LdapResult>();
	if (result.contains("member"))
	{

		foreach(QByteArray value, result["member"])
		{
			list << value;
		}
	}

	ret = sid();
	if (ret.isError()) return(list);
	RdsSid tmpsid = ret.value<RdsSid>();

	//qDebug() << "Searching:" << QString("(primaryGroupID = %1)").arg(tmpsid.rid());
	ret = rdsLdapSession()->list(RdsUtils::baseDn(), QString("(primaryGroupID=%1)").arg(tmpsid.rid()), true);
	if (ret.isError())
	{
		qWarning() << "Failed to search for primary groups:" << ret;
		return(list);
	}

	//qDebug() << "LIST:" << list << ret.toStringList();
	list << ret.toStringList();

	return(list);
}

ReturnValue RdsGroup::addMember(const QString &dn)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "member", dn);
	return(modify(actions));
}

ReturnValue RdsGroup::removeMember(const QString &dn)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Remove, "member", dn);
	return(modify(actions));
}


ReturnValue RdsGroup::groups() const
{
	ReturnValue ret = read(QStringList() << "memberOf");
	if (ret.isError()) return(ret);

	LdapResult result = ret.value<LdapResult>();
	if (!result.contains("memberof")) return(QStringList());

	QStringList list;
	foreach(QByteArray value, result["memberof"])
	{
		list << value;
	}

	return(list);
}

ReturnValue RdsGroup::joinGroup(const QString &groupdn)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Add, "member", dn());
	return(rdsLdapSession()->modify(groupdn, actions));

}

ReturnValue RdsGroup::leaveGroup(const QString &groupdn)
{
	RdsLdapActions actions;
	actions.add(RdsLdapActions::Remove, "member", dn());
	return(rdsLdapSession()->modify(groupdn, actions));
}

ReturnValue RdsGroup::description() const
{
	ReturnValue ret = read();
	if (ret.isError()) return(ret);
	LdapResult entry = ret.value<LdapResult>();

	if (!entry.contains("description") || entry["description"].count() == 0) return("");
	else return(entry["description"][0]);
}

ReturnValue RdsGroup::setDescription(const QString &desc)
{
	RdsLdapActions actions;

	if (desc == "")
		actions.add(RdsLdapActions::Remove, "description", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "description", desc);
	return(modify(actions));
}

ReturnValue RdsGroup::notes() const
{
	ReturnValue ret = readAttribute("info");
	if (ret.isError()) return(ret);

	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setNotes(const QString &notes)
{
	RdsLdapActions actions;

	if (notes == "")
		actions.add(RdsLdapActions::Remove, "info", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "info", notes);

	return(modify(actions));
}

ReturnValue RdsGroup::displayName() const
{
	ReturnValue ret = readAttribute("displayname");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setDisplayName(const QString &name)
{
	RdsLdapActions actions;
	if (name == "")
		actions.add(RdsLdapActions::Remove, "displayName", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "displayName", name);
	
	return(modify(actions));
}

ReturnValue RdsGroup::office() const
{
	ReturnValue ret = readAttribute("physicalDeliveryOfficeName");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setOffice(const QString &office)
{
	RdsLdapActions actions;
	if (office == "")
		actions.add(RdsLdapActions::Remove, "physicalDeliveryOfficeName", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "physicalDeliveryOfficeName", office);
	
	return(modify(actions));
}

ReturnValue RdsGroup::phoneNumber() const
{
	ReturnValue ret = readAttribute("telephoneNumber");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setPhoneNumber(const QString &number)
{
	RdsLdapActions actions;
	if (number == "")
		actions.add(RdsLdapActions::Remove, "telephoneNumber", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "telephoneNumber", number);
	
	return(modify(actions));
}

ReturnValue RdsGroup::email() const
{
	ReturnValue ret = readAttribute("mail");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setEmail(const QString &mail)
{
	RdsLdapActions actions;
	if (mail == "")
		actions.add(RdsLdapActions::Remove, "mail", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "mail", mail);
	
	return(modify(actions));
}

ReturnValue RdsGroup::webPage() const
{
	ReturnValue ret = readAttribute("wWWHomePage");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setWebPage(const QString &url)
{
	RdsLdapActions actions;
	if (url == "")
		actions.add(RdsLdapActions::Remove, "wWWHomePage", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "wWWHomePage", url);
	
	return(modify(actions));
}

ReturnValue RdsGroup::street() const
{
	ReturnValue ret = readAttribute("streetAddress");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setStreet(const QString &street)
{
	RdsLdapActions actions;
	if (street == "")
		actions.add(RdsLdapActions::Remove, "streetAddress", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "streetAddress", street);
	
	return(modify(actions));
}

ReturnValue RdsGroup::poBox() const
{
	ReturnValue ret = readAttribute("postOfficeBox");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setPoBox(const QString &po)
{
	RdsLdapActions actions;
	if (po == "")
		actions.add(RdsLdapActions::Remove, "postOfficeBox", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "postOfficeBox", po);
	
	return(modify(actions));
}

ReturnValue RdsGroup::city() const
{
	ReturnValue ret = readAttribute("l");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setCity(const QString &city)
{
	RdsLdapActions actions;
	if (city == "")
		actions.add(RdsLdapActions::Replace, "l", city);
	else
		actions.add(RdsLdapActions::Replace, "l", city);
	
	return(modify(actions));
}

ReturnValue RdsGroup::state() const
{
	ReturnValue ret = readAttribute("st");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setState(const QString &state)
{
	RdsLdapActions actions;
	
	if (state == "")
		actions.add(RdsLdapActions::Remove, "st", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "st", state);
	
	return(modify(actions));
}

ReturnValue RdsGroup::country() const
{
	ReturnValue ret = readAttribute("c");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setCountry(const QString &country)
{
	RdsLdapActions actions;
	if (country == "")
		actions.add(RdsLdapActions::Remove, "c", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "c", country);
	
	return(modify(actions));
}

ReturnValue RdsGroup::postalCode() const
{
	ReturnValue ret = readAttribute("postalCode");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setPostalCode(const QString &code)
{
	RdsLdapActions actions;
	if (code == "")
		actions.add(RdsLdapActions::Remove, "postalCode", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "postalCode", code);
	
	return(modify(actions));
}


ReturnValue RdsGroup::pagerNumber() const
{
	ReturnValue ret = readAttribute("pager");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setPagerNumber(const QString &pager)
{
	RdsLdapActions actions;
	
	if (pager == "")
		actions.add(RdsLdapActions::Remove, "pager", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "pager", pager);
	
	return(modify(actions));
}

ReturnValue RdsGroup::mobileNumber() const
{
	ReturnValue ret = readAttribute("mobile");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setMobileNumber(const QString &number)
{
	RdsLdapActions actions;
	
	if (number == "")
		actions.add(RdsLdapActions::Remove, "mobile", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "mobile", number);
	
	return(modify(actions));
}

ReturnValue RdsGroup::faxNumber() const
{
	ReturnValue ret = readAttribute("facsimileTelephoneNumber");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setFaxNumber(const QString &number)
{
	RdsLdapActions actions;
	
	if (number == "")
		actions.add(RdsLdapActions::Remove, "facsimileTelephoneNumber", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "facsimileTelephoneNumber", number);
	
	return(modify(actions));
}

ReturnValue RdsGroup::homeNumber() const
{
	ReturnValue ret = readAttribute("homePhone");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setHomeNumber(const QString &number)
{
	RdsLdapActions actions;
	
	if (number == "")
		actions.add(RdsLdapActions::Remove, "homePhone", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "homePhone", number);
	
	return(modify(actions));
}


ReturnValue RdsGroup::title() const
{
	ReturnValue ret = readAttribute("title");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setTitle(const QString &title)
{
	RdsLdapActions actions;
	
	if (title == "")
		actions.add(RdsLdapActions::Remove, "title", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "title", title);
	
	return(modify(actions));
}

ReturnValue RdsGroup::department() const
{
	ReturnValue ret = readAttribute("department");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setDepartment(const QString &department)
{
	RdsLdapActions actions;
	
	if (department == "")
		actions.add(RdsLdapActions::Remove, "department", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "department", department);
	
	return(modify(actions));
}

ReturnValue RdsGroup::company() const
{
	ReturnValue ret = readAttribute("company");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setCompany(const QString &company)
{
	RdsLdapActions actions;
	
	if (company == "")
		actions.add(RdsLdapActions::Remove, "company", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "company", company);
	
	return(modify(actions));
}

ReturnValue RdsGroup::emailAliases()
{
	QStringList list;
	ReturnValue ret = read(QStringList() << "proxyaddresses");
	if (ret.isError()) return(ret);
	
	LdapResult result = ret.value<LdapResult>();
	if (!result.contains("proxyaddresses")) return(list);
	
	foreach(QByteArray value, result["proxyaddresses"])
	{
		list << value;
	}
	
	return(list);
}

ReturnValue RdsGroup::setEmailAliases(QStringList aliases)
{
	RdsLdapActions actions;
	if (aliases.size() == 0)
		actions.add(RdsLdapActions::Remove, "proxyaddresses", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "proxyaddresses", aliases);
	return(modify(actions));
}

ReturnValue RdsGroup::accountName()
{
	ReturnValue ret = readAttribute("samaccountname");
	if (ret.isError()) return(ret);
	
	LdapValues value = ret.value<LdapValues>();
	return(QString(value[0]));
}

ReturnValue RdsGroup::setAccountName(QString name)
{
	RdsLdapActions actions;
	
	if (name == "")
		actions.add(RdsLdapActions::Remove, "samaccountname", QStringList());
	else
		actions.add(RdsLdapActions::Replace, "samaccountname", name);
	
	return(modify(actions));
}


QDataStream& operator<<(QDataStream& d, const RdsGroup& object)
{
	d << object.dn();
	return(d);
}

QDataStream& operator>>(QDataStream& d, RdsGroup& object)
{
	QString dn;
	d >> dn;
	object.setDn(dn);
	return(d);
}
