/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsvolume.h"
#include "rdsvolume_p.h"
#include "rdsstoragedevice_p.h"

#include <QTemporaryFile>
#include <QProcess>
#include <QFile>
#include <QFileInfo>
#include <QDebug>

#include "RdsSettings"
#include "RdsFileManager"
#include "RdsUtils"
#include "rdssinglevolume_p.h"

#include <sys/vfs.h>
#include <errno.h>
#include <attr/xattr.h>
#include "rdsstoragemanager_p.h"
#include "rdsvolumemanager_p.h"

using namespace QtRpc;

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsVolume);
RDS_REGISTER_VOLUME(RdsVolume);

#define RDS_LOCK QMutexLocker __locker(qxt_d().getMutex());
#define CREATE_DBUS_OBJECT RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), qxt_d().path, QDBusConnection::systemBus());
#define RDS_CHECK_UDISKS \
	if (!RdsSingleVolumePrivate::hasUDisks()) return RdsSingleVolume()

#define ITERCOUNT 20
#define ENDBENCH(section) \
	qDebug() << section " initialization at a rate of" << (1000.0f * ITERCOUNT / timer.elapsed()) << "per second";

QMutex* RdsVolumePrivate::getMutex() const
{
	return const_cast<QMutex*>(&mutex);
}

void RdsVolumePrivate::init(const QString& volume)
{
	if (!this->path.isEmpty())
		this->path = QString();
	QString uuid;
	QString name;

	static unsigned char rotater = 0;
	rotater++;
	bool updateData = (rotater & 0x1) == 0x1;
	
	if (QRegExp("[A-Fa-f0-9]{8}-[A-Fa-f0-9]{4}-[A-Fa-f0-9]{4}-[A-Fa-f0-9]{4}-[A-Fa-f0-9]{12}").exactMatch(volume) || QRegExp("[A-Fa-f0-9]{16}").exactMatch(volume))
	{
		uuid = volume;
	}
	else
	{
		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + volume);
		if (!rdssettings()->value("uuid", QString()).toString().isEmpty())
		{
			uuid = rdssettings()->value("uuid", QString()).toString();
		}
		rdssettings()->endGroup();
		rdssettings().unlock();
	}
	if (volume.startsWith("/org/freedesktop/UDisks/devices/"))
	{
		RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), volume, QDBusConnection::systemBus());
		if (!isVolume(device))
			return;
		this->path = volume;
		uuid = device.idUuid();
		name = device.idLabel();
	}
	else
	{
		RdsUDisksInterface interface(RdsUDisksInterface::staticInterfaceName(), "/org/freedesktop/UDisks", QDBusConnection::systemBus());
		QDBusPendingReply<QList<QDBusObjectPath> > ret = interface.EnumerateDevices();
		ret.waitForFinished();
		if (!ret.isError())
		{
			foreach(QDBusObjectPath path, ret.value())
			{
				RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), path.path(), QDBusConnection::systemBus());
				if (!isVolume(device))
					continue;
				if ((!uuid.isEmpty() && device.idUuid() == uuid) || device.deviceFile() == volume || device.idLabel() == volume)
				{
					this->path = path.path();
					uuid = device.idUuid();
					name = device.idLabel();
					break;
				}
			}
		}
	}
	rdssettings().lock();
	rdssettings()->beginGroup("volume-" + name);
	if (!rdssettings()->contains("uuid"))
	{
		rdssettings()->endGroup();

		foreach(QString group, rdssettings()->childGroups())
		{
			if (!group.startsWith("volume-"))
				continue;

			if (group == ("volume-" + volume))
			{
				this->volume = volume;
				break;
			}
			if (group == ("volume-" + name))
			{
				this->volume = name;
				break;
			}
			if (uuid.isEmpty())
				break;

			rdssettings()->beginGroup(group);
			if (rdssettings()->value("uuid", QString()).toString() == uuid)
			{
				rdssettings()->endGroup();
				this->volume = group.mid(7);
				break;
			}
			rdssettings()->endGroup();
		}
	}
	else
	{
		this->volume = name;
		rdssettings()->endGroup();
	}
	rdssettings().unlock();
	if (!this->volume.isEmpty() && !this->path.isEmpty())
	{
		if (!qxt_p().isAdded().toBool())
			return; //Don't bother

		if (this->path == volume)
			return;
		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + this->volume);


		RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), this->path, QDBusConnection::systemBus());
		if (updateData || !rdssettings()->contains("uuid"))
			rdssettings()->setValue("uuid", uuid);
		if (updateData || !rdssettings()->contains("size"))
			rdssettings()->setValue("size", device.deviceSize());
		if (updateData || !rdssettings()->contains("fsType"))
			rdssettings()->setValue("fsType", device.idType());
		if (updateData || !rdssettings()->contains("model"))
			rdssettings()->setValue("model", device.driveModel());
		if (updateData || !rdssettings()->contains("serial"))
			rdssettings()->setValue("serial", device.driveSerial());

		if (updateData || !rdssettings()->contains("type"))
		{
			QString type;
			if (device.deviceIsOpticalDisc())
				type = "CDROM";
			else if (device.driveConnectionInterface() == "usb")
				type = "USB Disk";
			else
				type = "Hard Disk";
			rdssettings()->setValue("type", type);
		}

		if (updateData || !rdssettings()->contains("supportsNtAcls"))
		{
			ReturnValue ret = qxt_p().supportsNtAcls();
			if (!ret.isError())
				rdssettings()->setValue("supportsNtAcls", ret.toBool());
		}

		if (device.idLabel() != this->volume)
			device.FilesystemSetLabel(this->volume);

		rdssettings()->endGroup();
		rdssettings().unlock();
	}
}

RdsVolume::RdsVolume()
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsVolume);
}

RdsVolume::RdsVolume(const QString &volume)
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsVolume);

	qxt_d().init(volume);
	/*
	QTime timer;
	timer.start();
	for (int i = 0; i < ITERCOUNT; ++i)
	{
		qxt_d().init(volume);
	}
	qDebug() << "Volume initialization at a rate of" << (1000.0f * ITERCOUNT / timer.elapsed()) << "per second";
	*/
}

RdsVolume::RdsVolume(const char *volume)
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsVolume);
	qxt_d().init(volume);
	/*
	QTime timer;
	timer.start();
	for (int i = 0; i < ITERCOUNT; ++i)
	{
		qxt_d().init(volume);
	}
	qDebug() << "Volume initialization at a rate of" << (1000.0f * ITERCOUNT / timer.elapsed()) << "per second";
	*/
}

RdsVolume::RdsVolume(const RdsVolume &other)
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsVolume);
	qxt_d().init(other.id().toString());
	/*
	QTime timer;
	timer.start();
	for (int i = 0; i < ITERCOUNT; ++i)
	{
		qxt_d().init(other.id().toString());
	}
	qDebug() << "Volume initialization at a rate of" << (1000.0f * ITERCOUNT / timer.elapsed()) << "per second";
	*/
}

RdsVolume::~RdsVolume()
{
}

ReturnValue RdsVolume::initialize()
{
	RDS_CHECK_UDISKS .initialize();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Device is offline");
	}
	bool isAdded = this->isAdded().toBool();
	QString myName = name().toString();
	QString oldId = id().toString();
	if (isMounted().toBool())
		return ReturnValue(1, "Cannot initialize an active volume");
	ReturnValue ret = RdsStorageDevicePrivate::setPartitionType(deviceFile().toString(), "ext3");
	if (ret.isError())
		qCritical() << "Failed to change partition type:" << ret;
	ret = RdsStorageDevicePrivate::createFilesystem(deviceFile().toString(), "ext3");
	QString newId;
	for (int i = 0; i < 50; ++i)
	{
		newId = id().toString();
		if (!newId.isEmpty())
			break;
		usleep(100000); // :<
	}
	if (newId.isEmpty())
		return ReturnValue(1, "Failed to detect the newly initialized device.");
	CREATE_DBUS_OBJECT;
	if (isAdded)
	{
		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		rdssettings()->setValue("uuid", newId);
		rdssettings()->endGroup();
		rdssettings().unlock();
	}
	if (!myName.isEmpty())
		device.FilesystemSetLabel(myName).waitForFinished();
	mount();
// 	qDebug() << "I just changed my name from" << oldId << "to" << newId;
	RdsVolumeManager().removeEntity(oldId);
	RdsVolumeManager().addEntity(newId);
	RdsVolumeManager().updateEntity(newId);
	return newId;
}

ReturnValue RdsVolume::mountPoint() const
{
	RDS_CHECK_UDISKS .mountPoint();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Device is offline");
	}
	CREATE_DBUS_OBJECT;

	return device.deviceMountPaths();
}

ReturnValue RdsVolume::path() const
{
	RDS_CHECK_UDISKS .path();
	RDS_LOCK;
	return "/volumes/" + qxt_d().volume;
}

ReturnValue RdsVolume::mount(const QString &path, const QStringList &options)
{
	RDS_CHECK_UDISKS .mount(path, options);
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Cannot mount offline devices");
	CREATE_DBUS_OBJECT;
	if (!QFile::exists(path))
	{
		QDir().mkdir(path);
	}
	QFileInfo info(path);
	if (!info.isDir())
		return ReturnValue(1, "The selected mount point is not a directory.");
	int ret = QProcess::execute("mount", QStringList() << device.deviceFile() << path << options);
	if (ret == 0)
		return true;
	QString reason;
	if (ret & 32)
		reason = "Mount failed: ";
	if (ret & 64)
		reason = "Mount partially succeeded: ";
	if ((ret & 31) == 0)
		reason += "Unknown reason.";
	if (ret & 1)
		reason += "Permission denied; ";
	if (ret & 2)
		reason += "System error; ";
	if (ret & 4)
		reason += "Internal mount bug; ";
	if (ret & 8)
		reason += "Unexpectedly interrupted; ";
	if (ret & 16)
		reason += "Permission denied";
	return ReturnValue(1, reason);
}

ReturnValue RdsVolume::isMounted() const
{
	RDS_CHECK_UDISKS .isMounted();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Device is offline");
	}
	CREATE_DBUS_OBJECT;
	return device.deviceIsMounted();
}

ReturnValue RdsVolume::size() const
{
	RDS_CHECK_UDISKS .size();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		quint64 ret = rdssettings()->value("size", 0).toULongLong();
		rdssettings()->endGroup();
		rdssettings().unlock();
		return ret;
	}
	CREATE_DBUS_OBJECT;
	return device.deviceSize();
}

ReturnValue RdsVolume::mkfs(const QString &type, const QStringList &options, quint64 blocks)
{
	RDS_CHECK_UDISKS .mkfs(type, options, blocks);
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Cannot make filesystems on offline devices");
	int ret = 0;
	if (blocks == 0)
		ret = QProcess::execute("mkfs", QStringList() << "-t" << type << options << qxt_d().volume);
	else
		ret = QProcess::execute("mkfs", QStringList() << "-t" << type << options << qxt_d().volume << QString::number(blocks));
	if (ret == 1)
		return ReturnValue(1, "Failed to run mkfs");

	return true;
}

ReturnValue RdsVolume::isAdded() const
{
	RDS_CHECK_UDISKS .isAdded();
	RDS_LOCK;
	if (!rdssettings()->childGroups().contains("volume-" + qxt_d().volume))
		return false;

	QString uuid = id().toString();
	if (uuid.isEmpty())
		return true;

	rdssettings().lock();
	rdssettings()->beginGroup("volume-" + qxt_d().volume);

	bool isitme = (rdssettings()->value("uuid").toString() == uuid);

	rdssettings()->endGroup();
	rdssettings().unlock();

	return isitme;
}

ReturnValue RdsVolume::isActivated() const
{
	RDS_CHECK_UDISKS .isActivated();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Device is offline");
	}

	QString rdspath = ("/volumes/" + qxt_d().volume);

	if (!QFile::exists(rdspath))
		return false;

	ReturnValue ret = mountPoint();
	if (ret.isError())
	{
		qWarning() << "Mount point returned an error:" << ret;
		return false;
	}

	QString mp = ret.toString();

	ret = RdsVolumeManager().rootVolume();
	if (!ret.isError() && id().toString() != ret.toString())
	{
		RdsVolume v(ret.toString());
		if (qxt_d().volume == v.name())
			return ReturnValue(2, "Another device is active in this device's place in volumes.");
	}

	if (RdsFileManager::isMounted(rdspath))
	{
		if (mp == RdsUtils::followSymLink(rdspath))
			return true;
	}
	else if (mp == "/")
		return true;
	else
		return false;

	return ReturnValue(2, "Another device is active in this device's place in volumes.");
}

ReturnValue RdsVolume::storageDevice()
{
	RDS_CHECK_UDISKS .storageDevice();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Device is offline");
	}
	CREATE_DBUS_OBJECT;

	QString volumeid = device.idUuid();
	RdsUDisksInterface interface(RdsUDisksInterface::staticInterfaceName(), "/org/freedesktop/UDisks", QDBusConnection::systemBus());
	QDBusPendingReply<QList<QDBusObjectPath> > ret = interface.EnumerateDevices();
	ret.waitForFinished();
	if (ret.isError())
		return ReturnValue(1, ret.error().message());
	foreach(QDBusObjectPath path, ret.value())
	{
		{
			RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), path.path(), QDBusConnection::systemBus());
			if (path.path() == "/org/freedesktop/UDisks/devices/fd0" || !device.deviceIsDrive() || device.deviceIsLinuxLoop())
				continue;
		}
		RdsStorageDevice device(path.path());

		if (device.listVolumes().toStringList().contains(volumeid))
			return new RdsStorageDevice(path.path());
	}
	if (device.deviceIsDrive() && !device.deviceIsLinuxLoop())
		return new RdsStorageDevice(device.path());

	return ReturnValue(1, "Failed to find device for this volume");
}

ReturnValue RdsVolumePrivate::findUnusedName()
{
	ReturnValue ret = qxt_p().name().toString(); // This will fill in qxt_d().volume with my current "name"... This name will be changed if it's in use...
	if (ret.isError())
		return ret; // This should _never_ happen
	QString oldname = volume;
	volume = ret.toString();
	bool needsRename = false;
	for (int i = 1; i < 20; ++i)
	{
		needsRename = false;

		if (!qxt_p().isAdded().toBool() && rdssettings()->childGroups().contains("volume-" + volume)) // A device by this name is already added
		{
			//qDebug() << "Renaming device because someone else already has that name";
			needsRename = true;
		}

		ret = qxt_p().isActivated();
		if (ret.isError())
		{
			//qDebug() << "Renaming device because activated said error" << ret;
			needsRename = true;
		}
		if (!needsRename)
			break;
		volume.replace(QRegExp("[0-9 ]*$"), "");
		volume.append(" " + QString::number(i));
	}
	QString name = volume;
	volume = oldname;
	if (needsRename)
		return ReturnValue(1, "Failed to find a suitable name for the device. Please check the status of the /volumes/ folder");
	return name;
}

ReturnValue RdsVolume::mount()
{
	RDS_CHECK_UDISKS .mount();
	RDS_LOCK;
	if (isActivated().toBool())
		return true;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Cannot mount disconnected devices");
	}
	ReturnValue ret = qxt_d().findUnusedName();
	if (ret.isError())
		return ret;
	qxt_d().volume = ret.toString();

	{
		CREATE_DBUS_OBJECT;
		QDBusPendingReply<> ret = device.FilesystemSetLabel(qxt_d().volume);
		ret.waitForFinished();
	}

	QString mountPoint = "/volumes/" + qxt_d().volume;
	if (isMounted().toBool())
	{
		if (id().toString() == RdsVolumeManager().rootVolume().toString())
		{
			// This is the root device, so simple do the thing at the place with the awesome.
			QDir().mkdir(mountPoint);
			return true;
		}
		QString mountedPoint = this->mountPoint().toString();
		if (!QFile::exists(mountedPoint))
			return ReturnValue(1, "The existing mount point appears to not exist.");
		return (symlink(qPrintable(mountedPoint), qPrintable(mountPoint)) == -1) ? ReturnValue(1, strerror(errno)) : true;
	}
	QStringList options;
	if (fsType().toString().startsWith("ext"))
		options << "-o" << "acl,user_xattr";
	return mount(mountPoint, options);
}

ReturnValue RdsVolume::unmount()
{
	RDS_CHECK_UDISKS .unmount();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Device is offline");
	}
	if (!isMounted().toBool())
		return true;
	if (mountPoint().toString() == "/")
		return ReturnValue(1, "You may not unmount the root device.");
	QString mountPoint = "/volumes/" + name().toString() + "/";
	QProcess p;
	p.setProcessChannelMode(QProcess::MergedChannels);
	p.start("umount", QStringList() << mountPoint);
	if (!p.waitForFinished(25000))
	{
		qWarning() << p.readAll();
		return(ReturnValue(1, "Unmount timed out"));
	}

	if (p.exitCode() != 0)
	{
		return(ReturnValue(1, p.readLine()));
	}

	QDir().rmdir(mountPoint);

	return(true);
}

ReturnValue RdsVolume::deviceFile() const
{
	RDS_CHECK_UDISKS .deviceFile();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Device is offline");
	}
	CREATE_DBUS_OBJECT;
	return device.deviceFile();
}

ReturnValue RdsVolume::freeSpace() const
{
	RDS_CHECK_UDISKS .freeSpace();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");
		return ReturnValue(1, "Device is offline");
	}
	if (!isMounted().toBool())
		return ReturnValue(1, "Device is not mounted.");

	ReturnValue ret = mountPoint();
	if (ret.isError())
		return ret;
	if (!QFile::exists(ret.toString()))
		return ReturnValue("The mount point no longer exists.");
	struct statfs buff;
	if (statfs(qPrintable(ret.toString()), &buff) != 0)
		return ReturnValue(1, strerror(errno));

	quint64 value = 0xC0000000;
	value = value << 32;
	if ((static_cast<quint64>(buff.f_blocks)) & value)
		return ReturnValue(1, "Size is too large to be displayed in a 64 bit integer");
	return (static_cast<qlonglong>(buff.f_bfree) * buff.f_bsize);
}

ReturnValue RdsVolume::supportsNtAcls() const
{
	RDS_LOCK;
	if (!isMounted().toBool())
	{
		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		QVariant size = rdssettings()->value("supportsNtAcls", QVariant());
		rdssettings()->endGroup();
		rdssettings().unlock();
		if (size.isNull())
			return ReturnValue(1, "Device is not mounted.");
		else
			return size.toBool();
	}
	ReturnValue ret = mountPoint();
	if (ret.isError())
		return ret;
	if (!QFile::exists(ret.toString()))
		return ReturnValue("The mount point no longer exists.");
	if (listxattr(qPrintable(ret.toString()), 0, 0) == -1)
	{
		if (errno == ENOTSUP)
			return false;
	}
	return true;
}

ReturnValue RdsVolume::fsType() const
{
	RDS_CHECK_UDISKS .fsType();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");

		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		QVariant size = rdssettings()->value("fsType", QVariant());
		rdssettings()->endGroup();
		rdssettings().unlock();
		if (size.isNull())
			return ReturnValue(1, "Device is offline.");
		else
			return size.toString();
	}
	CREATE_DBUS_OBJECT;
	return device.idType();
}

ReturnValue RdsVolume::type() const
{
	RDS_CHECK_UDISKS .type();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");

		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		QVariant size = rdssettings()->value("type", QVariant());
		rdssettings()->endGroup();
		rdssettings().unlock();
		if (size.isNull())
			return ReturnValue(1, "Device is offline.");
		else
			return size.toString();
	}
	CREATE_DBUS_OBJECT;
	QString type = device.driveConnectionInterface();
	if (device.deviceIsOpticalDisc())
		type = "CDROM";
	else if (type == "usb")
		type = "USB Disk";
	else
		type = "Hard Disk";
	/*
	if (type == "linux_raid_member")
		type = "Raid Member";
	else if (type == "LVM2_member")
		type = "LVM2 Member";
	else if (type == "swap")
		type = "Swap";
	else if (type == "ext3")
		type = "Hard Disk";
	*/
	return type;
}

ReturnValue RdsVolume::status() const
{
	RDS_CHECK_UDISKS .status();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
		return "Disconnected";
	return "Normal";
}

ReturnValue RdsVolume::statusText() const
{
	RDS_CHECK_UDISKS .statusText();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
		return "Disconnected";
	if (!isActivated().toBool())
		return "Inactive";
	return "Active";
}

ReturnValue RdsVolume::statusDetails() const
{
	RDS_CHECK_UDISKS .statusDetails();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
		return "Disconnected";
	if (!isActivated().toBool())
		return "Inactive";
	return "Active";
}

ReturnValue RdsVolume::name()
{
	RDS_CHECK_UDISKS .name();
	RDS_LOCK;
	QString name = qxt_d().volume;
	if (!name.isEmpty())
		return name;
	else if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid volume object");

	CREATE_DBUS_OBJECT;
	name = device.idLabel();

	if (name.isEmpty())
		name = type().toString();

	if (name.isEmpty())
		name = "Volume";

	qxt_d().volume = name; // volume is currently empty so we fill it in
	return name;
	/*
	QString tmpname = name + "%1";
	for (int i = 1; true; ++i)
	{
		if (RdsFileManager::isMounted("/volumes/" + name))
		{
			QString path = "/volumes/" + name;
			for (int j = 0; QFileInfo(path).isSymLink() && j < 16; ++j)
			{
				QString sympath = QFileInfo(path).symLinkTarget();
				if (path == sympath)
					break;
				path = sympath;
			}
			if (QFileInfo(path).isSymLink())
				return ReturnValue(1, "Too many levels of symbolic links in " + path);
			if (path == mountPoint().toString())
			{
				setName(name);
				return name;
			}
		}
		else if (rdssettings()->childGroups().contains("volume-" + name))
		{
			rdssettings().lock();
			rdssettings()->beginGroup("volume-" + name);
			QString id = this->id().toString();
			if (qxt_d().path.isEmpty() || (!id.isEmpty() && id == rdssettings()->value("uuid").toString()))
			{
				rdssettings()->endGroup();
				rdssettings().unlock();
				setName(name);
				return name;
			}
			rdssettings()->endGroup();
			rdssettings().unlock();
		}
		else
		{
			setName(name);
			return name;
		}
		name = tmpname.arg(i);
	}
	return ReturnValue(1, "Failed to detect the devices name. Please manually unmount this device");
	*/
}

ReturnValue RdsVolume::id() const
{
	RDS_CHECK_UDISKS .id();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");

		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		QVariant size = rdssettings()->value("uuid", QVariant());
		rdssettings()->endGroup();
		rdssettings().unlock();
		if (size.isNull())
			return ReturnValue(1, "Device is offline.");
		else
			return size.toString();
	}
	CREATE_DBUS_OBJECT;
	///@todo: This will break CD roms
	return device.idUuid();
}

ReturnValue RdsVolume::list()
{
	if (!RdsSingleVolumePrivate::hasUDisks())
	{
		return QStringList() << "Hard Drive";
	}
	QStringList list;

	foreach(QString group, rdssettings()->childGroups())
	{
		if (!group.startsWith("volume-"))
			continue;

		rdssettings().lock();
		rdssettings()->beginGroup(group);
		if (!list.contains(rdssettings()->value("uuid", QString()).toString()))
			list << rdssettings()->value("uuid", QString()).toString();
		rdssettings()->endGroup();
		rdssettings().unlock();
	}

	RdsUDisksInterface interface(RdsUDisksInterface::staticInterfaceName(), "/org/freedesktop/UDisks", QDBusConnection::systemBus());
	QDBusPendingReply<QList<QDBusObjectPath> > ret = interface.EnumerateDevices();
	ret.waitForFinished();
	if (!ret.isError())
	{
		foreach(QDBusObjectPath path, ret.value())
		{
			RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), path.path(), QDBusConnection::systemBus());
			if (RdsVolumePrivate::isVolume(device) && !list.contains(device.idUuid()))
			{
				///@todo: This will break CD roms
				list << device.idUuid();
			}
		}
	}
	list.removeAll(QString());

	return list;
}

ReturnValue RdsVolume::setName(const QString &n)
{
	RDS_CHECK_UDISKS .setName(n);
	RDS_LOCK;
	if (n == qxt_d().volume)
		return true;

	QString oldName = qxt_d().volume;
	qxt_d().volume = n;
	ReturnValue r = qxt_d().findUnusedName(); // I named it r because i use ret down there and i didn't want to change it
	if (r.isError())
		return r;
	qxt_d().volume = oldName;

	QString name = r.toString();

	QDBusPendingReply<> ret;
	if (!qxt_d().path.isEmpty())
	{
		CREATE_DBUS_OBJECT;
		if (device.idLabel() != name)
			ret = device.FilesystemSetLabel(name);
	}

	if (isAdded().toBool())
	{
		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		QMap<QString, QVariant> data;
		foreach(QString str, rdssettings()->childKeys())
		{
			data[str] = rdssettings()->value(str);
			rdssettings()->remove(str);
		}
		rdssettings()->endGroup();
		rdssettings().unlock();

		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + name);
		foreach(QString str, data.keys())
		{
			rdssettings()->setValue(str, data[str]);
		}
		rdssettings()->endGroup();
		rdssettings().unlock();
	}

	if (isActivated().toBool() && !((qxt_d().volume == "") || (qxt_d().volume == "/")))
	{
		//qDebug() << "Renaming mounted device:" << qxt_d().volume << name;
		if (QFileInfo("/volumes/" + qxt_d().volume).isSymLink())
		{
			RdsFileManager().rename("/volumes/" + qxt_d().volume, name);
		}
		else if (id().toString() == RdsVolumeManager().rootVolume())
		{
			RdsFileManager().rename("/volumes/" + qxt_d().volume, name);
		}
		else
		{
			QDir().mkdir("/volumes/" + name);
			QProcess::execute("mount", QStringList() << "--move" << "/volumes/" + qxt_d().volume << "/volumes/" + name);
			QDir().rmdir("/volumes/" + qxt_d().volume);
		}
	}
	qxt_d().volume = name;
	if (!qxt_d().path.isEmpty())
		ret.waitForFinished();
	return true;
}

ReturnValue RdsVolume::serialNumber() const
{
	RDS_CHECK_UDISKS .serialNumber();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");

		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		QVariant size = rdssettings()->value("serial", QVariant());
		rdssettings()->endGroup();
		rdssettings().unlock();
		if (size.isNull())
			return ReturnValue(1, "Device is offline.");
		else
			return size.toString();
	}
	CREATE_DBUS_OBJECT;
	return device.driveSerial();
}

ReturnValue RdsVolume::model() const
{
	RDS_CHECK_UDISKS .model();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
	{
		if (qxt_d().volume.isEmpty())
			return ReturnValue(1, "Invalid volume object");

		rdssettings().lock();
		rdssettings()->beginGroup("volume-" + qxt_d().volume);
		QVariant size = rdssettings()->value("model", QVariant());
		rdssettings()->endGroup();
		rdssettings().unlock();
		if (size.isNull())
			return ReturnValue(1, "Device is offline.");
		else
			return size.toString();
	}
	CREATE_DBUS_OBJECT;
	return device.driveModel();
}

ReturnValue RdsVolume::isEjectable() const
{
	RDS_CHECK_UDISKS .isEjectable();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
		return "Disconnected";
	CREATE_DBUS_OBJECT;
	return(device.driveIsMediaEjectable());
}

ReturnValue RdsVolume::eject()
{
	RDS_CHECK_UDISKS .eject();
	RDS_LOCK;
	if (qxt_d().path.isEmpty())
		return "Disconnected";
	CREATE_DBUS_OBJECT;
	device.DriveEject(QStringList());
	return(true); ///TODO is there a way to check for errors here, its asynchronous?
}

RdsVolume &RdsVolume::operator=(const RdsVolume & other)
{
	RDS_LOCK;
	QXT_INIT_PRIVATE(RdsVolume);
	qxt_d().init(other.id().toString());
	return(*this);
}

bool RdsVolumePrivate::isVolume(RdsUDisksDeviceInterface& device)
{
	///@todo: This will break CD roms
	return /*(device.deviceIsOpticalDisc() && !device.idType().isEmpty()) || */((device.deviceIsPartition() || !device.idType().isEmpty()) && !device.idUuid().isEmpty() && !device.deviceIsLinuxLoop() && device.partitionType() != "0x05" && device.idType() != "swap" && device.idType() != "linux_raid_member" && device.idType() != "LVM2_member");
}

/*
 * Implementation of interface class RdsUDisksInterface
 */

RdsUDisksInterface::RdsUDisksInterface(const QString &service, const QString &path, const QDBusConnection &connection, QObject *parent)
		: QDBusAbstractInterface(service, path, staticInterfaceName(), connection, parent)
{
}

RdsUDisksInterface::~RdsUDisksInterface()
{
}

/*
 * Implementation of interface class RdsUDisksAdapterInterface
 */

RdsUDisksAdapterInterface::RdsUDisksAdapterInterface(const QString &service, const QString &path, const QDBusConnection &connection, QObject *parent)
		: QDBusAbstractInterface(service, path, staticInterfaceName(), connection, parent)
{
}

RdsUDisksAdapterInterface::~RdsUDisksAdapterInterface()
{
}

/*
 * Implementation of interface class RdsUDisksDeviceInterface
 */

RdsUDisksDeviceInterface::RdsUDisksDeviceInterface(const QString &service, const QString &path, const QDBusConnection &connection, QObject *parent)
		: QDBusAbstractInterface(service, path, staticInterfaceName(), connection, parent)
{
}

RdsUDisksDeviceInterface::~RdsUDisksDeviceInterface()
{
}

/*
 * Implementation of interface class RdsUDisksExpanderInterface
 */

RdsUDisksExpanderInterface::RdsUDisksExpanderInterface(const QString &service, const QString &path, const QDBusConnection &connection, QObject *parent)
		: QDBusAbstractInterface(service, path, staticInterfaceName(), connection, parent)
{
}

RdsUDisksExpanderInterface::~RdsUDisksExpanderInterface()
{
}

/*
 * Implementation of interface class RdsUDisksPortInterface
 */

RdsUDisksPortInterface::RdsUDisksPortInterface(const QString &service, const QString &path, const QDBusConnection &connection, QObject *parent)
		: QDBusAbstractInterface(service, path, staticInterfaceName(), connection, parent)
{
}

RdsUDisksPortInterface::~RdsUDisksPortInterface()
{
}
