/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "rdsdnszone.h"
#include "rdsdnsrecord_p.h"
#include "rdsdnszone_p.h"
#include <QDebug>
#include <QTemporaryFile>
#include <QProcess>
#include <QFileInfo>
#include <QDir>

#include "rdsglobal.h"
#include "rdsdnsmanager.h"
#include "rdsdnsmanager_p.h"

using namespace QtRpc;

#define CHECK_VALIDITY \
	if(qxt_d().data == 0) \
		return ReturnValue(ReturnValue::GenericError, "This zone object has no data"); \
	qxt_d().data->mutex.lock(); \
	if(!qxt_d().data->valid) \
	{ \
		QString error = "Invalid zone object (" + qxt_d().data->errorString + ")"; \
		qxt_d().data->mutex.unlock(); \
		return ReturnValue(ReturnValue::GenericError, error); \
	} \
	if (qxt_d().domainName.isEmpty()) \
	{ \
		if (qxt_d().data->domainName.isEmpty()) \
		{ \
			qCritical() << this << "Empty domain name found on a zone!" << Q_FUNC_INFO << qxt_d().data->domainName; \
			return ReturnValue(1, "Zone object does not know what domain it belongs to."); \
		} \
		const_cast<QString&>(qxt_d().domainName) = qxt_d().data->domainName; \
	} \
	qxt_d().data->mutex.unlock();

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsDnsZone);

QMutex &RdsDnsZonePrivate::dataMutex()
{
	static QMutex dataMutex(QMutex::Recursive);
	return dataMutex;
}

QHash<QString, RdsDnsZoneData* > &RdsDnsZonePrivate::dataHash()
{
	static QHash<QString, RdsDnsZoneData* > dataHash;
	return dataHash;
}

RdsDnsZone::RdsDnsZone()
		: ServiceProxy(NULL)
{
	qxt_d().data = new RdsDnsZoneData();
	connect(qxt_d().data, SIGNAL(removed()), this, SLOT(removed()));
}

RdsDnsZone::RdsDnsZone(RdsDnsManager* mgr)
		: ServiceProxy(NULL)
{
	qxt_d().data = new RdsDnsZoneData();
	qxt_d().data->manager = mgr;
	connect(qxt_d().data, SIGNAL(removed()), this, SLOT(removed()));
}

RdsDnsZone::RdsDnsZone(const RdsDnsZone& other)
		: ServiceProxy(NULL)
{
	operator=(other);
}

RdsDnsZone::~RdsDnsZone()
{
}

RdsDnsZone& RdsDnsZone::operator=(const RdsDnsZone & other)
{
	qxt_d().dataMutex().lock();
	if (qxt_d().data)
	{
		disconnect(qxt_d().data, SIGNAL(removed()), this, SLOT(removed()));
		qxt_d().data->deref();
		qxt_d().data = 0;
	}
	qxt_d().data = other.qxt_d().data;
	if (qxt_d().data)
	{
		connect(qxt_d().data, SIGNAL(removed()), this, SLOT(removed()));
		qxt_d().data->ref();
		QMutexLocker locker(qxt_d().data->getMutex());
		qxt_d().domainName = qxt_d().data->domainName;
	}
	qxt_d().dataMutex().unlock();
	return *this;
}

ReturnValue RdsDnsZone::isValid() const
{
	QMutexLocker locker(qxt_d().data->getMutex());
	return qxt_d().data->valid;
}

ReturnValue RdsDnsZone::errorString() const
{
	QMutexLocker locker(qxt_d().data->getMutex());
	return qxt_d().data->errorString;
}

#define FAIL_PARSING(x) \
	{ \
		qCritical() << "Failed to parse DNS zone information:" << x; \
		qxt_d().data = new RdsDnsZoneData(); \
		connect(qxt_d().data, SIGNAL(removed()), this, SLOT(removed())); \
		qxt_d().data->errorString = x; \
		qxt_d().data->valid = false; \
		qxt_d().data->mutex.unlock(); \
		return ReturnValue(ReturnValue::GenericError, qxt_d().data->errorString); \
	}

ReturnValue RdsDnsZone::parse(const QString &config)
{
	RdsDnsZoneData* data = new RdsDnsZoneData();
	data->deref();
	data->valid = false;
	data->errorString = "Unknown error";
	bool endOfLine = false;
	quint32 lineNumber = 1;
	quint32 parentheses = 0;
	QStringList lineWords;
	QString ttl;
	QString origin;
	QString word;
	QString line;
	QString key;
	QStringList comments;
	QHash<QString, QHash<RecordType, QString> > ttlCache;
	for (QString::const_iterator i = config.begin(); i != config.end(); ++i)
	{
		if (*i == '\n')
			lineNumber++;
		if (endOfLine)
		{
			if (*i == '\n')
			{
				endOfLine = false;
			}
			else
			{
				comments[comments.count() - 1] += *i;
				continue;
			}
		}
		if (*i == '\n' || *i == ' ' || *i == '\t')
		{
			// word seperation!
			word = word.trimmed();
			if (word.isEmpty())
			{
				if (lineWords.count() == 0 && *i != '\n') // This line starts with whitespace! Oh snap!
					lineWords << QString();
			}
			else // if word is not empty
			{
				lineWords << word;
			}
			word = QString();

			// This is the end!!
			// At this point we have a QStringList of all the values, so we begin parsing them
			if (parentheses == 0 && *i == '\n')
			{
				if (lineWords.count() == 0)
					continue;
				// First just check if it's a special value
				if (lineWords.first().startsWith("$"))
				{
					// This is an origin or ttl value
					QString specialKey = lineWords.takeFirst();
					if (specialKey == "$ORIGIN")
					{
						origin = lineWords.join(" ");
					}
					else if (specialKey == "$TTL")
					{
						ttl = lineWords.join(" ");
					}
					lineWords = QStringList();
					continue;
				}

				// If it's an empty key then we fill in the most recent key we got.
				if (lineWords.first().isEmpty())
				{
					if (key.isEmpty())
						FAIL_PARSING("Empty key found on line " + QString::number(lineNumber - 1) + " without a preceeding key");
					lineWords[0] = key;
				}
				key = lineWords.takeFirst();
				if (key == "@")
					key = origin;

				RecordType type = Invalid;

				if (lineWords.count() < 2)
					FAIL_PARSING("Not enough values on line " + QString::number(lineNumber - 1));
				if (lineWords.first() == "IN")
				{
					if (lineWords.count() < 3)
						FAIL_PARSING("Not enough values on line " + QString::number(lineNumber - 1));
					lineWords.takeFirst();
				}
				type = RdsDnsZone::stringToType(lineWords.takeFirst());
				QString value = lineWords.join(" ");

				QString localKey = key;
				QString localOrigin = origin;
				if (key.endsWith("."))
				{
					if (key != data->domainName && !key.endsWith("." + data->domainName))
						FAIL_PARSING("Out of zone key found on line " + QString::number(lineNumber - 1));

					if (key == data->domainName)
					{
						localOrigin = key;
					}
					else
					{
						localKey = key.left(key.indexOf('.'));
						localOrigin = (key.mid(key.indexOf('.') + 1));
					}
				}
				else
				{
					localKey = key.left(key.indexOf('.'));
					localOrigin = ((key.indexOf('.') > 0) ? (key.mid(key.indexOf('.') + 1) + "." + origin) : origin);
				}
				/**
				 * THIS IS WHERE I PUT THE VALUES IN
				 * 	key = key ("servname", "_kerberose", "test.blah.servers")
				 * 	origin = origin value ("resara.local.", "something.resara.local.")
				 * 	localKey = resolved key ("servname", "_kerberose", "test")
				 * 	localOrigin = resolved origin value ("resara.local.", "test.blah.resara.local.")
				 * 	ttl = ttl value ("123123", "1D")
				 * 	type = RecordType type of that record
				 * 	lineWords = QStringList of all the values after the type
				 * 	values = lineWords.join(" ")
				 * keys might be something like servname.blah in origina this.resara.local., that means the origin is blah.this.resara.local.
				 * with a key of servname. localKey and localOrigin contain the resolved versions of these values.
				 */
				if (type != SOA)
				{
					RdsDnsRecord record;
					record.setType(type);
					record.setKey(localKey);
					record.setComments(comments);
					record.setOrigin(localOrigin);
					// this is because, for example, in origin resara.local., a value of "servname" really means "servname.resara.local."
					if ((record.type() == CNAME || record.type() == SRV) && !value.contains('.'))
					{
						record.setValue(value + "." + origin);
					}
					else
					{
						record.setValue(value);
					}
					QString fullKey = localKey + "." + localOrigin;
					if (ttlCache.contains(fullKey) && ttlCache.value(fullKey).contains(type))
					{
						record.setTtl(ttlCache.value(fullKey).value(type));
					}
					else
					{
						record.setTtl(ttl);
						ttlCache[fullKey][type] = ttl;
					}

					if (!data->extraOrigins.contains(record.origin()))
						data->extraOrigins << record.origin();
					data->records << record;
				}
				else
				{
					// This is the allmighty SOA entry, the special entry with all the important stuff in it
					data->domainName = key;
					if(!data->domainName.endsWith(".")) data->domainName += ".";
					data->soaTtl = ttl;
					data->soaOrigin = origin;
					data->soaComments = comments;
					data->soaValues.clear();

					if (lineWords.count() != 7)
						FAIL_PARSING("Incorrect number of values in the SOA field");
					data->primaryNameServer = lineWords.takeFirst();
					data->adminNameServer = lineWords.takeFirst();
					data->soaValues = lineWords;
				}
				/**
				 * FINISHED INSERTING DATA
				 */
				lineWords = QStringList();
				comments = QStringList();
				word = QString();
				continue;
			}
			continue;
		}
		if (*i == ';')
		{
			comments << QString::Null();
			endOfLine = true;
			continue;
		}
		if (*i == '(')
		{
			parentheses++;
			continue;
		}
		if (*i == ')')
		{
			parentheses--;
			continue;
		}
		if (*i == '\\')
		{
			i++;
		}
		word += *i;
	}
	qxt_d().dataMutex().lock();
	if (qxt_d().data)
	{
		disconnect(qxt_d().data, SIGNAL(removed()), this, SLOT(removed()));
		qxt_d().data->deref();
	}
// 	if (qxt_d().dataHash().contains(data->domainName))
// 	{
// 		qxt_d().data = qxt_d().dataHash().value(key);
// 		qxt_d().data->ref();
// 	}
// 	else
	{
		qxt_d().data = new RdsDnsZoneData();
		if (!qxt_d().dataHash().contains(data->domainName))
			qxt_d().dataHash()[key] = qxt_d().data;
	}
	if (qxt_d().domainName.isEmpty())
	{
		qxt_d().domainName = data->domainName;
	}
	qxt_d().data->mutex.lock();
	qxt_d().dataMutex().unlock();
	*qxt_d().data = *data;
	connect(qxt_d().data, SIGNAL(removed()), this, SLOT(removed()));
	qxt_d().data->errorString = "No error";
	qxt_d().data->valid = true;
	qxt_d().data->mutex.unlock();
	return true;
}

ReturnValue RdsDnsZone::toString() const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	QString output;
	QStringList extraOrigins = qxt_d().data->extraOrigins;
	int maxKeyLength = 0;
	if (qxt_d().data->soaValues.count() < 5)
	{
		qCritical() << "Writing out invalid SOA values" << qxt_d().data->soaValues;
		return ReturnValue(ReturnValue::GenericError, "Incomplete SOA values");
	}
	int serial = qxt_d().data->soaValues[0].toInt();
	if (++serial == 0)
		serial++;
	qxt_d().data->soaValues[0] = QString::number(serial);
	foreach(RdsDnsRecord record, qxt_d().data->records)
	{
		maxKeyLength = qMax(maxKeyLength, record.key().count());
	}
	foreach(QString line, qxt_d().data->soaComments)
	{
		output += ";" + line + "\n";
	}
	QMap<QString, QMap<QString, RdsDnsRecordList> > sortedRecordHash;
	QString curTtl = (qxt_d().data->soaTtl.isEmpty() ? qxt_d().data->soaValues.at(4) : qxt_d().data->soaTtl);
	QString curOrigin = qxt_d().data->soaOrigin;
	sortedRecordHash[curOrigin][curTtl] = RdsDnsRecordList();
	if (!curOrigin.isEmpty())
		output += "$ORIGIN " + curOrigin + "\n";
	extraOrigins.removeAll(curOrigin);
	if (curTtl != "0" && !curTtl.isEmpty())
		output += "$TTL " + curTtl + "\n";
	output += qxt_d().data->domainName + " IN SOA " + qxt_d().data->primaryNameServer + " " + qxt_d().data->adminNameServer + " (\n";
	for (int i = 0; i < 5; ++i)
	{
		output += QString("             %1\n").arg(qxt_d().data->soaValues.at(i));
	}
	output += ")\n";
	foreach(RdsDnsRecord record, qxt_d().data->records)
	{
		QString recordOrigin = (record.origin().isEmpty() ? qxt_d().data->domainName : record.origin());
		if ((record.key() == "IN" || record.key().isEmpty()) && recordOrigin == curOrigin)
		{
			output += QString().leftJustified(maxKeyLength) + "    IN    " + typeToString(record.type()).leftJustified(10) + "    " + record.value() + "\n";
			continue;
		}
		sortedRecordHash
		[recordOrigin]
		[record.ttl()]
		.append(record);

		extraOrigins.removeAll(recordOrigin);
//		qDebug() << "Origin" << (record.origin.isEmpty() ? qxt_d().data->domainName : record.origin) << "because" << record.origin.isEmpty() << qxt_d().data->domainName << record.origin;
	}
	for (QMap<QString, QMap<QString, RdsDnsRecordList> >::const_iterator i = sortedRecordHash.constBegin(); i != sortedRecordHash.end(); ++i)
	{
		if (curOrigin != i.key())
		{
			//qDebug() << curOrigin << "and" << i.key() << "are different";
			///TODO: Quick hack to make the 1.1 release work, we need to find the actual cause of this
			QString tmp = i.key();
			tmp = tmp.replace("..",".");
			output += "$ORIGIN " + tmp + "\n";
			curOrigin = i.key();
		}
		for (QMap<QString, RdsDnsRecordList>::const_iterator j = i.value().constBegin(); j != i.value().end(); ++j)
		{
			if (curTtl != j.key() && !j.key().isEmpty())
			{
				output += "$TTL " + j.key() + "\n";
				curTtl = j.key();
			}
			foreach(RdsDnsRecord record, j.value())
			{
				foreach(QString line, record.comments())
				{
					output += ";" + line + "\n";
				}
				if (record.type() == SOA)
				{
					qCritical() << "SOA records should never be stored as a normal record!";
					continue;
				}
				if (record.key() == "IN")
					output += QString().leftJustified(maxKeyLength) + "    IN    " + typeToString(record.type()).leftJustified(10) + "    " + record.value() + "\n";
				else
					output += record.key().leftJustified(maxKeyLength) + "    IN    " + typeToString(record.type()).leftJustified(10) + "    " + record.value() + "\n";
			}
		}
	}

	output += "\n\n; These are the \"empty\" origins. In reality, it'll likely be all of the origins listed one by one...\n"
	          "; Just ignore this section, it's only here to make the RDS admin tool support a better tree structure for DNS\n";
	foreach(QString origin, extraOrigins)
	{
		output += "$ORIGIN " + origin + "\n";
	}
	QTemporaryFile file;
	file.open();
	file.write(output.toLocal8Bit());
	//qDebug() << "alsdkjfhasldkjfh" << output << qxt_d().data->domainName.left(qxt_d().data->domainName.count()) << file.fileName();
	file.flush();
// 	file.close();
	//qDebug() << "My domain name is" << qxt_d().data->domainName.left(qxt_d().data->domainName.count());
	QProcess proc;
	proc.setProcessChannelMode(QProcess::SeparateChannels);
	proc.start("named-checkzone", QStringList() << qxt_d().data->domainName.left(qxt_d().data->domainName.count()) << file.fileName());
	proc.waitForFinished();
	if (proc.exitCode() != 0)
	{
		QString errorString;
		QStringList errors = QString(proc.readAllStandardOutput()).split('\n', QString::SkipEmptyParts);
		foreach(QString error, errors)
		{
			// /tmp/qt_temp.XXXXXX:##: ERROR STRING
			error = error.mid(error.indexOf(": ") + 2);
			if (error.contains("ignor"))
				continue;
			if (error.contains("bad owner name"))
				continue;
			errorString.append(error);
			errorString.append('\n');
		}
		QProcess::execute("cp", QStringList() << file.fileName() << "/tmp/badzone");
		return ReturnValue(ReturnValue::GenericError, "Failed to validate DNS config file:\n" + errorString);
	}
	return "\n\n" + output + "\n\n";
}

ReturnValue RdsDnsZone::save()
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	QString zonePath = RDS_DNS_ZONEDIR + qxt_d().data->domainName + "db";
// 	QString shortdn = qxt_d().data->domainName.left(qxt_d().data->domainName.count() - 1);
	QString shortdn = qxt_d().data->domainName;
	RdsDnsManagerData* mdata = &RdsDnsManagerPrivate::staticData();
	QMutexLocker mlocker(mdata->getMutex());
	if (!mdata->zones.contains(shortdn))
	{
// 		qDebug() << "I'm not in the zones list!" << shortdn << mdata->zones.keys();
		mdata->zones[shortdn].values["type"] = "master";
		mdata->zones[shortdn].values["file"] = zonePath;
		mdata->zones[shortdn].zone = shortdn;
		mdata->zones[shortdn].zoneKey = "\"" + shortdn + "\"";
		mdata->values = insertData(mdata->values, QStringList() << "zone" << mdata->zones[shortdn].zoneKey << "type", "master");
		mdata->values = insertData(mdata->values, QStringList() << "zone" << mdata->zones[shortdn].zoneKey << "file", "\"" + zonePath + "\"");
	}
	mdata->zones[shortdn].values["file"] = zonePath;
	mdata->zones[shortdn].zone = shortdn;
	mdata->values = insertData(mdata->values, QStringList() << "zone" << mdata->zones[shortdn].zoneKey << "file", "\"" + zonePath + "\"");

	QVariantMap zoneMap = mdata->values.value("zone").toMap();
	QVariantMap disMap = zoneMap.value(mdata->zones[shortdn].zoneKey).toMap();
	disMap.remove("file");
	disMap["file"] = "\"" + zonePath + "\"";
	zoneMap[mdata->zones[shortdn].zoneKey] = disMap;
	mdata->values["zone"] = zoneMap;

	if (QFile::exists(RDS_DNS_ZONEDIR))
	{
		QFileInfo info(RDS_DNS_ZONEDIR);
		if (!info.isDir())
		{
			if (!QFile::remove(RDS_DNS_ZONEDIR))
				return ReturnValue(1, QString("Zones directory exists as a file and is not removable. Please remove ") + RDS_DNS_ZONEDIR + " and try again");
			QDir dir(RDS_DNS_CONFIGDIR);
// 			qDebug() << "zonepath (exists)!" << RDS_DNS_ZONEDIR;
			if (!dir.mkdir(RDS_DNS_ZONEDIR))
				return ReturnValue(1, QString("Failed to create the zones directory. Check the permissions of ") + RDS_DNS_CONFIGDIR + " and try again");
		}
	}
	else
	{
// 		QDir dir(RDS_DNS_CONFIGDIR);
// 		qDebug() << "zonepath!" << RDS_DNS_ZONEDIR;
		if (!QDir().mkdir(RDS_DNS_ZONEDIR))
			return ReturnValue(1, QString("Failed to create the zones directory. Check the permissions of ") + RDS_DNS_CONFIGDIR + " and try again");
	}

	ReturnValue ret = toString();
	if (ret.isError())
		return ret;

	QFile file(zonePath);
	if (!file.open(QFile::WriteOnly | QFile::Truncate))
	{
		return ReturnValue(1, "Failed to open the zone file for writing");
	}

	file.write(ret.toString().toLocal8Bit());
	return true;
}

ReturnValue RdsDnsZone::domainName() const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	QString dn = qxt_d().data->domainName;
	if (dn.endsWith("."))
		dn = dn.left(dn.count() - 1);
	return dn;
}

ReturnValue RdsDnsZone::setDomainName(const QString &domainName)
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	QString fullDomain = (domainName.endsWith(".") ? domainName : domainName + '.');
	qxt_d().data->soaOrigin = qxt_d().data->soaOrigin.replace(qxt_d().data->domainName, fullDomain);

	for (RdsDnsRecordList::iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		i->setOrigin(i->origin().replace(qxt_d().data->domainName, fullDomain));
		i->setValue(i->value().replace(qxt_d().data->domainName, fullDomain));
	}

	if (qxt_d().data->soaOrigin.isEmpty())
		qxt_d().data->soaOrigin = fullDomain;
	else
		qxt_d().data->soaOrigin = qxt_d().data->soaOrigin.replace(qxt_d().data->domainName, fullDomain);

	qxt_d().data->domainName = fullDomain;
	qxt_d().domainName = fullDomain;

	return true;
}

ReturnValue RdsDnsZone::primaryNameServer() const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	return qxt_d().data->primaryNameServer;
}

ReturnValue RdsDnsZone::setPrimaryNameServer(const QString &primaryNameServer)
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	qxt_d().data->primaryNameServer = primaryNameServer;
	removeRecord(NS, "IN");
	addRecord(NS, "IN", primaryNameServer);
	return true;
}

ReturnValue RdsDnsZone::adminNameServer() const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	return qxt_d().data->adminNameServer;
}

ReturnValue RdsDnsZone::setAdminNameServer(const QString &ns)
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	QString adminns = ns;
	adminns = adminns.replace('@', '.');
	qxt_d().data->adminNameServer = adminns;
	return true;
}

ReturnValue RdsDnsZone::soaValues() const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	QVariantList list;
	foreach(QString value, qxt_d().data->soaValues)
	{
		list << value;
	}
	return list;
}

ReturnValue RdsDnsZone::setSoaValues(const QStringList &values)
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	qxt_d().data->soaValues = values;
	return true;
}

ReturnValue RdsDnsZone::addRecord(int type, const QString &key, const QString &value)
{
	return addRecord(type, key, value, QString(), QString());
}

ReturnValue RdsDnsZone::addRecord(int type, const QString &key, const QStringList &value)
{
	return addRecord(type, key, value, QString(), QString());
}

ReturnValue RdsDnsZone::addRecord(int type, const QString &key, const QStringList &value, const QString &origin)
{
	return addRecord(type, key, value, origin, QString());
}

ReturnValue RdsDnsZone::addRecord(int type, const QString &key, const QString &value, const QString &origin)
{
	return addRecord(type, key, value, origin, QString());
}

ReturnValue RdsDnsZone::addRecord(int type, const QString &key, const QString &value, const QString &origin, const QString &ttl)
{
	return addRecord(type, key, QStringList() << value, origin, ttl);
}

ReturnValue RdsDnsZone::addRecord(int type, const QString &k, const QStringList &value, const QString &o, const QString &ttl)
{
	CHECK_VALIDITY;
	if (type == Invalid)
		return ReturnValue(1, "Cannot add an invalid record type");
	if (k.contains(" "))
		return ReturnValue(1, "Cannot have a key with a space in it");
	if (ttl.isEmpty() && !value.count())
		return ReturnValue(1, "Neither TTL nor Value was set, no action may be made");

	QMutexLocker locker(qxt_d().data->getMutex());
	if (qxt_d().data->soaValues.count() != 5)
		return ReturnValue(1, "The zone has an invalid SOA entry, this must be corrected before adding records");

	QString origin = o;
	if (origin.isEmpty())
		origin = qxt_d().domainName;
	if (!origin.endsWith('.'))
		origin = origin + '.';

	QString key = k;

// 	qDebug() << "Before normalization" << key << origin << "(" << o << ")" << qxt_d().domainName;
	RdsDnsRecordPrivate::normalizeRecord(key, origin);

	if (!origin.endsWith(qxt_d().domainName))
		origin = origin + qxt_d().domainName;
// 	qDebug() << "Adding this record" << origin << key << type << value;
	bool found = false;
	QString oldTtl = ttl;
	for (RdsDnsRecordList::iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); i)
	{
//		qDebug() << "ADD RECORD SEARCH FOR <type, key> <" << type << "," << key << "type:" << i->type << "key:" << i->key;
		if (i->type() == type && i->key() == key && i->origin() == origin)
		{
			if (value.count())
			{
				if (oldTtl.isEmpty())
					oldTtl = i->ttl();
				found = true;
				i = qxt_d().data->records.erase(i);
			}
			else
			{
				i->setTtl(ttl);
				return false;
			}
		}
		else
			++i;
	}
	if (!value.count())
		return ReturnValue(1, "Failed to find an existing record to set TTL on");

	RdsDnsRecord record;
	record.setType((RecordType)type);
	record.setKey(key);
	record.setOrigin(origin);
	record.setTtl(oldTtl.isEmpty() ? qxt_d().data->soaValues.at(4) : oldTtl);

	foreach(const QString& str, value)
	{
		record.setValue(str);
		qxt_d().data->records << record;
	}

	return !found;
}

ReturnValue RdsDnsZone::addRecord(const RdsDnsRecord &record)
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	return addRecord(record.type(), record.key(), record.value(), record.origin(), record.ttl());
}

ReturnValue RdsDnsZone::removeRecord(int type, const QString &key)
{
	CHECK_VALIDITY;
	return removeRecord(type, key, QString());
}

ReturnValue RdsDnsZone::removeRecord(int type, const QString &key, const QString& o)
{
	CHECK_VALIDITY;
	QString origin = (o.isEmpty() ? qxt_d().data->domainName : o);
// 	qDebug() << "Attempting to remove record" << type << key << origin;

	QMutexLocker locker(qxt_d().data->getMutex());
	for (int i = 0; i < qxt_d().data->records.count(); ++i)
	{
		if (qxt_d().data->records.at(i).type() == type && qxt_d().data->records.at(i).key() == key && qxt_d().data->records.at(i).origin() == origin)
		{
			qxt_d().data->records.removeAt(i);
			return true;
		}
	}
	return false;
}

ReturnValue RdsDnsZone::removeRecord(const RdsDnsRecord &record)
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	return removeRecord(record.type(), record.key(), record.origin());
}

ReturnValue RdsDnsZone::records(int type, const QString &key) const
{
	CHECK_VALIDITY;
	RdsDnsRecordList list;
	QMutexLocker locker(qxt_d().data->getMutex());
	for (RdsDnsRecordList::const_iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		if (i->type() == type && i->key() == key)
		{
			list << *i;
		}
	}
	if (list.count())
		return ReturnValue::fromValue(list);
	return ReturnValue(1, "Record does not exist.");
}

ReturnValue RdsDnsZone::record(int type, const QString &key) const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	for (RdsDnsRecordList::const_iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		if (i->type() == type && i->key() == key)
		{
			return ReturnValue::fromValue(*i);
		}
	}
	return ReturnValue(1, "Record does not exist.");
}

ReturnValue RdsDnsZone::records(int type, const QString &key, const QString &origin) const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	RdsDnsRecordList list;
	for (RdsDnsRecordList::const_iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		if (i->type() == type && i->key() == key && i->origin() == origin)
		{
			list << *i;
		}
	}
	if (list.count())
		return ReturnValue::fromValue(list);
	return ReturnValue(1, "Record does not exist.");
}

ReturnValue RdsDnsZone::record(int type, const QString &key, const QString &origin) const
{
	CHECK_VALIDITY;
// 	qDebug() << "Searching for record" << type << key << origin;
	QMutexLocker locker(qxt_d().data->getMutex());
	for (RdsDnsRecordList::const_iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		if (i->type() == type && i->key() == key && i->origin() == origin)
		{
			return ReturnValue::fromValue(*i);
		}
	}
	return ReturnValue(1, "Record does not exist");
}

ReturnValue RdsDnsZone::listRecords() const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	return ReturnValue::fromValue(qxt_d().data->records);
}

ReturnValue RdsDnsZone::listRecordsByOrigin(const QString &o) const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	RdsDnsRecordList list;
	for (RdsDnsRecordList::const_iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		QString origin = (i->origin().isEmpty() ? qxt_d().data->domainName + "." : i->origin());
		if (origin == o)
		{
			list << *i;
		}
	}
	return ReturnValue::fromValue(list);
}

ReturnValue RdsDnsZone::listRecordsByKey(const QString &key) const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	RdsDnsRecordList list;
	for (RdsDnsRecordList::const_iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		if (i->key() == key)
		{
			list << *i;
		}
	}
	return ReturnValue::fromValue(list);
}

ReturnValue RdsDnsZone::listRecordsByType(int type) const
{
	CHECK_VALIDITY;
	QMutexLocker locker(qxt_d().data->getMutex());
	RdsDnsRecordList list;
	for (RdsDnsRecordList::const_iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		if (i->type() == type)
		{
			list << *i;
		}
	}
	return ReturnValue::fromValue(list);
}

ReturnValue RdsDnsZone::listOrigins(const QString &r) const
{
	CHECK_VALIDITY;
	QStringList ret;
	QString root = r;

	if (!root.endsWith(qxt_d().data->domainName))
	{
		if (root.endsWith('.'))
			root = root.append(qxt_d().data->domainName);
		else
			root = root.append('.' + qxt_d().data->domainName);
	}
	while (root.startsWith('.'))
		root = root.mid(1);

	foreach(RdsDnsRecord record, qxt_d().data->records)
	{
		QString origin = (record.origin().isEmpty() ? qxt_d().data->domainName : record.origin());

		if (!origin.endsWith(root))
		{
			continue;
		}
		origin = origin.left(origin.count() - root.count() - 1);

		if (origin.isEmpty())
		{
			continue;
		}

		origin = origin.mid(origin.lastIndexOf(".") + 1);
		if (origin.isEmpty())
		{
			continue;
		}

		if (!ret.contains(origin))
			ret << origin;
	}
	foreach(QString origin, qxt_d().data->extraOrigins)
	{
		if (!origin.endsWith(root))
		{
			continue;
		}
		origin = origin.left(origin.count() - root.count() - 1);

		if (origin.isEmpty())
		{
			continue;
		}

		origin = origin.mid(origin.lastIndexOf(".") + 1);
		if (origin.isEmpty())
		{
			continue;
		}

		if (!ret.contains(origin))
			ret << origin;
	}
	return ret;
}

ReturnValue RdsDnsZone::listOrigins() const
{
	return listOrigins(QString::null);
}

ReturnValue RdsDnsZone::addOrigin(const QString &o)
{
	CHECK_VALIDITY;
	QString origin = (o.endsWith('.') ? o : o + '.');
	if (!origin.endsWith(qxt_d().domainName))
		origin.append(qxt_d().domainName);

// 	qDebug() << "Adding origin" << o << "(" << origin << ")";
	if (!qxt_d().data->extraOrigins.contains(origin))
		qxt_d().data->extraOrigins << origin;

	return true;
}

ReturnValue RdsDnsZone::removeOrigin(const QString &origin)
{
	CHECK_VALIDITY;
// 	qDebug() << "Removing origin" << origin;
	ReturnValue ret = listOrigins(origin);
	if (ret.isError())
		return ret;
	QStringList originList = ret.toStringList();
	foreach(QString suborigin, originList)
	{
		ret = removeOrigin(suborigin + "." + origin);
		if (ret.isError())
			return ret;
	}
	ret = listRecordsByOrigin(origin);
	if (ret.isError())
		return ret;
	RdsDnsRecordList list = ret.value<RdsDnsRecordList>();
	foreach(RdsDnsRecord record, list)
	{
		ret = removeRecord(record);
		if (ret.isError())
			return ret;
	}
	qxt_d().data->extraOrigins.removeAll(origin);
	return true;
}

ReturnValue RdsDnsZone::renameOrigin(const QString &oldname, const QString &newname)
{
	ReturnValue ret;

	for (RdsDnsRecordList::iterator i = qxt_d().data->records.begin(); i != qxt_d().data->records.end(); ++i)
	{
		QString origin = i->origin();
		if (origin == oldname || origin.endsWith('.' + oldname))
		{
			origin.replace(oldname, newname);
			i->setOrigin(origin);
		}
		origin = i->value();
		if (origin.endsWith('.' + oldname))
		{
			origin.replace(oldname, newname);
			i->setValue(origin);
		}
	}

	for (QStringList::iterator i = qxt_d().data->extraOrigins.begin(); i != qxt_d().data->extraOrigins.end(); ++i)
	{
		if (*i == oldname || i->endsWith('.' + oldname))
		{
			i->replace(oldname, newname);
		}
	}

	return true;
}

void RdsDnsZone::removed()
{
	qCritical() << "I don't exist anymore!";
	qxt_d().data->errorString = "Zone no longer exists";
	qxt_d().data->valid = false;
}

