/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsdhcpvalues.h"
#include "rdsdhcpvalues_p.h"

#include "rdsdhcpgroup.h"
#include "rdsdhcphost.h"
#include "rdsdhcpsubnet.h"
#include "rdsdhcpsharednetwork.h"

#include <QStringList>

#define RDS_LOCK \
	QMutexLocker ___locker(&RdsDhcpManagerPrivate::mutex()); \
	if(!qxt_d().values) \
		return ReturnValue(1, "Internal data object is missing in RdsDhcpValues");

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsDhcpValues);

RdsDhcpValues::RdsDhcpValues()
		: QtRpc::ServiceProxy()
{
}

RdsDhcpValues::RdsDhcpValues(const RdsDhcpValues &other)
		: QtRpc::ServiceProxy()
{
	setData(other.qxt_d().values);
}

RdsDhcpValues::~RdsDhcpValues()
{
	if (qxt_d().values)
		qxt_d().values->refs.removeAll(this);
}

ReturnValue RdsDhcpValues::values() const
{
	RDS_LOCK;
	return qxt_d().values->values;
}

ReturnValue RdsDhcpValues::setValues(const QVariantMap &m)
{
	RDS_LOCK;
	QVariantMap map = m; //because we need to edit it
	for (QVariantMap::iterator i = map.begin(); i != map.end(); ++i)
	{
		if (i.key() != "option")
			continue;

		QString value = i.value().toString();
		QString optionKey = value.left(value.indexOf(" "));
		QString optionValue = value.mid(value.indexOf(" ") + 1);
		setOption(optionKey, optionValue);
		i = map.erase(i)--;
	}
	qxt_d().values->values = map;
	return true;
}

ReturnValue RdsDhcpValues::valueExists(const QString& key)
{
	return(qxt_d().values->values.contains(key));
}


ReturnValue RdsDhcpValues::removeValue(const QString &key)
{
	RDS_LOCK;
	if (key == "option")
		return ReturnValue(1, "You cannot remove options via removeValue, please use removeOption.");
	qxt_d().values->values.remove(key);
	return true;
}

ReturnValue RdsDhcpValues::value(const QString &key) const
{
	RDS_LOCK;
	return qxt_d().values->values.value(key);
}

ReturnValue RdsDhcpValues::setValue(const QString &key, const QString &value)
{
	RDS_LOCK;
	if (key == "option")
	{
		if(!value.contains(" "))
			return ReturnValue(1, "Options must have both a key and a value to be valid.");
		QString optionKey = value.left(value.indexOf(" "));
		QString optionValue = value.mid(value.indexOf(" ") + 1);
		return setOption(optionKey, optionValue);
	}
	qxt_d().values->values.insert(key, value);
	return true;
}

ReturnValue RdsDhcpValues::options() const
{
	RDS_LOCK;
	return qxt_d().values->options;
}

ReturnValue RdsDhcpValues::setOptions(const QVariantMap &map)
{
	RDS_LOCK;
	qxt_d().values->options = map;
	return true;
}

ReturnValue RdsDhcpValues::option(const QString &key) const
{
	RDS_LOCK;
	return qxt_d().values->options.value(key);
}

ReturnValue RdsDhcpValues::setOption(const QString &key, const QString &value)
{
	RDS_LOCK;
	qxt_d().values->options.insert(key, value);
	return true;
}

ReturnValue RdsDhcpValues::optionExists(const QString& key)
{
	return(qxt_d().values->options.contains(key));
}

ReturnValue RdsDhcpValues::removeOption(const QString &key)
{
	RDS_LOCK;
	qxt_d().values->options.remove(key);
	return true;
}

ReturnValue RdsDhcpValues::groups() const
{
	RDS_LOCK;
	return QStringList(qxt_d().values->groups.keys());
}

ReturnValue RdsDhcpValues::subnets() const
{
	RDS_LOCK;
	return QStringList(qxt_d().values->subnets.keys());
}

ReturnValue RdsDhcpValues::sharedNetworks() const
{
	RDS_LOCK;
	return QStringList(qxt_d().values->sharedNetworks.keys());
}

ReturnValue RdsDhcpValues::hosts() const
{
	RDS_LOCK;
	return QStringList(qxt_d().values->hosts.keys());
}

ReturnValue RdsDhcpValues::sharedNetwork(const QString &sharedNetwork) const
{
	RDS_LOCK;
	if (!qxt_d().values->sharedNetworks.contains(sharedNetwork))
		return ReturnValue(1, "Unable to find a shared network by that name.");
	RdsDhcpValues* nw = new RdsDhcpSharedNetwork();
	nw->setData(qxt_d().values->sharedNetworks.value(sharedNetwork));
	return nw;
}

ReturnValue RdsDhcpValues::host(const QString &hostname) const
{
	RDS_LOCK;
	if (!qxt_d().values->hosts.contains(hostname))
		return ReturnValue(1, "Unable to find a host by that name.");
	RdsDhcpValues* nw = new RdsDhcpHost();
	nw->setData(qxt_d().values->hosts.value(hostname));
	return nw;
}

ReturnValue RdsDhcpValues::subnet(const QString &subnet) const
{
	RDS_LOCK;
	if (!qxt_d().values->subnets.contains(subnet))
		return ReturnValue(1, "Unable to find a subnet by that name.");
	RdsDhcpValues* nw = new RdsDhcpSubnet();
	nw->setData(qxt_d().values->subnets.value(subnet));
	return nw;
}

ReturnValue RdsDhcpValues::group(const QString &group) const
{
	RDS_LOCK;
	if (!qxt_d().values->groups.contains(group))
		return ReturnValue(1, "Unable to find a group by that name.");
	RdsDhcpValues* nw = new RdsDhcpGroup();
	nw->setData(qxt_d().values->groups.value(group));
	return nw;
}

ReturnValue RdsDhcpValues::addSharedNetwork(const QString &sharedNetwork)
{
	RDS_LOCK;
	if (qxt_d().values->sharedNetworks.contains(sharedNetwork))
		return ReturnValue(1,sharedNetwork + " already exists");
	RdsDhcpManagerPrivate::SharedNetwork* data = new RdsDhcpManagerPrivate::SharedNetwork();
	data->name = sharedNetwork;
	data->parent = qxt_d().values;
	qxt_d().values->sharedNetworks.insert(sharedNetwork, data);
	return this->sharedNetwork(sharedNetwork);
}

ReturnValue RdsDhcpValues::addHost(const QString &host)
{
	RDS_LOCK;
	if (qxt_d().values->hosts.contains(host))
		return ReturnValue(1,host + " already exists");
	RdsDhcpManagerPrivate::Host* data = new RdsDhcpManagerPrivate::Host();
	data->name = host;
	data->parent = qxt_d().values;
	qxt_d().values->hosts.insert(host, data);
	return this->host(host);
}

ReturnValue RdsDhcpValues::addGroup(const QString &group)
{
	RDS_LOCK;
	if (qxt_d().values->groups.contains(group))
		return ReturnValue(1,group + " already exists");;
	RdsDhcpManagerPrivate::Group* data = new RdsDhcpManagerPrivate::Group();
	data->name = group;
	data->parent = qxt_d().values;
	qxt_d().values->groups.insert(group, data);
	return this->group(group);
}

ReturnValue RdsDhcpValues::addSubnet(const QString &subnet)
{
	RDS_LOCK;
	if (qxt_d().values->subnets.contains(subnet))
		return ReturnValue(1,subnet + " already exists.");
	RdsDhcpManagerPrivate::Subnet* data = new RdsDhcpManagerPrivate::Subnet();
	data->name = subnet;
	data->parent = qxt_d().values;
	data->address = QHostAddress("0.0.0.0");
	data->netmask = QHostAddress("0.0.0.0");
	qxt_d().values->subnets.insert(subnet, data);
	return this->subnet(subnet);
}

ReturnValue RdsDhcpValues::removeSharedNetwork(const QString &sharedNetwork)
{
	RDS_LOCK;
	if (!qxt_d().values->sharedNetworks.contains(sharedNetwork))
		return true;
	RdsDhcpManagerPrivate::SharedNetwork* data = qxt_d().values->sharedNetworks.take(sharedNetwork);
	delete data;
	return true;
}

ReturnValue RdsDhcpValues::removeHost(const QString &host)
{
	RDS_LOCK;
	if (!qxt_d().values->hosts.contains(host))
		return true;
	RdsDhcpManagerPrivate::Host* data = qxt_d().values->hosts.take(host);
	delete data;
	return true;
}

ReturnValue RdsDhcpValues::removeGroup(const QString &group)
{
	RDS_LOCK;
	if (!qxt_d().values->groups.contains(group))
		return true;
	RdsDhcpManagerPrivate::Group* data = qxt_d().values->groups.take(group);
	delete data;
	return true;
}

ReturnValue RdsDhcpValues::removeSubnet(const QString &subnet)
{
	RDS_LOCK;
	if (!qxt_d().values->subnets.contains(subnet))
		return true;
	RdsDhcpManagerPrivate::Subnet* data = qxt_d().values->subnets.take(subnet);
	delete data;
	return true;
}

RdsDhcpValues& RdsDhcpValues::operator=(const RdsDhcpValues & other)
{
	setData(other.qxt_d().values);
	return *this;
}

void RdsDhcpValues::setData(void* data)
{
	QMutexLocker ___locker(&RdsDhcpManagerPrivate::mutex());
	RdsDhcpManagerPrivate::DHCPValues* v = static_cast<RdsDhcpManagerPrivate::DHCPValues*>(data);
	if (qxt_d().values)
		qxt_d().values->refs.removeAll(this);
	qxt_d().values = v;
	if (qxt_d().values)
		qxt_d().values->refs << this;
}



