/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef RDSLDAPOBJECT_H
#define RDSLDAPOBJECT_H

#include <RdsLdapSession>
#include <QxtPimpl>
#include <RdsLdapActions>
#include <rdsglobal.h>

class RdsLdapObjectPrivate;

/**
	@author Brendan Powers <brendan@resara.com>
*/
class RDS_CLIENT_EXPORT RdsLdapObject
{
	QXT_DECLARE_PRIVATE(RdsLdapObject);
public:
	RdsLdapObject();
	RdsLdapObject(const QString &dn);
	RdsLdapObject(const QString &dn, RdsLdapSession *session);
	RdsLdapObject(const RdsLdapObject &object);
	~RdsLdapObject();
	RdsLdapObject& operator=(const RdsLdapObject &other);

	/**
	 *        Searches for objects that match the filter
	 * @param filter search filter (see RFC 2254)
	 * @param recursive (optional) true will look in "sub-directories", false will not.
	 * @return LdapResults containing the information about the objects that matched the filter
	 */
	virtual ReturnValue search(const QString &filter, bool recursive = true) const;
	/**
	 * 	  Reads the values of the attributes in attrs for the object
	 * @param attrs (optional) list of attributes to get the values of
	 * @return LdapResult (which is QHash<QString, LdapValues>.  LdapValues is QList<QByteArray>.)
	 */
	virtual ReturnValue read(const QStringList &attrs = QStringList()) const;
	/**
	 * 	  Reads the values for the attribute attr.
	 * @param attr the atribute to return the values for.
	 * @return LdapValues (QList<QByteArray>) a list of all the attributes values.
	 */
	virtual ReturnValue readAttribute(const QString &attr) const;
	/**
	 * 	  Lists DN's that match the search filter
	 * @param filter (optional) search filter (see RFC 2254)
	 * @param recursive (optional) true will look in "sub-directories", false will not.
	 * @return StringList of the DN's of the objects that matched the search filter
	 */
	virtual ReturnValue list(const QString &filter = "(objectClass=*)", bool recursive = true) const;
	/**
	 *        Performs the actions on a new object. If the object already exists use modify()
	 * @param actions the actions to be done to construct the object
	 * @return true on success, check error message on failure
	 */
	virtual ReturnValue add(const RdsLdapActions &actions);
	/**
	 *        Performs actions on an already existing object
	 * @param actions the actions to perform
	 * @return true on success, check error message on failure
	 */
	virtual ReturnValue modify(const RdsLdapActions &actions);
	/**
	 *        Removes the object
	 * @return true on success, check error message on failure
	 */
	virtual ReturnValue remove(bool recursive=false);
	/**
	 * 	  Renames the object
	 * @param newname the new name for the object (ex "ou=newname")
	 * @return the new DN of the object
	 */
	ReturnValue rename(const QString &newname);
	/**
	 *        Moves the object to the location of newparent
	 * @param newparent the DN of the new location
	 * @return the new DN of the object
	 */
	ReturnValue move(const QString &newparent);
	/**
	 *        Gets the DN of the object
	 * @return DN string
	 */
	QString dn() const;
	/**
	 *        Sets the DN of the object
	 * @param dn the DN to give the object
	 */
	void setDn(const QString &dn);
	/**
	 *        Gets a pointer to the RdsLdapSession
	 * @return RdsLdapSession pointer
	 */
	RdsLdapSession *session() const;
};

QDataStream& operator<<(QDataStream& d, const RdsLdapObject& object);
QDataStream& operator>>(QDataStream& d, RdsLdapObject& object);
Q_DECLARE_METATYPE(RdsLdapObject);

#endif
