/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "rdsprovisionator.h"
#include "rdsprovisionator_p.h"

#include <QDebug>
#include <QStringList>
#include <QDateTime>
#include <QFile>
#include <QRegExp>
#include <QTextStream>
#include <RdsNetworkSettings>
#include <RdsUtils>
#include <RdsSettings>
#include "rdsnetmonitorthread.h"
#include "logmanager.h"
#include "rdsprovisionthread.h"
#include "rdsnetworksettingsthread.h"
#include "rdscheckin.h"

static RdsNetMonitorThread *_netthread = NULL;

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsProvisionator);

RdsProvisionator::RdsProvisionator(QObject *parent)
		: QtRpc::ServiceProxy(parent)
{
}

RdsProvisionator::RdsProvisionator(const RdsProvisionator &other)
		: QtRpc::ServiceProxy()
{
	Q_UNUSED(other);
}

RdsProvisionator::~RdsProvisionator()
{
}

RdsProvisionator& RdsProvisionator::operator=(const RdsProvisionator & other)
{
	Q_UNUSED(other);
	return *this;
}

ReturnValue RdsProvisionator::networkWorked()
{
	if (_netthread != NULL)
	{
		_netthread->cancel();
		_netthread = NULL;
	}

	return(true);
}

ReturnValue RdsProvisionator::provision(const QVariantMap &map)
{
	if (_netthread != NULL)
	{
		_netthread->cancel();
		_netthread = NULL;
	}

#define CHECK_RDS_SAMBA_PATH(value) \
	if (!QFile::exists(map[#value].toString())) \
	{ \
		qCritical() << "Attempt to provision with " #value " set to:" << map[#value].toString() << "which does not exist"; \
		return ReturnValue(1, "Path, " + map[#value].toString() + " was not found."); \
	}
	CHECK_RDS_SAMBA_PATH(RDS_SAMBA_PATH);
	CHECK_RDS_SAMBA_PATH(SAMBA_NET_PATH);
	CHECK_RDS_SAMBA_PATH(RDS_SAMBA_PRIVATE);
	CHECK_RDS_SAMBA_PATH(RDS_SAMBA_PROVISION);

	RdsProvisionThread *thread = new RdsProvisionThread(this);
	ReturnValue ret = thread->start(this, map);

	return(ret);
}

ReturnValue RdsProvisionator::applyNetworkSettings(const QVariantMap &map)
{
	if (_netthread != NULL)
	{
		_netthread->cancel();
		_netthread = NULL;
	}

	if (!map.contains("ip") ||
	        !map.contains("netmask") ||
	        !map.contains("gateway") ||
	        !map.contains("interface") ||
	        !map.contains("dns"))
		return ReturnValue(1, "Incomplete values passed to applyNetworkSettings");

	RdsNetworkSettingsThread* thread = new RdsNetworkSettingsThread(this);
	thread->start(map);
	///@todo: We need status signals and stoufs. Also, this return value should change.
	return "LET IT BEGIN";
}

ReturnValue RdsProvisionator::provisionExistingDomain(const QVariantMap &settings)
{
	RdsProvisionThread thread(settings, this);

#define CHECK_RDS_SAMBA_PATH2(value) \
	if (!QFile::exists(settings[#value].toString())) \
	{ \
		qCritical() << "Attempt to provision with " #value " set to:" << settings[#value].toString() << "which does not exist"; \
		return ReturnValue(1, "Path, " + settings[#value].toString() + " was not found."); \
	}
	CHECK_RDS_SAMBA_PATH2(RDS_SAMBA_PATH);
	CHECK_RDS_SAMBA_PATH2(RDS_SAMBA_SYSVOL);
	CHECK_RDS_SAMBA_PATH2(RDS_SAMBA_CONFIGFILE);
	CHECK_RDS_SAMBA_PATH2(SAMBA_NET_PATH);
	CHECK_RDS_SAMBA_PATH2(RDS_SAMBA_PRIVATE);
	CHECK_RDS_SAMBA_PATH2(RDS_SAMBA_PROVISION);
	return thread.provisionRds();
}

ReturnValue RdsProvisionator::writeRawConf(const QVariantMap &map)
{
	rdssettings().lock();
	for (QVariantMap::const_iterator i = map.begin(); i != map.end(); ++i)
	{
		if (i.value().type() == QVariant::Map)
		{
			rdssettings()->beginGroup(i.key());
			QVariantMap submap = i.value().toMap();
			for (QVariantMap::const_iterator j = submap.begin(); j != submap.end(); ++j)
			{
				rdssettings()->setValue(j.key(), j.value());
			}
			rdssettings()->endGroup();
		}
		else
		{
			rdssettings()->setValue(i.key(), i.value());
		}
	}
	rdssettings().unlock();
	return true;
}

ReturnValue RdsProvisionator::getDefaultSettings() const
{
	QVariantMap settings;

	RdsNetworkSettings net;
	net.loadNetworkConfig();
	ReturnValue ret = net.getDefaultInterface();
	if (!ret.isError()) settings["interface"] = ret;
	QString interface = ret.toString();
	//qDebug() << "Default Interface:" << interface;
	ret = net.getInterfaceMac(interface);
	if (!ret.isError()) settings["mac"] = ret;
	//qDebug() << "MAC:" << ret.toString();
	ret = net.getInterfaceIp(interface);
	if (!ret.isError()) settings["ip"] = ret;
	//qDebug() << "IP:" << ret.toString();
	ret = net.getInterfaceNetmask(interface);
	if (!ret.isError()) settings["netmask"] = ret;
	//qDebug() << "Netmask:" << ret.toString();
	ret = net.getDefaultGateway();
	if (!ret.isError()) settings["gateway"] = ret;
	//qDebug() << "Gateway:" << ret.toString();
	ret = net.getDnsServers();
	if (!ret.isError()) settings["dns"] = ret;
	//qDebug() << "DNS:" << ret.toStringList();
	ret = net.getDomain();
	if (!ret.isError()) settings["realm"] = ret;
	//qDebug() << "Realm:" << ret.toString();
	QStringList parts = ret.toString().split(".");
	if (parts.size() > 0)
	{
		settings["domain"] = parts[0].toUpper();
		//qDebug() << "domain:" << parts[0].toUpper();
	}
	ret = RdsUtils::fileContents(RDS_TIMEZONE);
	if (!ret.isError()) settings["timezone"] = ret.toString().replace("\n", "");
	//qDebug() << "Time Zone:" << ret.toString();
	settings["datetime"] = QDateTime::currentDateTime();
	RdsNetworkSettings::Interface iface = net.getInterface(interface);
	//qDebug() << iface.name() << iface.type() << iface.autoUp();
	bool netconfigured = ((iface.name() == interface) && iface.autoUp() && (iface.type() == "static"));
	settings["netconfigured"] = netconfigured;
	//qDebug() << "Network Configured:" << netconfigured;

	char hostname[128];
	gethostname(hostname, 128);
	settings["hostname"] = hostname;

	settings["interfaces"] = net.listAllInterfaces().toStringList();

	QFile file("/usr/share/zoneinfo/zone.tab");
	if (file.open(QFile::ReadOnly))
	{
		QStringList zones;
		QTextStream stream(&file);
		QString line;

		while ((line = stream.readLine()) != QString::Null())
		{
			if (line.startsWith("#")) continue;
			QStringList parts = line.split(QRegExp("\\s"), QString::SkipEmptyParts);
			if (parts.size() < 3) continue;

			zones << parts[2];
		}

		settings["timezones"] = zones;
	}

	settings["RDS_SAMBA_PATH"] = RDS_SAMBA_PATH;
	settings["RDS_SAMBA_SYSVOL"] = RDS_SAMBA_SYSVOL;
	settings["RDS_SAMBA_CONFIGFILE"] = RDS_SAMBA_CONFIGFILE;
	settings["SAMBA_NET_PATH"] = SAMBA_NET_PATH;
	settings["RDS_SAMBA_PRIVATE"] = RDS_SAMBA_PRIVATE;
	settings["RDS_SAMBA_PROVISION"] = RDS_SAMBA_PROVISION;

	return(settings);
}

ReturnValue RdsProvisionator::setNetworkSettings(const QVariantMap &settings, bool updatehosts)
{
	qInfo() << "Configuring network settings.";

	if (!settings.contains("interface"))
	{
		qCritical() << "The interface was not filled in on the settings";
		return(false);
	}
	if (!settings.contains("ip"))
	{
		qCritical() << "The IP address was not filled in on the settings";
		return(false);
	}
	if (!settings.contains("netmask"))
	{
		qCritical() << "The netmask was not filled in on the settings";
		return(false);
	}
	if (!settings.contains("gateway"))
	{
		qCritical() << "The gateway was not filled in on the settings";
		return(false);
	}

	QString interface = settings.value("interface").toString();
	QString ip = settings.value("ip").toString();
	QString netmask = settings.value("netmask").toString();
	QString gateway = settings.value("gateway").toString();
	QStringList dns = settings.value("dns").toStringList();
	RdsNetworkSettings net;
	QString oldip = net.getInterfaceIp(interface).toString();
	QString oldnetmask = net.getInterfaceNetmask(interface).toString();
	QString oldgateway = net.getDefaultGateway().toString();
	QStringList olddns = net.getDnsServers().toStringList();

	_netthread = new RdsNetMonitorThread(interface,
	                                     ip,
	                                     netmask,
	                                     gateway,
					     dns,
	                                     oldip,
	                                     oldnetmask,
	                                     oldgateway,
					     olddns,
					     updatehosts,
	                                     NULL);
	_netthread->start();

	return(true);
}

ReturnValue RdsProvisionator::provisionDhcp(QVariantMap settings)
{
	settings["provisiondhcp"] = true;
	RdsProvisionThread thread;
	thread.setSettings(settings);
	return(thread.provisionDhcp());
}

ReturnValue RdsProvisionator::installFinished()
{
	RdsCheckin *checkin = new RdsCheckin();
	QObject::connect(checkin, SIGNAL(finished()), checkin, SLOT(deleteLater()));
	checkin->install();
	return(true);
}


