/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsdaemon.h"
#include "rdsdaemon_p.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <poll.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>

#include <QCoreApplication>
#include <QFileInfo>

RdsDaemon* RdsDaemonPrivate::instance = NULL;
int RdsDaemonPrivate::fd = 0;

RdsDaemon::RdsDaemon(int argc, char** argv, const char* name)
{
	// 0 = read
	// 1 = write
	bool quiet = false;
	if (name == NULL)
	{
		int offset = 0;
		for (int i = 0; argv[0][i] != '\0'; ++i)
		{
			if (argv[0][i] == '/')
				offset = i + 1;
		}
		name = argv[0] + offset;
	}
	QXT_INIT_PRIVATE(RdsDaemon);
	if (RdsDaemonPrivate::instance != NULL)
	{
		cerr << "Multiple RdsDaemon instances were created. This one will be useless.";
		return;
	}
	qxt_d().instance = this;
	qxt_d().filename = new char[(strlen(name) + 14)];
	int ret = sprintf(qxt_d().filename, "/var/run/%s.pid", name);
	if (ret < 1)
	{
		qxt_d().filename = NULL;
	}
	else
	{
		struct stat stat_p;
		if (stat(qxt_d().filename, &stat_p) == 0)
		{
			FILE* file = fopen(qxt_d().filename, "r");
			int pid = 0;
			if (file == NULL)
			{
				cerr << "There was an error trying to open the file thing (" << qxt_d().filename << "): " << strerror(errno) << endl;
				exit(1);
			}
			fscanf(file, "%d", &pid); /* read from file */
			fclose(file);
			//struct stat stat_p;
			char procFile[21]; //maximum length for /proc/4294967295/exe\0
			sprintf(procFile, "/proc/%d/exe", pid);
			char myProcFile[21]; //maximum length for /proc/4294967295/exe\0
			sprintf(myProcFile, "/proc/%d/exe", getpid());
			if (QFileInfo(procFile).symLinkTarget() == QFileInfo(myProcFile).symLinkTarget())
			{
				cerr << "There is already a daemon of this type running" << endl;
				exit(1);
			}
		}
	}
	for (int i = 1; i < argc; ++i)
	{
		if (strcmp("--foreground", argv[i]) == 0)
		{
			if (qxt_d().filename != NULL)
			{
				FILE* file = fopen(qxt_d().filename, "w");
				if (file == NULL)
				{
					cerr << "There was an error trying to open the file thing (" << qxt_d().filename << "): " << strerror(errno) << endl;
					exit(0);
				}
				fprintf(file, "%d", ret);
				fclose(file);
			}
			return;
		}
		else if (strcmp("--quiet", argv[i]) == 0)
		{
			quiet = true;
		}
	}
	int pipefd[2] = {0, 0};
	ret = pipe(pipefd);
	if (ret != 0)
	{
		cerr << "Failed to fork, the process will not daemonize: " << strerror(errno) << endl;
		return;
	}

	ret = fork();
	if (ret < 0) // error when forking
	{
		int err = errno;
		cerr << "There was an error attempting to fork: " << strerror(err) << endl;
		exit(err);
	}

	chdir("/"); // change working directory

	if (ret == 0)// child process
	{
		qxt_d().fd = pipefd[1];
		close(pipefd[0]);
		ret = setsid();
		if (ret == -1)
		{
			cerr << "Failed to call setsid, this process will not daemonize properly: " << strerror(errno) << endl;
		}
		ret = fork();
		if (ret < 0) // error when forking
		{
			cerr << "There was an error attempting to fork after setsid: " << strerror(errno) << endl;
		}
		if (ret == 0)// child process
		{
			umask(0); //reset umask
			signal(SIGTSTP, SIG_IGN); /* Various TTY signals */
			signal(SIGTTOU, SIG_IGN);
			signal(SIGTTIN, SIG_IGN);
			signal(SIGHUP, SIG_IGN); /* Ignore hangup signal */
			qxt_d().child = true;
		}
		else
		{
			if (qxt_d().filename != NULL)
			{
				FILE* file = fopen(qxt_d().filename, "w");
				if (file == NULL)
				{
					cerr << "There was an error trying to open the file thing (" << qxt_d().filename << "): " << strerror(errno) << endl;
					abort();
				}
				fprintf(file, "%d", ret);
				fclose(file);
			}
			abort();
		}

		if (quiet)
		{
			ret = open("/dev/null", O_WRONLY);
			if (ret < 1)
			{
				cerr << "Failed to open /dev/null for console redirection: " << strerror(errno) << endl;
			}
			else
			{
				if (dup2(ret, 1) == -1)
				{
					cerr << "Failed to duplicated the file descriptor of /dev/null over stdout: " << strerror(errno) << endl;
				}
				if (dup2(ret, 2) == -1)
				{
					cerr << "Failed to duplicated the file descriptor of /dev/null over stdout: " << strerror(errno) << endl;
				}
			}

			ret = open("/dev/null", O_RDONLY);
			if (ret < 1)
			{
				cerr << "Failed to open /dev/null for console redirection - input: " << strerror(errno) << endl;
			}
			else
			{
				if (dup2(ret, 0) == -1)
				{
					cerr << "Failed to duplicated the file descriptor of /dev/null over stdin: " << strerror(errno) << endl;
				}
			}
		}
	}
	else // parent process -- block reading the pipe until there's something there then kill yourself...
	{
		qxt_d().fd = pipefd[0];
		close(pipefd[1]);

		while (true)
		{
			pollfd pfd = {qxt_d().fd, (POLLIN | POLLPRI | POLLRDHUP | POLLERR | POLLHUP | POLLNVAL), 0};
			ret = poll(&pfd, 1, -1);
			if (ret == -1)
			{
				int err = errno;
				if (err == EINTR)
					continue;
				cout << "There was an error polling the pipe: " << strerror(err) << endl;
				exit(1);
			}
			switch (pfd.revents)
			{
				case POLLIN:
				case POLLPRI:
				{
					int buff = 0;
					int ret = read(qxt_d().fd, &buff, sizeof(int));
					if (ret != sizeof(int))
					{
						cerr << "Failed to read the return code from the child process" << endl;
						exit(1);
					}
					exit(buff);
					break;
				}
				case POLLRDHUP:
				case POLLERR:
				case POLLHUP:
				case POLLNVAL:
				default:
					cout << "Child process appears to have died" << endl;
					exit(1);
					break;
			}
		}
	}
}

RdsDaemon::~RdsDaemon()
{
	if (qxt_d().filename != NULL)
		remove(qxt_d().filename);
}

void RdsDaemon::daemonize()
{
	if (RdsDaemonPrivate::instance == NULL)
	{
		cerr << "Daemonize called when no RdsDaemon was created.";
		return;
	}
	int fd = RdsDaemonPrivate::instance->getFd();
	if (fd == 0)
		return;
	int data = 0;

	pollfd pfd = {fd, (POLLOUT | POLLRDHUP | POLLERR | POLLHUP | POLLNVAL), 0};
	int ret = poll(&pfd, 1, 100);
	if (ret < 1)
	{
		return;
	}
	switch (pfd.revents)
	{
		case POLLOUT:
			break;
		case POLLRDHUP:
		case POLLERR:
		case POLLHUP:
		case POLLNVAL:
		default:
			return;
	}

	ret = write(fd, &data, sizeof(int));
	if (ret != sizeof(int))
	{
		cerr << "Daemonizing failed (" << fd << "): " << strerror(errno) << endl;
	}
	else
	{
		ret = open("/dev/null", O_WRONLY);
		if (ret < 1)
		{
			cerr << "Failed to open /dev/null for console redirection: " << strerror(errno) << endl;
		}
		else
		{
			if (dup2(ret, 1) == -1)
			{
				cerr << "Failed to duplicated the file descriptor of /dev/null over stdout: " << strerror(errno) << endl;
			}
			if (dup2(ret, 2) == -1)
			{
				cerr << "Failed to duplicated the file descriptor of /dev/null over stdout: " << strerror(errno) << endl;
			}
		}

		ret = open("/dev/null", O_RDONLY);
		if (ret < 1)
		{
			cerr << "Failed to open /dev/null for console redirection - input: " << strerror(errno) << endl;
		}
		else
		{
			if (dup2(ret, 0) == -1)
			{
				cerr << "Failed to duplicated the file descriptor of /dev/null over stdin: " << strerror(errno) << endl;
			}
		}
	}
}

int RdsDaemon::getFd()
{
	return qxt_d().fd;
}

