/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/

#include <QCoreApplication>
#include <QHostInfo>
#include <Server>
#include <ServerProtocolListenerTcp>
#include <ServerProtocolListenerSocket>
#include <ServicePublisher>
#include <RdsSettings>
#include <RdsUtils>
#include <RdsLdapSession>
#include <RdsUser>
#include <RdsSambaConfigParser>
#include "rdsservice.h"
#include <RdsUserGroupComputerManager>
#include <RdsDnsManager>
#include <RdsShareManager>
#include <RdsDhcpManager>
#include <RdsDaemonManager>
#include <RdsFileManager>
#include <RdsMountPointManager>
#include <RdsVolumeManager>
#include <RdsVolume>
#include <RdsStorageDeviceManager>
#include <RdsScheduleManager>
#include <RdsJobManager>
#include <RdsNetworkSettings>
#include <RdsSid>
#include <QFile>
#include <QProcess>
#include <QThread>
#include <signal.h>
#include <pthread.h>
#include "rdsdaemon.h"
#include "logmanager.h"
#include "publishmonitor.h"
#include "config.h"
#include "main.h"
#include "rdssambaldapsession.h"
#include "rdsprovisionthread.h"
#include "rdscheckin.h"

using namespace QtRpc;

#define TCP_PORT 10123

Server *srv;
RdsSambaLdapSession session(LDAP_URI);
ServicePublisher *pub;
PublishMonitor *mon;
bool published = false;
RdsdThreadFixer* rdsdfixer(0);

int getUptime()
{
	QFile file("/proc/uptime");
	if (!file.open(QFile::ReadOnly))
		return -1;
	QString str = file.readAll();
	str = str.split(" ").first();
	return str.toDouble();
}

void republishServices()
{
	pub->unpublish();
	pub->setFriendlyName(RdsUtils::realm());
	pub->publish();
}

void registerServices()
{
	if (QThread::currentThread() != rdsdfixer->thread())
		QMetaObject::invokeMethod(rdsdfixer, "registerServices", Qt::BlockingQueuedConnection);
	else
		rdsdfixer->registerServices();
}

void RdsdThreadFixer::registerServices()
{
	int count = 0;
	ReturnValue ret;

	if(!RdsLdapSession::isGlobalSessionSet())
	{
		do
		{
			if (count != 0) sleep(1);

			ret = session.kerberosBind();
			if (ret.isError()) qWarning() << "Failed to connect to LDAP server:" << ret;
			count++;
		}
		while ((ret.isError()) && (count < 30));

		if (ret.isError())
		{
			qCritical() << "Failed to initialize LDAP session:" << ret;
			exit(1);
		}

		RdsLdapSession::setGlobalSession(&session);
	}

	//Look up the path to the domain administrator
	QString admindn;
	RdsSid sid = RdsSid::getDomainSid();
	sid.setRid(500);

	ReturnValue ret2 = RdsUtils::getObjectBySid(sid);
	if(ret.isError())
	{
		admindn = QString("CN=Administrator,CN=Users,%2").arg(rdssettings()->value("adminuser").toString()).arg(RdsUtils::baseDn());
	}
	else
	{
		admindn = ret2.toString();
	}

	//Make sure the domain administrator user's password doesn't go bad
        RdsUser user(admindn);
        ret2 = user.setFlag(RdsUser::PasswordNeverExpires);
        if (ret2.isError())
        {
                qWarning() << "Failed set the domain admin user flags:" << ret2;
        }

	//Make sure the RDS admin user's password doesn't go bad
	user = RdsUser(QString("CN=%1,CN=Users,%2").arg(rdssettings()->value("adminuser").toString()).arg(RdsUtils::baseDn()));
	ret2 = user.setFlag(RdsUser::PasswordNeverExpires);
	if (!ret2.isError())
	{
		ret2 = user.setFlag(RdsUser::CannotChangePassword);
		if (ret2.isError())
		{
			qWarning() << "Failed to set admin user flags:" << ret;
		}
	}
	else
	{
		qWarning() << "Failed to get admin user flags:" << ret;
	}

	srv->registerService<RdsUserGroupComputerManager>("UserGroupComputerManager");
	srv->registerService<RdsDnsManager>("DnsManager");
	srv->registerService<RdsDhcpManager>("DhcpManager");
	srv->registerService<RdsShareManager>("ShareManager");
	srv->registerService<RdsDaemonManager>("DaemonManager");
	srv->registerService<RdsFileManager>("FileManager");
	srv->registerService<RdsMountPointManager>("MountPointManager");
	srv->registerService<RdsVolumeManager>("VolumeManager");
	srv->registerService<RdsStorageDeviceManager>("StorageDeviceManager");
	srv->registerService<RdsScheduleManager>("ScheduleManager");
	srv->registerService<RdsJobManager>("JobManager");

	// Mount root if it's not already!
	ret = RdsVolumeManager().rootVolume();
	if (!ret.isError() && !ret.toString().isEmpty())
	{
		ret = RdsVolume(ret.toString()).mount();
		if (ret.isError())
			qWarning() << "Failed to mount root device:" << ret;
	}

	if (!published)
	{
		pub->unpublish();
		pub->setFriendlyName(RdsUtils::realm());
		pub->publish();
	}
	
	//Write out DNS config files if needed
	if (rdssettings()->value("updates/dns").toInt() < 1)
	{
		qDebug() << "Updating DNS config files.";
		
		RdsDnsManager dns;
		dns.save();
		
		ret = RdsDaemonManager().restartService("Dns");
		if (ret.isError())
		{
			qCritical() << "Failed to restart dns:" << ret;
		}
		
		rdssettings()->setValue("updates/dns", 1);
	}
	else
	{
		RdsDnsManager dns;
		dns.save();
		
		ret = RdsDaemonManager().reloadService("Dns");
		if (ret.isError())
		{
			qCritical() << "Failed to reload dns:" << ret;
		}
	}
	
	RdsScheduleManager().addEvent("checkin", RdsSchedule::weekly(), RdsCheckin::instance(), "checkin()");
}

//Fix configuration issues
void fixThings()
{
	ReturnValue ret;
	RdsSambaConfigParser smb;

	//Configure ntpd
	ret = RdsProvisionThread::provisionNtpd();
	if (ret.isError())
	{
		qWarning() << "Failed to provision ntpd:" << ret;
	}

	//Update server string
	ret = smb.keyExists("global", "server string");
	if (ret.isError())
	{
		qWarning() << "Failed to get server string:" << ret;
		return;
	}

	if (!ret.toBool())
	{
		qInfo() << "Updating server string...";

		ret = smb.setValue("global", "server string", QHostInfo::localHostName());
		if (ret.isError())
		{
			qWarning() << "Failed to set server string:" << ret;
			return;
		}

		ret = smb.save();
		if (ret.isError())
		{
			qWarning() << "Failed to save smb.conf file:" << ret;
			return;
		}

		ret = RdsDaemonManager().restartService("Samba");
		if (ret.isError())
		{
			qCritical() << "Failed to restart samba:" << ret;
			exit(1);
		}
	}

	//Set up interfaces for the samba config file
	//Update server string
	ret = smb.keyExists("global", "interfaces");
	if (ret.isError())
	{
		qWarning() << "Failed to get interfaces:" << ret;
		return;
	}

	if (!ret.toBool())
	{
		qDebug() << "Setting interfaces";
		
		RdsNetworkSettings net;
		ret = net.getDefaultInterface();
		if (!ret.isError())
		{
			ret = smb.setValue("global", "interfaces", ret.toString());
			if (ret.isError())
			{
				qWarning() << "Failed to set interfaces:" << ret;
				return;
			}

			ret = smb.save();
			if (ret.isError())
			{
				qWarning() << "Failed to save smb.conf file:" << ret;
				return;
			}
		}
	}
}

int main(int argc, char *argv[])
{
	RdsDaemon daemon(argc, argv, "rdsd");
	QCoreApplication app(argc, argv);
	LogManager log("rdsd");
	log.addFile(RDS_LOGFILE);
	rdsdfixer = new RdsdThreadFixer();

	sigset_t mask;
	sigemptyset(&mask);
	sigaddset(&mask, SIGPIPE);
	pthread_sigmask(SIG_BLOCK, &mask, NULL);

	fixThings();

	ReturnValue ret;

	int port = TCP_PORT;
	int tmp = QString(getenv("RDS_PORT")).toInt();
	if (tmp != 0) port = tmp;

	QString servicename = getenv("RDS_SERVICE");
	if (servicename == "") servicename = "Rds";

	srv = new Server();
	RdsService *service = (RdsService *)srv->registerService<RdsService>(servicename);

	ServerProtocolListenerTcp tcp(srv);
	if (!tcp.listen(QHostAddress::Any, port))
	{
		qFatal("Failed to listen on %d", port);
		exit(1);
	}

	ServerProtocolListenerSocket socket(srv);
	ret = socket.listen("/var/run/rdsd.socket");
	if (ret.isError())
	{
		qFatal("Failed to listen on %s: %s", "/var/run/rdsd.socket", qPrintable(ret.errString()));
		exit(1);
	}

	int uptime = getUptime();
	int lastUptime = rdssettings()->value("uptime", -1).toInt();
	if (lastUptime == -1 || lastUptime > uptime)
	{
		qInfo() << "Restarting Avahi daemon before publishing my service";
		QProcess::execute("service", QStringList() << "avahi-daemon" << "restart");
		uptime = getUptime(); //restarting could take a few secconds
		if (uptime > 0)
			rdssettings()->setValue("uptime", uptime);
	}

	rdssettings()->remove("adminpw");

	pub = new ServicePublisher(service);
	mon = new PublishMonitor(pub);
	pub->setPort(port);
	if (rdssettings()->value("provisioned").toBool())
	{
		pub->setFriendlyName(RdsUtils::realm());
		pub->publish();
		published = true;
		registerServices();
		RdsCheckin::instance()->checkUpgrade();
	}
	else
	{
		pub->setFriendlyName("Unconfigured");
		pub->publish();
	}

	daemon.daemonize();
	return(app.exec());
}
