/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "sharemanagerwidget.h"
#include <QMessageBox>
#include <RdsClient>
#include <QDebug>
#include <QInputDialog>
#include <QMenu>
#include <QSettings>
#include <QShortcut>
#include <RdsFileManager>
#include <RdsUtils>
#include <RdsDaemonManager>
#include "shareeditwidget.h"
#include "foldereditwidget.h"
#include "addsharedialog.h"
#include "config.h"

ShareManagerWidget::ShareManagerWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::ShareManagerWidget()
{
	setupUi(this);
	pushButton_4->setIcon(QPixmap(findRdsIcon("icons/16x16/clear_left.png")));
	AddButton->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	DeleteButton->setIcon(QPixmap(findRdsIcon("icons/32x32/remove.png")));
	ReloadButton->setIcon(QPixmap(findRdsIcon("icons/32x32/reload.png")));

	ReturnValue ret = rdsClient()->getService("ShareManager");
	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get user manager", ret.errString());
		return;
	}

	_manager = new RdsShareManager(this);
	*_manager = ret;

	_model = new RdsShareModel(_manager, this);
	_sort = new RdsEntitySortModel(_model);
	_sort->setSourceModel(_model);
	View->setModel(_sort);
	View->sortByColumn(0, Qt::AscendingOrder);
	_selection = View->selectionModel();
	_model->setupSelection(_sort, View);
	View->header()->resizeSection(0,200);

	
	QObject::connect(_model, SIGNAL(inputChanged(QString, QString)), this, SLOT(inputChanged(QString, QString)));
	QObject::connect(_model, SIGNAL(outputsChanged(QStringList)), this, SLOT(outputsChanged(QStringList)));
	QObject::connect(FilterEntry, SIGNAL(textChanged(QString)), _sort, SLOT(setFilterRegExp(QString)));

	ShareEditWidget *w = new ShareEditWidget(_manager, EditFrame);
	EditLayout->addWidget(w);
	_edit = w;
	w->setInput("");

	//Actions and Menu stuff
	View->setContextMenuPolicy(Qt::CustomContextMenu);
	QObject::connect(View, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(contextMenu(const QPoint&)));

	_newshareaction = new QAction("New Share", this);
	QObject::connect(_newshareaction, SIGNAL(triggered()), this, SLOT(addShare()));
	_newfolderaction = new QAction("New Folder", this);
	QObject::connect(_newfolderaction, SIGNAL(triggered()), this, SLOT(addFolder()));
	_removeaction = new QAction("Delete", this);
	QObject::connect(_removeaction, SIGNAL(triggered()), this, SLOT(on_DeleteButton_clicked()));
	_showsystemaction = new QAction("Show System Shares", this);
	QObject::connect(_showsystemaction, SIGNAL(toggled(bool)), this, SLOT(showSystemObjects(bool)));
	_showsystemaction->setCheckable(true);

	QSettings settings;
	if (settings.value("Shares/ShowSystemsObjects").toBool())
	{
		_showsystemaction->setChecked(true);
		showSystemObjects(true);
	}
	else
	{
		_showsystemaction->setChecked(false);
		showSystemObjects(false);
	}

	QMenu *addmenu = new QMenu(this);
	addmenu->addAction(_newshareaction);
	addmenu->addAction(_newfolderaction);
	AddButton->setMenu(addmenu);
	
	_removeaction->setShortcut(QKeySequence("Del"));
	new QShortcut(QKeySequence("Del"), this, SLOT(on_DeleteButton_clicked()));
}

ShareManagerWidget::~ShareManagerWidget()
{
}

void ShareManagerWidget::inputChanged(QString id, QString type)
{
	_input = id;
	//qDebug() << "Input Changed:" << id << type;
	if (_edit != NULL)
	{
		if (_edit->unsavedChanges() && _edit->isEnabled())
		{
			if (QMessageBox::question(this, "Unsaved Changes", "The user you were editing has unsaved changes. Would you like to apply them?", QMessageBox::Apply | QMessageBox::Discard, QMessageBox::Discard) == QMessageBox::Apply)
			{
				_edit->apply();
			}
		}

		delete _edit;
		_edit = NULL;
	}

	if (type == "share")
	{
		ShareEditWidget *w = new ShareEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else if (type == "folder")
	{
		FolderEditWidget *w = new FolderEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);
	}
	else
	{
		ShareEditWidget *w = new ShareEditWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput("");
	}
}

void ShareManagerWidget::outputsChanged(QStringList outputs)
{
	//qDebug() << "Output Changed" << outputs;
	if (_edit != NULL) _edit->setOutputs(outputs);
}

void ShareManagerWidget::addShare()
{
	AddShareDialog dialog;
	dialog.exec();
}

void ShareManagerWidget::addFolder()
{
	if (_model->input() == "")
	{
		QMessageBox::warning(this, "Error", "You must select a place to create the folder in.");
		return;
	}

	bool ok = false;
	QString name = QInputDialog::getText(this, "New Folder", "Enter the name for the new folder", QLineEdit::Normal, "New Folder", &ok);

	if (!ok)
	{
		return;
	}

	if (name == "")
	{
		QMessageBox::warning(this, "Error", "You must give the new folder a name.");
		return;
	}

	if (name.contains("/"))
	{
		QMessageBox::critical(NULL, "Error", "The name must not contain the / character.");
		return;
	}

	RdsFileManager filemgr;
	ReturnValue ret = filemgr.init();
	if (ret.isError())
	{
		QMessageBox::warning(this, "Error", "Failed to initialize file manager: " + ret.errString());
		return;
	}

	ret = filemgr.createDirectory(this, SLOT(newfolder(uint, ReturnValue)), RdsUtils::normalizePath(_model->input() + "/" + name));
	if (ret.isError())
	{
		QMessageBox::warning(this, "Error", "Failed to create folder: " + ret.errString());
		return;
	}
}

void ShareManagerWidget::on_DeleteButton_clicked()
{
	QStringList outputs = _model->outputs();

	if (outputs.size() == 0)
	{
		QMessageBox::information(this, "Error", "You must select something to delete.");
		return;
	}
	else if (outputs.size() == 1)
	{
		RdsEntityModel::Cache *cache = _model->getCache(outputs[0]);
		if (cache == NULL) return;

		int result;

		if (cache->type() == "share")
		{
			result = QMessageBox::question(this, "Remove Share", QString("Are you shure you want to remove the share %1?").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else if (cache->type() == "folder")
		{
			result = QMessageBox::question(this, "Delete Folder", QString("Do you want to delete the folder %1?").arg(cache->name()),
			                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		}
		else return;

		if (result != QMessageBox::Yes) return;
	}
	else
	{
		int result = QMessageBox::question(this, "Delete Items", QString("Do you want to delete %1 items?").arg(outputs.count()),
		                                   QMessageBox::Yes | QMessageBox::No, QMessageBox::No);

		if (result != QMessageBox::Yes)	return;
	}

	View->selectionModel()->clear();
	inputChanged("","");

	ReturnValue ret, err(true);
	bool removedshares = false;

	///TODO proper error checking
	RdsFileManager filemanager;
	ret = filemanager.init();
	if (ret.isError()) err = ret;

	foreach(QString output, outputs)
	{
		RdsEntityModel::Cache *cache = _model->getCache(output);
		if (cache == NULL) continue;

		if (cache->type() == "share")
		{
			if ((output == "@SHARE/sysvol/") || (output == "@SHARE/netlogon/"))
			{
				err = ReturnValue(1, QString(cache->name() + " is a system share."));
				continue;
			}
			QString share = RdsUtils::getShareName(output).toString();
			ret = _manager->removeShare(share);
			if (ret.isError())
			{
				err = ret;
				continue;
			}

			_manager->removeEntity(output);
			removedshares = true;
		}
		else if (cache->type() == "folder")
		{
			ret = filemanager.remove(output, true);
			//qDebug() << "Removing" << output << ret;
			if (ret.isError()) err = ret;
		}
		else
		{
			continue;
		}
	}

	if (removedshares)
	{
		ret = _manager->save();
		if (ret.isError()) err = ret;
		
		RdsDaemonManager mgr;
		ret = mgr.init();
		
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		}
		else
		{
			ret = mgr.reloadService("Samba");
			
			if (ret.isError())
			{
				QMessageBox::critical(this, "Error", "Failed to restart samba: " + ret.errString());
			}
		}
	}

	if (err.isError())
	{
		QMessageBox::warning(this, "Warning", "Some items could not be deleted: " + err.errString());
	}
}

void ShareManagerWidget::contextMenu(const QPoint & pos)
{
	Q_UNUSED(pos);
	QMenu menu;
	menu.addAction(_newshareaction);
	menu.addAction(_newfolderaction);
	menu.addAction(_removeaction);
	menu.addSeparator();
	menu.addAction(_showsystemaction);

	menu.exec(QCursor::pos());
}

void ShareManagerWidget::showSystemObjects(bool show)
{
	if (show)
		_sort->setHiddenEntities(QStringList());
	else
		_sort->setHiddenEntities(QStringList() << "@SHARE/sysvol/" << "@SHARE/netlogon/");

	QSettings settings;
	settings.setValue("Shares/ShowSystemsObjects", show);
}

void ShareManagerWidget::on_ReloadButton_clicked()
{
	_model->reload();
}

