/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "networkwidget.h"
#include <RdsClient>
#include <QDebug>
#include <QMessageBox>
#include <RdsLdapSession>
#include <RdsUtils>
#include <QHostAddress>
#include "networksavedialog.h"

#define IP_VALIDATOR new QRegExpValidator(QRegExp("^([0-9]{0,3}\\.){0,3}[0-9]{0,3}$"),this)
#define IP_VALIDATOR_tmp new QRegExpValidator(QRegExp("^(([0-9]?[0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9]?[0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9]?[0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9]?[0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$"),this)
#define IP_VALIDATOR_STRONG new QRegExpValidator(QRegExp("^(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$"),this)

NetworkWidget::NetworkWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::NetworkWidget()
{
	setupUi(this);
	
	IpEdit->setValidator(IP_VALIDATOR);
	NetmaskEdit->setValidator(IP_VALIDATOR);
	GatewayEdit->setValidator(IP_VALIDATOR);
	Dns1Edit->setValidator(IP_VALIDATOR);
	Dns2Edit->setValidator(IP_VALIDATOR);
	
	_provisionator = new RdsProvisionator(NULL);
	QObject::connect(this, SIGNAL(destroyed()), _provisionator, SLOT(deleteLater()));
	
	ReturnValue ret = rdsClient()->provisionator();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get provisioning service.");
		setEnabled(false);
		return;
	}
	
	*_provisionator = ret;
	
	load();
}

NetworkWidget::~NetworkWidget()
{
}

void NetworkWidget::load()
{
	setEnabled(false);
	ReturnValue ret = _provisionator->getDefaultSettings(this, SLOT(loadReturned(uint, ReturnValue)));
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get default settings.");
		return;
	}
}

void NetworkWidget::loadReturned(uint id, ReturnValue ret)
{
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get default settings.");
		return;
	}
	
	_settings = ret.toMap();
	
	IpEdit->setText(_settings.value("ip").toString());
	NetmaskEdit->setText(_settings.value("netmask").toString());
	GatewayEdit->setText(_settings.value("gateway").toString());
	
	QStringList dns = _settings.value("dns").toStringList();
	if (dns.size() >= 1) Dns1Edit->setText(dns[0]);
	else Dns1Edit->setText("0.0.0.0");
	if (dns.size() >= 2) Dns2Edit->setText(dns[1]);
	else Dns2Edit->setText("0.0.0.0");
	
	QStringList interfaces = _settings.value("interfaces").toStringList();
	QString interface = _settings.value("interface").toString();
	
	if (interface == "")
		interface = interfaces.first();
	else
	{
		if (!interfaces.contains(interface)) interfaces << interface;
	}
	
	InterfaceChooser->addItems(interfaces);
	InterfaceChooser->setCurrentIndex(InterfaceChooser->findText(interface));
	
	_interface = InterfaceChooser->currentText();
	_ip = IpEdit->text();
	_netmask = NetmaskEdit->text();
	_gateway = GatewayEdit->text();
	_dns1 = Dns1Edit->text();
	_dns2 = Dns2Edit->text();
	
	ApplyButton->setEnabled(false);
	DiscardButton->setEnabled(false);
	
	setEnabled(true);
}

void NetworkWidget::save()
{
	if (!validTest(IpEdit, "The IP address you entered is not valid.", IP_VALIDATOR_STRONG)) return;
	if (!validTest(NetmaskEdit, "The netmask you entered is not valid.", IP_VALIDATOR_STRONG)) return;
	if (!validTest(GatewayEdit, "The gateway you entered is not valid.", IP_VALIDATOR_STRONG)) return;
	if (!validTest(Dns1Edit, "The primary DNS server you entered is not valid.", IP_VALIDATOR_STRONG)) return;
	if (Dns2Edit->text() != "")
	{
		if (!validTest(Dns2Edit, "The secondary DNS server you entered is not valid.", IP_VALIDATOR_STRONG)) return;
	}
	
	_newsettings.clear();
	_newsettings["ip"] = IpEdit->text();
	_newsettings["netmask"] = NetmaskEdit->text();
	_newsettings["gateway"] = GatewayEdit->text();
	_newsettings["interface"] = InterfaceChooser->currentText();
	QStringList dns;
	if ((Dns1Edit->text() != "0.0.0.0") && (Dns1Edit->text() != "")) dns << Dns1Edit->text();
	if ((Dns2Edit->text() != "0.0.0.0") && (Dns2Edit->text() != "")) dns << Dns2Edit->text();
	_newsettings["dns"] = dns;
	_newsettings["oldip"] = _settings["ip"];
	
	NetworkSaveDialog dialog;
	if(dialog.setup(_provisionator, _newsettings).isError())
	{
		return;
	}
	
	dialog.exec();
	
	load();
}

void NetworkWidget::on_ApplyButton_clicked()
{
	save();
}

void NetworkWidget::on_DiscardButton_clicked()
{
	load();
}

void NetworkWidget::dataChanged()
{
	if((_interface == InterfaceChooser->currentText()) &&
		(_ip == IpEdit->text()) &&
		(_netmask == NetmaskEdit->text()) &&
		(_gateway == GatewayEdit->text()) && 
		(_dns1 == Dns1Edit->text()) &&
		(_dns2 == Dns2Edit->text()))
	{
		ApplyButton->setEnabled(false);
		DiscardButton->setEnabled(false);
	}
	else
	{
		ApplyButton->setEnabled(true);
		DiscardButton->setEnabled(true);
	}
}

bool NetworkWidget::validTest(QLineEdit *edit, QString error, QRegExpValidator *validator)
{
	QString text = edit->text();
	int size = edit->text().size();
	if ((validator != NULL) && (validator->validate(text, size) == QValidator::Acceptable))
		return(true);
	else
	{
		QMessageBox::critical(NULL, "Error", error);
		return(false);
	}
}

