/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "loginwindow.h"
#include <QHostAddress>
#include <QMessageBox>
#include <AuthToken>
#include "config.h"

using namespace QtRpc;

#define RDS_USE_ZERCONF
#if defined(Q_OS_WIN32) && defined(Q_CC_GNU)
#undef RDS_USE_ZERCONF
#endif

LoginWindow::LoginWindow(RdsClient *client, QString user, QString pass, QString host, int port, QString service, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::LoginWindow()
{
	setupUi(this);
	setWindowIcon(QPixmap(findRdsIcon("icons/32x32/kcontrol.png")));
	BannerLabel->setPixmap(QPixmap(findRdsIcon("icons/banner.png")));
	_client = client;
	resize(0, 0);
#ifdef RDS_USE_ZERCONF
	_finder = new ServiceFinder(service);
	QObject::connect(_finder, SIGNAL(serviceEvent(ServiceFinder::Service)), this, SLOT(serviceEvent(ServiceFinder::Service)));
	_finder->scan();
#endif
	if (!user.isEmpty())
		UserEdit->setText(user);
	if (!pass.isEmpty())
		PasswordEdit->setText(pass);
	if (!host.isEmpty())
	{
		ServerEntry svr;
		svr.address = host;
		svr.hostname = host;
		svr.name = host;
		svr.service = "Rds";
		svr.port = port;

		_servers << svr;

		ServersCombo->addItem(host);
	}

	UserEdit->setFocus();

	QObject::connect(ServersCombo->lineEdit(), SIGNAL(textChanged(QString)), this, SLOT(textChanged(QString)));
}

LoginWindow::~LoginWindow()
{
}

void LoginWindow::on_LoginButton_clicked()
{
	ServersCombo->setEnabled(false);
	UserEdit->setEnabled(false);
	PasswordEdit->setEnabled(false);
	LoginButton->setEnabled(false);

	int index = ServersCombo->findText(ServersCombo->currentText());

	QStringList parts = ServersCombo->currentText().split(" ", QString::SkipEmptyParts);
	if (parts.count() < 1)
	{
		QMessageBox::critical(this, "Error", "You must specify a server to connect to.");
		ServersCombo->setEnabled(true);
		UserEdit->setEnabled(true);
		PasswordEdit->setEnabled(true);
		LoginButton->setEnabled(true);
		UserEdit->setFocus();
		return;
	}

	ServerEntry entry;

	if ((index == -1) || (index >= _servers.size()))
	{
		QString tmp = "tcp://" + parts[0];
		QUrl url(tmp);
		entry.name = url.host();
		entry.hostname = url.host();
		entry.address = url.host();
		entry.service = url.path();
		entry.port = (url.port() != -1) ? url.port() : 10123;

		if ((entry.service == "") || (entry.service == "/")) entry.service = "Rds";
		if (entry.service.startsWith("/")) entry.service = entry.service.mid(1);

		if (entry.address == "")
		{
			QMessageBox::critical(this, "Error", "You must specify a server to connect to.");
			ServersCombo->setEnabled(true);
			UserEdit->setEnabled(true);
			PasswordEdit->setEnabled(true);
			LoginButton->setEnabled(true);
			UserEdit->setFocus();
			return;
		}
	}
	else
	{
		entry = _servers[index];
	}

	ReturnValue ret = connect(entry);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Connection Error", ret.errString() + "\nWhile connecting to " + entry.address);
		ServersCombo->setEnabled(true);
		UserEdit->setEnabled(true);
		PasswordEdit->setEnabled(true);
		LoginButton->setEnabled(true);
		UserEdit->setFocus();
	}
}

void LoginWindow::serviceEvent(ServiceFinder::Service service)
{
	QString server;
	if (service.validAddresses().count() < 1) return;
	server = service.serviceName();

        //Don't add blank entries
        if(server == "") return;

        //Don't add duplicate entries
	foreach(ServerEntry entry, _servers)
	{
		if (server == entry.name) return;
	}

	ServerEntry entry;
	QString host = service.hostname();
	host = host.mid(0, host.indexOf("."));
	entry.name = server;
	entry.hostname = host;
	entry.port = service.port();
	entry.address = service.validAddresses().first().toString();
#ifdef RDS_USE_ZERCONF
	entry.service = _finder->service();
#else
	entry.service = "Rds";
#endif
	_servers << entry;

	if (_servers.size() == 2) //we are adding are seccond item, redo the first one
	{
		ServersCombo->setItemText(0, ServersCombo->itemText(0) + " (" + _servers[0].hostname + ")");
	}

	if (_servers.size() > 1)
	{
		ServersCombo->addItem(server + " (" + host + ")");
	}
	else
	{
		ServersCombo->addItem(server);
	}
}

void LoginWindow::connectReturned(uint id, ReturnValue ret)
{
	Q_UNUSED(id);
	if (ret.isError())
	{
		if ((ret.errNumber() == 49) || (ret.errNumber() == 82))
		{
			QMessageBox::critical(this, "Login Failed", "The username or password is not correct.");
		}
		else
		{
			QMessageBox::critical(this, "Connection Error", ret.errString());
		}
		ServersCombo->setEnabled(true);
		UserEdit->setEnabled(true);
		PasswordEdit->setEnabled(true);
		LoginButton->setEnabled(true);
		UserEdit->setFocus();
	}
	else
	{
		accept();
	}
}

QString LoginWindow::user()
{
	return(_user);
}

QString LoginWindow::pass()
{
	return(_pass);
}

QString LoginWindow::host()
{
	return(_host);
}

int LoginWindow::port()
{
	return(_port);
}

QString LoginWindow::service()
{
	return(_service);
}

void LoginWindow::textChanged(QString text)
{
	if (text.toLower().startsWith("unconfigured"))
	{
		_prevuser = UserEdit->text();
		_prevpass = PasswordEdit->text();
		UserEdit->setText("default");
		PasswordEdit->setText("default");
		UserEdit->setEnabled(false);
		PasswordEdit->setEnabled(false);
	}
	else if (_lasttext.toLower().startsWith("unconfigured"))
	{
		UserEdit->setEnabled(true);
		PasswordEdit->setEnabled(true);
		if((_prevuser == "default") && (_prevpass == "default"))
		{
			UserEdit->setText("");
			PasswordEdit->setText("");
		}
		else
		{
			UserEdit->setText(_prevuser);
			PasswordEdit->setText(_prevpass);
		}
	}
	_lasttext = text;
	//qDebug() << "Text Changed:" << text;
}

ReturnValue LoginWindow::connect(LoginWindow::ServerEntry server)
{
	_user = UserEdit->text();
	_pass = PasswordEdit->text();
	_host = server.address;
	_port = server.port;
	_service = server.service;

	QString url = QString("tcp://%1:%2/%3").arg(_host).arg(_port).arg(_service);
	QtRpc::AuthToken token(UserEdit->text(), PasswordEdit->text());

	return _client->connect(QUrl(url), this, SLOT(connectReturned(uint, ReturnValue)), token);
}

