/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "dnszoneeditwidget.h"
#include <QMessageBox>
#include <QDebug>
#include <RdsDnsManager>
#include <RdsDnsZone>
#include <RdsUtils>
#include <RdsDaemonManager>
#include <RdsFileManager>
#include "config.h"

DnsZoneEditWidget::DnsZoneEditWidget(RdsDnsManager *manager, QWidget* parent)
		: RdsEntityWidget(manager, parent), Ui::DnsZoneEditWidget(),
		_zone(0)
{
	setupUi(this);
	setApplyButton(ApplyButton);
	setDiscardButton(DiscardButton);

	addWidget("domain", DomainName, NoMulti);
	addWidget("admin", SOAAdministrator, None, "^[a-zA-Z0-9_%+-]+(@[a-zA-Z0-9.-]+\\.[a-zA-Z]{2,4})?$", "Must be a valid email address, and must not have any '.' characters in the name");
	addWidget("primary", SOAPrimary);
	addWidget("refresh", SOARefresh);
	addWidget("retry", SOARetry);
	addWidget("expiry", SOAExpiry);
	addWidget("minimum", SOAMinimum);

	QObject::connect(manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));
}

DnsZoneEditWidget::~DnsZoneEditWidget()
{
}

void DnsZoneEditWidget::entityRenamed(QString oldid, QString newid)
{
	if (oldid != input()) return;

	QStringList outlist = outputs();
	for (int i = 0; i < outlist.size(); i++)
	{
		if (outlist[i] == input()) outlist[i] = newid;
	}

	setInput(newid);
	setOutputs(outlist);

	setField("domain", newid.left(newid.count() - 1));
}

#define GET_DATA(value, getter) \
	ret = _zone->getter(); \
	if (ret.isError()) \
		return ret; \
	fields[value] = ret;

ReturnValue DnsZoneEditWidget::getData()
{
	QVariantMap fields;
	ReturnValue ret;

	delete _zone;
	_zone = 0;
	_oldName = QString();

	RdsDnsManager mgr;
	mgr.init();
	ret = mgr.zone(input());
	if (ret.isError())
		return ret;
	_zone = new RdsDnsZone(ret);

	GET_DATA("domain", domainName);
	GET_DATA("admin", adminNameServer);
	GET_DATA("primary", primaryNameServer);

	fields["admin"] = fields["admin"].toString().replace(fields["admin"].toString().indexOf("."), 1, '@');
	ret = _zone->soaValues();
	if (ret.isError())
		return ret;

	QVariantList list = ret.toList();

	fields["refresh"] = list.at(1);
	fields["retry"] = list.at(2);
	fields["expiry"] = list.at(3);
	fields["minimum"] = list.at(4);

	return(fields);
}

#define PUT_SOA_VALUE(name, index) \
	if (fields.contains(name)) \
	{ \
		ret = _zone->soaValues(); \
		if (ret.isError()) \
		{ \
			qCritical() << "DnsZoneEditWidget: Failed to fetch SOA values" << ret; \
			return false; \
		} \
		QStringList list = ret.toStringList();\
		list.replace(index, fields.value(name).toString());\
		ret = _zone->setSoaValues(list);\
		if (ret.isError())\
		{\
			qCritical() << "DnsZoneEditWidget: Failed to set SOA value, " name ":" << ret;\
			return false;\
		}\
	}

bool DnsZoneEditWidget::setData(QString id, QVariantMap fields)
{
	ReturnValue ret;

	if (fields.contains("domain"))
	{
		_oldName = id.left(id.count() - 1);
		ret = _zone->setDomainName(fields.value("domain").toString());
		if (ret.isError())
		{
			qCritical() << "DnsZoneEditWidget: Failed to set new domain name" << ret;
			return false;
		}
	}

	if (fields.contains("admin"))
	{
		ret = _zone->setAdminNameServer(fields.value("admin").toString().replace("@", "."));
		if (ret.isError())
		{
			qCritical() << "DnsZoneEditWidget: Failed to set admin contact" << ret;
			return false;
		}
	}
	if (fields.contains("primary"))
	{
		ret = _zone->setPrimaryNameServer(fields.value("primary").toString());
		if (ret.isError())
		{
			qCritical() << "DnsZoneEditWidget: Failed to set primary nameserver" << ret;
			return false;
		}
	}

	PUT_SOA_VALUE("refresh", 1);
	PUT_SOA_VALUE("retry", 2);
	PUT_SOA_VALUE("expiry", 3);
	PUT_SOA_VALUE("minimum", 4);

	return true;
}

void DnsZoneEditWidget::entityUpdated(QString id)
{
	if (id != input()) return;

	if (unsavedChanges())
	{
		if (QMessageBox::question(this, "Dns Changed",
		                          "Another user has made changes to the item you are editing. Would you like to overwrite your changes?",
		                          QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
		{
			RdsEntityWidget::entityUpdated(id);
		}
	}
	else
	{
		RdsEntityWidget::entityUpdated(id);
	}
}

#define CHECK_FOR_ERROR(function, errormsg) \
	ret = function; \
	if (ret.isError()) \
	{ \
		qCritical() << errormsg << ret; \
		QMessageBox::critical(this, "Error", errormsg + ret.errString()); \
		return false; \
	}

bool DnsZoneEditWidget::endSetData()
{
	ReturnValue ret;

	CHECK_FOR_ERROR(_zone->save(), "Failed to save the zone object:");

	if (!_oldName.isEmpty())
	{
		RdsDnsManager mgr;
		CHECK_FOR_ERROR(mgr.init(), "Failed to get RdsDnsManager service:");
		CHECK_FOR_ERROR(_zone->domainName(), "Failed to get new domain name:");

		QString newName = ret.toString();
		//qDebug() << "Renaming a zone!" << _oldName << newName;

		CHECK_FOR_ERROR(mgr.removeZone(_oldName + "."), "Failed to remove old zone (" + _oldName + ") for renaming: ");
// 		CHECK_FOR_ERROR(mgr.addZone(newName + "."), "DnsZoneEditWidget: Failed to add new zone (for renaming):");
		CHECK_FOR_ERROR(mgr.save(), "Failed to save the DNS object (for renaming): ");
		CHECK_FOR_ERROR(mgr.renameEntity(_oldName + ".", newName + "."), "Failed to rename entity: ");

		entityRenamed(_oldName + ".", newName + ".");
		_oldName = QString();
	}

	RdsDaemonManager mgr;

	CHECK_FOR_ERROR(mgr.init(), "Failed to get RdsDaemonManager service: ");
	CHECK_FOR_ERROR(mgr.reloadService("Dns"), "Failed to restart dns: ");

	return(true);
}

