/***************************************************************************	
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "dnseditwidget.h"
#include <QMessageBox>
#include <QDebug>
#include <RdsDnsManager>
#include <RdsUtils>
#include <RdsDaemonManager>
#include <RdsFileManager>
#include "config.h"

DnsEditWidget::DnsEditWidget(RdsDnsManager *manager, QWidget* parent)
		: RdsEntityWidget(manager, parent), Ui::DnsEditWidget()
{
	setupUi(this);
	setApplyButton(ApplyButton);
	setDiscardButton(DiscardButton);

// 	addWidget("name", Name, NoMulti, "^[a-zA-Z0-9 \\-_]*$", "The name may only contain letters, numbers, underscore(_), and hyphen(-).");
// 	addWidget("description", Description);
// 	addWidget("path", Path);
// 	addWidget("readonly", ReadOnly);
// 	addWidget("hidden", Hidden);

#define IP_VALIDATOR QRegExp("^([0-9]{0,3}\\.){0,3}[0-9]{0,3}$")
#define IP_VALIDATOR_STRONG QRegExp("^(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$")

	addWidget("dnsservers", DnsServers, "list", SIGNAL(changed()));
	DnsServers->setSeperator(QRegExp(",[ ]*"), ", ");
	DnsServers->setErrorText("You must specify a valid IP address.");
	DnsServers->setWeakValidator(IP_VALIDATOR);
	DnsServers->setStrongValidator(IP_VALIDATOR_STRONG);
	DnsServers->setUpDownArrows(true);

	QObject::connect(manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));

	_dnsmgr = manager;
}

DnsEditWidget::~DnsEditWidget()
{
}

void DnsEditWidget::entityRenamed(QString oldid, QString newid)
{
	if (oldid != input()) return;

	QStringList outlist = outputs();
	for (int i = 0; i < outlist.size(); i++)
	{
		if (outlist[i] == input()) outlist[i] = newid;
	}

	setInput(newid);
	setOutputs(outlist);
}

ReturnValue DnsEditWidget::getData()
{
	QVariantMap fields;
	if (input() != "manager") return(ReturnValue(1, "Invalid Dns"));

	ReturnValue ret;

	ret = _dnsmgr->forwarders();
	if (ret.isError())
		return ret;

	fields["dnsservers"] = ret.toStringList();

	return(fields);
}

bool DnsEditWidget::setData(QString id, QVariantMap fields)
{
	if (id != "manager")
	{
		qCritical() << "DnsEditWidget: Attempting to set data on something that isn't a manager";
		return false;
	}
	ReturnValue ret;

	ret = _dnsmgr->setForwarders(fields.value("dnsservers").toStringList());
	if (ret.isError())
	{
		qCritical() << "DnsEditWidget: Failed to set forwarders" << ret;
		return false;
	}
	return true;
}

void DnsEditWidget::entityUpdated(QString id)
{
	if (id != input()) return;

	if (unsavedChanges())
	{
		if (QMessageBox::question(this, "Dns Changed",
		                          "Another user has made changes to the item you are editing. Would you like to overwrite your changes?",
		                          QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
		{
			RdsEntityWidget::entityUpdated(id);
		}
	}
	else
	{
		RdsEntityWidget::entityUpdated(id);
	}
}

bool DnsEditWidget::endSetData()
{
	ReturnValue ret = _dnsmgr->save();
	if (ret.isError())
	{
		qWarning() << "Failed to save changes:" << ret;
		return(false);
	}

	RdsDaemonManager mgr;
	ret = mgr.init();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		return(false);
	}

	ret = mgr.reloadService("Dns");

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to restart dns: " + ret.errString());
		return(false);
	}

	return(true);
}

