/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "addsubnetdialog.h"
#include <RdsUtils>
#include <QMessageBox>
#include <QDebug>
#include <QRegExpValidator>
#include <RdsDaemonManager>
#include <RdsDhcpSubnet>
#include <RdsDhcpValues>
#include <RdsClient>
#include <rdsprovisionator.h>
#include <config.h>

#define IP_VALIDATOR QRegExp("^([0-9]{0,3}\\.){0,3}[0-9]{0,3}$")
#define IP_VALIDATOR_STRONG QRegExp("^(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$")

AddSubnetDialog::AddSubnetDialog(QString parentid, RdsDhcpManager* manager, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::AddSubnetDialog()
{
	setupUi(this);
	_manager = manager;
	_parent = parentid;
	
	NetworkEdit->setValidator(new QRegExpValidator(IP_VALIDATOR, this));
	NetmaskEdit->setValidator(new QRegExpValidator(IP_VALIDATOR, this));
	StartEdit->setValidator(new QRegExpValidator(IP_VALIDATOR, this));
	EndEdit->setValidator(new QRegExpValidator(IP_VALIDATOR, this));
	
	ReturnValue ret = rdsClient()->provisionator();
	if(ret.isError())
	{
		qWarning() << "Failed to get provisionator object:" << ret;
		return;
	}
	
	RdsProvisionator p = ret;
	ret = p.getDefaultSettings();
	if(ret.isError())
	{
		qWarning() << "Failed to get network settings:" << ret;
		return;
	}
	
	QVariantMap settings = ret.toMap();
	QHostAddress netmask(settings.value("netmask").toString());
	QHostAddress ip(settings.value("ip").toString());
	QHostAddress network(netmask.toIPv4Address() & ip.toIPv4Address());
	
	NetworkEdit->setText(network.toString());
	NetmaskEdit->setText(netmask.toString());
	
	QStringList parts = network.toString().split(".");
	if (parts.size() == 4)
	{
		parts[3] = "100";
		StartEdit->setText(parts.join("."));
		parts[3] = "200";
		EndEdit->setText(parts.join("."));
	}
	
}


AddSubnetDialog::~AddSubnetDialog()
{
}

void AddSubnetDialog::accept()
{
	if(NameEdit->text() == "")
	{
		QMessageBox::critical(this, "Error", "You must specify a name for the new subnet.");
		return;
	}
	
	if(NameEdit->text().contains("/"))
	{
		QMessageBox::critical(this, "Error", "The / character is not allowed in the name of this subnet.");
		return;
	}
	
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(NetworkEdit->text()))
	{
		QMessageBox::critical(this, "Error", "The network address must be a valid IP address");
		return;
	}
	
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(NetmaskEdit->text()))
	{
		QMessageBox::critical(this, "Error", "The netmask must be a valid IP address");
		return;
	}
	
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(StartEdit->text()))
	{
		QMessageBox::critical(this, "Error", "The start address must be a valid IP address");
		return;
	}
	
	if(!QRegExp(IP_VALIDATOR_STRONG).exactMatch(EndEdit->text()))
	{
		QMessageBox::critical(this, "Error", "The end address must be a valid IP address");
		return;
	}
	
	QHostAddress network(NetworkEdit->text());
	QHostAddress netmask(NetmaskEdit->text());
	QHostAddress start(StartEdit->text());
	QHostAddress end(EndEdit->text());
	
	if((network.toIPv4Address() & netmask.toIPv4Address()) != network.toIPv4Address())
	{
		QMessageBox::critical(this, "Error", "The network address and netmask are not consistent. Please make sure you enter a valid network address for the netmask.");
		return;
	}
	
	if((start.toIPv4Address() & netmask.toIPv4Address()) != network.toIPv4Address())
	{
		QMessageBox::critical(this, "Error", "The start address is not in the range defined by the network address and netmask.");
		return;
	}
	
	if((end.toIPv4Address() & netmask.toIPv4Address()) != network.toIPv4Address())
	{
		QMessageBox::critical(this, "Error", "The end address is not in the range defined by the network address and netmask.");
		return;
	}
	
	if(start.toIPv4Address() > end.toIPv4Address())
	{
		QMessageBox::critical(this, "Error", "The start address must be lower than the end address.");
		return;
	}
	
	ReturnValue ret = _manager->checkRange("", start, end);
	if(ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to check start and end addresses: " + ret.errString());
		return;
	}
	
	if(!ret.toBool())
	{
		QMessageBox::critical(this, "Error", "The range of this subnet overlaps the range of another subnet. Please modify the start and end addresses and try again.");
		return;
	}
	
	ret = _manager->values(_parent);
	if(ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get values object: " + ret.errString());
		return;
	}
	
	RdsDhcpValues values = ret;
	
	ret = values.addSubnet(NameEdit->text());
	if(ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to add subnet: " + ret.errString());
		return;
	}

	_manager->addEntity(_parent + "/s" + NameEdit->text());
	RdsDhcpSubnet subnet = ret;
	
	ret = subnet.setAddress(QHostAddress(NetworkEdit->text()));
	if(ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to set network address: " + ret.errString());
		QDialog::accept();
		return;
	}
	
	ret = subnet.setNetmask(QHostAddress(NetmaskEdit->text()));
	if(ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to set netmask: " + ret.errString());
		QDialog::accept();
		return;
	}
	
	QList<QPair<QHostAddress, QHostAddress> > ranges;
	ranges << QPair<QHostAddress, QHostAddress>(QHostAddress(StartEdit->text()), QHostAddress(EndEdit->text()));
	
	ret = subnet.setRanges(ranges);
	if(ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to set range: " + ret.errString());
		QDialog::accept();
		return;
	}
	
	RdsDaemonManager mgr;
	ret = mgr.init();
	
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		QDialog::accept();
		return;
	}
	
	ret = mgr.restartService("Dhcp");
	
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to restart the DHCP server: " + ret.errString());
		QDialog::accept();
		return;
	}
	
	//if we get here, everything worked
	QDialog::accept();
}
