/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "addsharedialog.h"
#include "browsedialog.h"
#include <RdsUtils>
#include <QMessageBox>
#include <QDebug>
#include <QFileInfo>
#include <RdsFileManager>
#include <RdsShareManager>
#include <RdsShare>
#include <RdsDaemonManager>
#include <RdsNtAcl>
#include <RdsSid>
#include <RdsVolumeManager>
#include <RdsVolume>
#include "initializevolumedialog.h"
#include <config.h>

QString AddShareDialog::_defaultpath = "";

AddShareDialog::AddShareDialog(QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::AddShareDialog()
{
	setupUi(this);
	BrowseButton->setIcon(QPixmap(findRdsIcon("icons/16x16/folder.png")));

	if (_defaultpath == "")
	{
		RdsVolumeManager mgr;
		ReturnValue ret = mgr.init();

		if (ret.isError())
		{
			QMessageBox::warning(this, "Error", "Unable to get the default volume manager:" + ret.errString());
			_defaultpath = "";
		}
		else
		{
			_defaultpath = "/volumes/Hard Drive/";
			ret = mgr.defaultVolume();

			if (ret.isError())
			{
				QMessageBox::warning(this, "Error", "Unable to get the default volume name:" + ret.errString());
				_defaultpath = "";
			}
			else
			{
				ret = mgr.volume(ret.toString());
				if (ret.isError())
				{
					QMessageBox::warning(this, "Error", "Unable to get the default volume:" + ret.errString());
					_defaultpath = "";
				}
				else
				{
					RdsVolume volume;
					volume = ret;
					_defaultpath = "/volumes/" + volume.name().toString() + "/";
				}
			}
		}
	}

	if (!_defaultpath.endsWith("/"))
	{
		_defaultpath += "/";
	}

	PathEdit->setText(_defaultpath);
	_lasttxt = "";
}

AddShareDialog::AddShareDialog(QString path, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::AddShareDialog()
{
	setupUi(this);
	PathEdit->setText(path);
	_lasttxt = "";
}


AddShareDialog::~AddShareDialog()
{
}

void AddShareDialog::accept()
{
	if (NameEdit->text().trimmed() == "")
	{
		QMessageBox::critical(this, "Error", "You must specify a share name.");
		return;
	}

	if (PathEdit->text().trimmed() == "")
	{
		QMessageBox::critical(this, "Error", "You must specify a path.");
		return;
	}

	if (!NameEdit->text().contains(QRegExp("^[a-zA-Z0-9 \\-_]+$")))
	{
		QMessageBox::critical(NULL, "Error", "The name may only contain letters, numbers, underscore(_), and hyphen(-).");
		return;
	}

	//Set the new default path
	QStringList parts = PathEdit->text().split("/", QString::SkipEmptyParts);
	if (parts.size() > 2)
	{
		parts.removeAt(parts.size() - 1);
	}
	_defaultpath = "/" + parts.join("/") + "/";

	ReturnValue ret = checkValidVolume(PathEdit->text());
	if (ret.isError()) return;

	RdsFileManager filemgr;
	ret = filemgr.init();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get file manager: " + ret.errString());
		return;
	}

	ret = filemgr.exists(PathEdit->text());
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get path: " + ret.errString());
		return;
	}

	if (!ret.toBool()) //if the path doesn't exist
	{
		ret = filemgr.createDirectory(PathEdit->text(), true);
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to create path: " + ret.errString());
			reject();
			return;
		}
	}

	if (filemgr.ntPermissions(PathEdit->text()).isError())
	{
		ret = setPermissions(PathEdit->text(), filemgr);
		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to set permissions: " + ret.errString());
			reject();
			return;
		}
	}

	RdsShareManager sharemgr;
	ret = sharemgr.init();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get share manager: " + ret.errString());
		return;
	}

	ret = sharemgr.addShare(NameEdit->text(), PathEdit->text());
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to create share: " + ret.errString());
		reject();
		return;
	}

	sharemgr.addEntity("@SHARE/" + NameEdit->text() + "/");

	ret = sharemgr.share(NameEdit->text());
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get share: " + ret.errString());
		reject();
		return;
	}

	RdsShare share = ret;

	ret = share.setReadOnly(false);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to set share read only: " + ret.errString());
		reject();
		return;
	}

	ret = share.setHidden(false);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to set share hidden: " + ret.errString());
		reject();
		return;
	}

	ret = sharemgr.save();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to save changes: " + ret.errString());
		reject();
		return;
	}

	RdsDaemonManager daemonmgr;
	ret = daemonmgr.init();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get daemon service: " + ret.errString());
		reject();
		return;
	}

	ret = daemonmgr.reloadService("Samba");
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to reload samba: " + ret.errString());
		reject();
		return;
	}

	//if we get here, everything worked
	QDialog::accept();
}

void AddShareDialog::on_NameEdit_textEdited(QString text)
{
	if (PathEdit->text() == (_defaultpath + _lasttxt))
	{
		PathEdit->setText(_defaultpath + text);
	}

	_lasttxt = text;
}

void AddShareDialog::on_BrowseButton_clicked()
{
	BrowseDialog dialog(BrowseDialog::Volumes | BrowseDialog::FileSystem);

	if (dialog.exec())
	{
		if (dialog.path() == "") return;
		ReturnValue ret = RdsUtils::getSystemPath(dialog.path());

		if (ret.isError())
		{
			QMessageBox::critical(this, "Error", "Failed to look up path: " + ret.errString());
			return;
		}

		PathEdit->setText(ret.toString());

		QFileInfo info(ret.toString());
		if(NameEdit->text() == "") NameEdit->setText(info.fileName());
	}
}

ReturnValue AddShareDialog::checkValidVolume(QString path)
{
	RdsVolumeManager volmgr;
	ReturnValue ret = volmgr.init();
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to get volume manager: " + ret.errString());
		return(ret);
	}

	if (path.startsWith("/volumes/"))
	{
		path = path.mid(9);
		int index = path.indexOf("/");
		if (index > 0) path.truncate(index);
	}
	else
	{
		return(false); //not exactly an error
	}

	QString name = path;
	ret = volmgr.volume(name);
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to get volume: " + ret.errString());
		return(ret);
	}

	RdsVolume volume;
	volume = ret;

	ret = volume.id();
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to get volume id: " + ret.errString());
		return(ret);
	}

	QString id = ret.toString();

	ret = volume.isAdded();
	if (ret.isError())
	{
		QMessageBox::critical(NULL, "Error", "Failed to get volume state: " + ret.errString());
		return(ret);
	}

	if (!ret.toBool())
	{
		ret = volume.fsType();
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to get volume type: " + ret.errString());
			return(ret);
		}

		if (!ret.toString().startsWith("ext"))
		{
			int result = QMessageBox::question(NULL, "Volume Type", "The volume you have selected is not a native volume type. You can share files from this volume, but features like file permissions and ownership will not work properly until the volume is initialized. Do you want to initialize this volume before using it?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No);

			if (result == QMessageBox::Yes)
			{
				ret = volume.unmount();
				if (ret.isError())
				{
					QMessageBox::critical(NULL, "Error", "Failed to unmount volume: " + ret.errString());
					return(ret);
				}

				InitializeVolumeDialog dialog;
				ret = dialog.initializeVolume(&volmgr, name);
				if (ret.isError())
				{
					QMessageBox::critical(NULL, "Error", "Failed to initialize volume: " + ret.errString());
					return(ret);
				}

				if (!ret.toBool()) return(false);

				dialog.exec();

				ret = dialog.result();
				if (ret.isError())
				{
					QMessageBox::critical(NULL, "Error", "Failed to initialize volume: " + ret.errString());
					return(ret);
				}

				id = ret.toString();
			}
		}

		//if we get this far, we need to add the volume
		ret = volmgr.addVolume(id, name);
		if (ret.isError())
		{
			QMessageBox::critical(NULL, "Error", "Failed to add volume: " + ret.errString());
			return(ret);
		}
	}


	return(true);
}

ReturnValue AddShareDialog::setPermissions(QString path, RdsFileManager& filemgr)
{
	RdsNtAcl acl;
	acl.setOwner(RdsSid("LA")); //Domain Administrator
	acl.setGroup(RdsSid("DA")); //Domain Administrators group
	acl.setDaclFlags(RdsNtAcl::Protected); //Do not inherit permissions from parent

	RdsAce ace;
	//Everyone Can read and modify
	ace.setSid(RdsSid("DU"));
	ace.setFlags(RdsAce::FileInherit | RdsAce::FolderInherit);
	ace.setAccess(RdsAce::ReadControl | RdsAce::Delete | RdsAce::ReadProperty | RdsAce::WriteProperty | RdsAce::CreateChild | RdsAce::DeleteChild | RdsAce::ListChildren | RdsAce::SelfWrite | RdsAce::ListObject | RdsAce::ControlAccess);
	acl.dacl() << ace;

	//Domain Admins have full control
	ace.setSid(RdsSid("DA"));
	ace.setFlags(RdsAce::FileInherit | RdsAce::FolderInherit);
	ace.setAccess(RdsAce::StandardAll | RdsAce::FileAll);
	acl.dacl() << ace;

	return(filemgr.setNtPermissions(path, acl));
}

