/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "adddomaindialog.h"
#include <RdsUtils>
#include <QMessageBox>
#include <QDebug>
#include <QRegExpValidator>
#include <RdsDaemonManager>
#include <RdsDnsZone>
#include <RdsDnsRecord>
#include <RdsClient>
#include <rdsprovisionator.h>
#include <config.h>

#define IP_VALIDATOR QRegExp("^([0-9]{0,3}\\.){0,3}[0-9]{0,3}$")
#define IP_VALIDATOR_STRONG QRegExp("^(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))\\.(([0-9][0-9]?)|(1[0-9][0-9]?)|(2[0-4][0-9])|(25[0-5]))$")


AddDomainDialog::AddDomainDialog(RdsDnsManager* manager, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::AddDomainDialog()
{
	setupUi(this);
	_manager = manager;

	NameEdit->setValidator(new QRegExpValidator(QRegExp("^[a-zA-Z0-9][a-zA-Z0-9_.-]+[a-zA-Z0-9]$"), this));
	IpEdit->setValidator(new QRegExpValidator(IP_VALIDATOR, this));
	NameServerEdit->setValidator(new QRegExpValidator(QRegExp("^[a-zA-Z0-9][a-zA-Z0-9]+[a-zA-Z0-9]$"), this));

	ReturnValue ret = rdsClient()->provisionator();
	if (ret.isError())
	{
		qWarning() << "Failed to get provisionator object:" << ret;
		return;
	}

	RdsProvisionator p = ret;
	ret = p.getDefaultSettings();
	if (ret.isError())
	{
		qWarning() << "Failed to get network settings:" << ret;
		return;
	}

	QVariantMap settings = ret.toMap();
	NameServerEdit->setText("NS1");
	ContactEdit->setText("Administrator@" + RdsUtils::realm().toLower());
	IpEdit->setText(settings.value("ip").toString());
}


AddDomainDialog::~AddDomainDialog()
{
}

void AddDomainDialog::accept()
{
	if (NameEdit->text() == "")
	{
		QMessageBox::critical(this, "Error", "You must specify a name for the new domain.");
		return;
	}

	if (ContactEdit->text() == "")
	{
		QMessageBox::critical(this, "Error", "You must specify the administrator contact.");
		return;
	}

	if (NameServerEdit->text() == "")
	{
		QMessageBox::critical(this, "Error", "You must specify the name server hostname");
		return;
	}

	if (!QRegExp(IP_VALIDATOR_STRONG).exactMatch(IpEdit->text()))
	{
		QMessageBox::critical(this, "Error", "The nameserver IP address must be a valid IP address");
		return;
	}

	QString domain = NameEdit->text()  + ".";

	ReturnValue ret = _manager->addZone(domain);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed add domain: " + ret.errString());
		QDialog::accept();
		_manager->reload();
		return;
	}

	RdsDnsZone zone = ret;

	ret = zone.setAdminNameServer(ContactEdit->text());
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to set the admin contact: " + ret.errString());
		QDialog::accept();
		_manager->reload();
		return;
	}

	RdsDnsRecord record;
	record.setType(RdsDnsZone::A);
	record.setKey(NameServerEdit->text());
	record.setValue(IpEdit->text());

	ret = zone.addRecord(record);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to add the name server record: " + ret.errString());
		QDialog::accept();
		_manager->reload();
		return;
	}

	record.setType(RdsDnsZone::NS);
	record.setKey("IN");
	record.setValue(NameServerEdit->text());

	ret = zone.addRecord(record);
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to add the default name server record: " + ret.errString());
		QDialog::accept();
		_manager->reload();
		return;
	}

	ret = zone.setPrimaryNameServer(NameServerEdit->text());
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to set the primary name server: " + ret.errString());
		QDialog::accept();
		_manager->reload();
		return;
	}

	ret = zone.save();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to save zone file: " + ret.errString());
		QDialog::accept();
		_manager->reload();
		return;
	}

	ret = _manager->save();
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to save the DNS config files: " + ret.errString());
		QDialog::accept();
		_manager->reload();
		return;
	}

	_manager->addEntity(domain);
	_manager->updateEntity(domain);

	RdsDaemonManager mgr;
	ret = mgr.init();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to get RdsDaemonManager service: " + ret.errString());
		QDialog::accept();
		return;
	}

	ret = mgr.reloadService("Dns");

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to restart the DNS server: " + ret.errString());
		QDialog::accept();
		return;
	}

	//if we get here, everything worked
	QDialog::accept();
}
