/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include <QCoreApplication>
#include <QString>
#include <QSettings>
#include <QDebug>
#include <QFileInfo>
#include <AuthToken>

#include <RdsClient>
#include <RdsFileManager>
#include <RdsUtils>
#include <RdsNtAcl>
#include <RdsLdapSession>

#include "qxtcommandoptions.h"

int main(int argc, char *argv[])
{
	QCoreApplication app(argc, argv);
	QxtCommandOptions options;
	bool socket = true;

	options.add("help", "Show this help message");
	options.alias("help", "h");
	options.add("propagate", "Propagate inherited permissions from the given directory");
	options.alias("propagate", "p");
	options.add("update", "Update inherited permissions for the given file or directory");
	options.alias("update", "u");
	options.add("read", "Print out the permissions on a given file");
	options.alias("read", "r");

	options.addSection("Connection options");
	options.add("host", "The hostname or IP to connect to", QxtCommandOptions::Required);
	options.add("port", "Port to connect to the RDS server", QxtCommandOptions::Required);
	options.add("service", "A custom service name to connect to", QxtCommandOptions::Required);
	options.add("user", "Username to authenticate with", QxtCommandOptions::Required);
	options.add("pass", "Password to authenticate with", QxtCommandOptions::Required);

	options.parse(argc, argv);

	if (options.count("help") || options.showUnrecognizedWarning())
	{
		options.showUsage();
		return -1;
	}

	QString service = getenv("RDS_SERVICE");
	if (options.count("service"))
		service = options.value("service").toString();
	if (service == "")
		service = "Rds";

	QString host = getenv("RDS_SERVER");
	if (options.count("host"))
		host = options.value("host").toString();
	if (host == "")
		host = "localhost";
	else
		socket = false;
	
	int port = 0;
	QString tmp = getenv("RDS_PORT");
	if (tmp.toInt() != 0) port = tmp.toInt();
	if (options.count("port"))
		port = options.value("port").toInt();
	if(port == 0)
		port = 10123;
	else
		socket = false;

	QSettings settings(RDS_CONF_PATH + "/rds.conf", QSettings::IniFormat);
	QString user = settings.value("adminuser").toString();
	if (options.count("user"))
		user = options.value("user").toString();
	QString pass = settings.value("adminpw").toString();
	if (options.count("pass"))
		pass = options.value("pass").toString();

	RdsClient client;
	QString url;
	if(socket)
		url = QString("socket:///var/run/rdsd.socket:%3").arg(service);
	else
		url = QString("tcp://%1:%2/%3").arg(host).arg(port).arg(service);
	QtRpc::AuthToken token(user, pass);
	ReturnValue ret = client.connect(url, token);
	if (ret.isError())
	{
		qCritical() << "Failed to connect to server:" << ret.errString();
		return(1);
	}

	RdsClient::setGlobalInstance(&client);
	
	ret = RdsUtils::init();
	if (ret.isError())
	{
		qCritical() << "Failed to fetch domain data:" << ret.errString();
		return(1);
	}
	
	ret = client.ldapSession();
	if (ret.isError())
	{
		qCritical() << "Failed to get ldap session:" << ret.errString();
		return(1);
	}
	
	RdsLdapSession ldap;
	ldap = ret;
	RdsLdapSession::setGlobalSession(&ldap);
	

	RdsFileManager filemgr;
	ret = filemgr.init();
	if (ret.isError())
	{
		qCritical() << "Failed to initialize FileManager instance:" << ret.errString();
		return(1);
	}

	if (options.count("read"))
	{
		foreach(QString file, options.positional())
		{
			file = QFileInfo(file).absoluteFilePath();
			ret = RdsUtils::ntPermissions(file);
			if (ret.isError())
			{
				qCritical() << "Failed to get permissions for " + file + ":" << ret;
				continue;
			}
			qDebug() << file + ":" << ret.value<RdsNtAcl>();
		}
		return 0;
	}
	else if (options.count("propagate"))
	{
		foreach(QString file, options.positional())
		{
			file = QFileInfo(file).absoluteFilePath();
			ret = filemgr.propagateInheritedPermissions(file);
			if (ret.isError())
			{
				qCritical() << "Failed to propagate permissions for " + file + ":" << ret;
				continue;
			}
		}
		return 0;
	}
	else if (options.count("update"))
	{
		foreach(QString file, options.positional())
		{
			file = QFileInfo(file).absoluteFilePath();
			ret = filemgr.updateInheritedPermissions(file);
			if (ret.isError())
			{
				qCritical() << "Failed to update permissions for " + file + ":" << ret;
				continue;
			}
		}
		return 0;
	}
	options.showUsage();
	return -1;
}
