/* win_events.c: event-response functions, more or less, for randim. */

/* This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.  Please see the file "COPYING"
   for details.  If you have not received it along with this program,
   please write to the Free Software Foundation, Inc., 675 Mass. Ave.,
   Cambridge, MA 02139, USA. */

#include "randim.h"

/**********************************************************************/
/* Main panel utilities (button response functions, in order) */

/* 'Next' button - generate a new image.*/
void
next_imt(Widget w, XtPointer cdata, XtPointer rdata)
{
  next_im();	/*panel text item activation for next image*/
}

/* 'Pause' button - ability to turn off drawing to conserve CPU. */
void
pause_toggle(Widget w, XtPointer cdata, XtPointer rdata)
{
  if (drawflag)
    halt_draw();
  else
    resume_draw();
}

/* 'Edit' button - pop up edit window.*/
void
edit_trans(Widget w, XtPointer cdata, XtPointer rdata)
{
  panel_off();
  XtPopup(epopup,XtGrabNone);  /* was XtGrabNonexclusive */
}

/* 'Zoom pt' button - sets up to input point*/
void
zoom_point(Widget w, XtPointer cdata, XtPointer rdata)
{
  message("Select pt");
  panel_off();
  diNew = diFlag;

  /*turn on a function to be called by the widget to respond to mouse*/
  XtAddEventHandler(cwidg,ButtonPressMask,False,
		    (XtEventHandler) zoom_point_aux,0);
}

/* 'Zoom rect' button - sets up to draw interactive rectangle */
void
zoom_rect(Widget w, XtPointer cdata, XtPointer rdata)
{
  message("Sel rect");
  panel_off();
  diNew = diFlag;
  XSetFunction(display,gc,GXxor);	/* change to draw overlap mode */

  XtAddEventHandler(cwidg,ButtonPressMask|ButtonReleaseMask|
		    ButtonMotionMask,False, (XtEventHandler) zoom_rect_aux,0);
}

/* 'Set Zmag' button - pops up a query panel */
void
zmag_query(Widget w, XtPointer cdata, XtPointer rdata)
{
  char s[20];

  sprintf(s,"%2f",zmag);
  panel_off();
  arg[0].name=XtNlabel,arg[0].value = (XtArgVal) "Enter magnification:";
  arg[1].name = XtNvalue,arg[1].value = (XtArgVal) s;
  XtSetValues(ss_in,arg,2);
  dpopstate = 2;
  XtPopup(dpopup,XtGrabNone);
}

/* 'XformColor' button - toggle colormode */
void
colorToggle1(Widget w, XtPointer cdata, XtPointer rdata)
{
  if (!gcflag) {
    message("B & W only!");
    return;
  } else if (cmode & 2) {
    message("invalid");
    return;
  }
  cmode = cmode ^ 1;
  next_im();
}

/* 'DepthColor' button - toggle colormode */
void
colorToggle2(Widget w, XtPointer cdata, XtPointer rdata)
{
  if (!gcflag) {
    message("B & W only!");
    return;
  } else if (cmode & 1) {
    message("invalid");
    return;
  }
  cmode = cmode ^ 2;
  if (cmode & 2) dimension = 3;
  next_im();
}

/*'DensIntens' button - toggle density-based intensity mode.*/
void
togDensIntens(Widget w, XtPointer cdata, XtPointer rdata)
{
  panel_off();
  if (diFlag == 0) {
    if (!isLargeCmap) {
      free_colors();
      isLargeCmap = 1;
      init_colors(cmapName);
    }
    diFlag = diNew = 1;
    /* currently not allowed in animation mode */
    if (animflag) {
      animpoints = 0, animflag = 0;
      message("Animate off");
    }
    /* set up denspoints */
    XtVaSetValues(ss_in,XtNlabel,"Enter #/image pts:",XtNvalue,"1000000",NULL);
    dpopstate = 5;
    XtPopup(dpopup,XtGrabNone);
  } else {
    diFlag = 0;
    denspoints = 0;
    panel_on();
    next_im();
  }
}

/* 'ChangeCmap' button - switch to a randomly selected colormap from the
   files in the data directory */
void
change_cmap(Widget w, XtPointer cdata, XtPointer rdata)
{
  int x;
  halt_draw();
  if (nColorFiles > 0) {
    free_colors();
    x = (int)nrand(nColorFiles);
    strncpy(cmapName, colorFileNames[x], 31);
    message(cmapName);
    init_colors(colorFileNames[x]);
  }
  resume_draw();
}

/*'DiffEq' button - toggle diffEq mode.*/
void
diffEqToggle(Widget w, XtPointer cdata, XtPointer rdata)
{
  scale_max = pow(10,4+diffEqFlag);
  diffEqFlag = 1 - diffEqFlag;
  next_im();
}

/* 'E.vals' button - print eigenvalues to terminal, when d=2 only, using
   quadratic formula (A-lambdaI = 0).  Adding for 3-d would be more complex. */
void
printEvals(Widget w, XtPointer cdata, XtPointer rdata)
{
  int		i;
  float		nb,d,aa;  /*negative b, discriminant, and 2a*/

  if (dimension == 2) {
    printf("\nEigenvalues are:\n#\t     l1\t\t\t     l2\n-\t    ----\t\t    ----\n");
    for (i=0;i<n_trans;i++) {
      nb = a[i][0] + a[i][4] - 2.0;
      d = (nb * nb - 4 * ((a[i][0]-1)*(a[i][4]-1) - a[i][1]*a[i][3]));
      aa = 2.0;
      printf("%d",i+1);
      if (d > 0)  /*real case, just calc and print*/
	printf("\t%f\t\t%f\n",(nb+sqrt(d))/aa,(nb-sqrt(d))/aa);
      else {
	d = -d;
	printf("\t%.4f + %.4fi\t%.4f - %.4fi\n",nb,sqrt(d),nb,sqrt(d));
	}
      }
    }
}

/* 'Slideshow' button - toggle slides mode */
void
slideshow_toggle(Widget w, XtPointer cdata, XtPointer rdata)
{
  if (slidepoints) {
    slidepoints = 0;
    message("Slides off");
  }
  else {
    panel_off();
    XtVaSetValues(ss_in,XtNlabel,"Enter #/image pts:", XtNvalue,"500000",NULL);
    dpopstate = 1;
    XtPopup(dpopup,XtGrabNone);
  }
}

/* 'Animate' button - toggle animation mode */
void
animation_toggle(Widget w, XtPointer cdata, XtPointer rdata)
{
  if (diFlag) {
    message("Not in D.I.");
  }
  else if (animflag) {
    halt_draw();
    animpoints = 0, animflag = 0;
    message("Animate off");
    resume_draw();
  }
  else {
    panel_off();
    XtVaSetValues(ss_in,XtNlabel,"Enter #/anim pts:",XtNvalue,"20000",NULL);
    dpopstate = 3;
    XtPopup(dpopup,XtGrabNone);
  }
}

/* 'Rescale button' - force rescale */
void
rescale(Widget w, XtPointer cdata, XtPointer rdata)
{
  panel_off();
  scale(SCALE_MAX);
  XClearWindow(display,canvas);	/*clear display*/
  plottedpoints = 0;
  panel_on();
}

/*'Quit' button - shut it all down.*/
void
shut_down(Widget w, XtPointer cdata, XtPointer rdata)
{
  halt_draw();
  XtDestroyWidget(pwidget);
  free_colors();
  exit(0);
}


/**********************************************************************/
/* Edit window utilities */

/* set_ttype(): a toggle activation will call this twice - once for
  button switched off, and once for button switched on. Therefore this
  function checks if it was called because of an on-switching - if so,
  it sets variable to new value; if not, will soon be called again, so
  it does nothing.  NOTE: variable found is correctly passed as a
  pointer to a pointer to int to XtVaGetValues.*/
void
set_ttype(Widget w, int *switched, XtPointer rdata)
{
  int		*found;		/*to hold radiodata value of calling toggle*/
  Boolean	is_on;		/*to hold on/off state of calling toggle*/

  XtVaGetValues(w,XtNradioData,&found,XtNstate,&is_on,NULL);
  if (is_on)
    ttype = *found + 1;
}

/* set_dim() is similar to set_ttype().*/
void
set_dim(Widget w, int *switched, XtPointer rdata)
{
  int	*found;
  Boolean	is_on;

  XtVaGetValues(w,XtNradioData,&found,XtNstate,&is_on,NULL);
  if (is_on)
    dimension = *found + 2;
}

/* close_edit() closes up shop for the edit window and takes it down.*/
void
close_edit(Widget w, XtPointer cdata, XtPointer rdata)
{
  /* collect variables from dialog input */
  logtn = dialog_get_int(e_in[0]);	/*e_in[0] = logtn*/
  if (logtn > LTRANS_MAX)
    logtn = LTRANS_MAX, dialog_set_int(e_in[0],LTRANS_MAX);

  loghtn = dialog_get_int(e_in[1]);	/*e_in[1] = loghtn*/
  if ((ttype != 6) || (loghtn >= logtn))
    loghtn = 0, dialog_set_int(e_in[1], 0);

  n_mods = dialog_get_int(e_in[2]);	/* e_in[2] = n_mods */
  if (n_mods > MOD_MAX)
    n_mods = MOD_MAX, dialog_set_int(e_in[2], MOD_MAX);

  animstep = dialog_get_float(e_in[3]);	/* e_in[3] = animstep */
  if ((animstep <= 0) || (animstep > 1.0))
    animstep = 0.05, dialog_set_float(e_in[3], 0.05);

  animpersist = dialog_get_int(e_in[4]); /*e_in[4] = animpersist*/

  seed = dialog_get_int(e_in[5]);	/*e_in[5] = seed*/

  /*take down window and restore normal situation*/
  XtPopdown(epopup);
  panel_on();

  next_im();	/*Start a new picture w/newly defined parameters.*/
}


/**********************************************************************/
/* other popup utilities */

/* dpopset() sets a variable for the popup dialog (put up by button press) */
void
dpopset(Widget w, XtPointer cdata, XtPointer rdata)
                           
{
  switch (dpopstate) {
  case 1:	/* slides on */
    slidepoints = dialog_get_int(ss_in);
    if ((slidepoints < 1000) || (slidepoints > 10000000))
      slidepoints = 1000000;
    break;
  case 2:	/* set zmag */
    zmag = dialog_get_float(ss_in);
    break;
  case 3:	/* animation on */
    animpoints = dialog_get_int(ss_in);
    if ((animpoints < 1000) || (animpoints > 10000000))
      animpoints = 100000;
    init_mods();
    animflag = 1;
    message("animating");
    break;
  case 4:
    animstep = dialog_get_float(ss_in);
    if ((animstep < 0) || (animstep > 1))
      animstep = 0.25;
    break;
  case 5:	/* density intensity: use denspoints */
    denspoints = dialog_get_int(ss_in);
    if ((denspoints < 1000000) || (denspoints > 100000000))
      denspoints = 10000000;
    break;
  }

  XtPopdown(dpopup);
  panel_on();

}

/* trep_bpress() gets called when a button is pressed in a transrep subwindow;
   sets tmodn and calls add_rand, add_rot, or edit_rep depending on button.*/
void
trep_bpress(Widget w, int *cdata, XEvent *rep, Boolean cont_disp)
{
  tmodn = *cdata;

  if (tmodn < n_trans) {	/*error check to make sure valid rep window*/
    switch (rep->xbutton.button) {
    case 1:
      add_a();
      break;
    case 2:
      edit_rep();
      break;
    case 3:
      add_r();
      break;
    default:
      break;
    }
  }
  else tmodn = 0;

}

/* edit_rep() allows user to specify transformation #tmodn graphically */
void
edit_rep()
{
  panel_off();
  message("Select T(0,1)");

  /* Clear window and plot cross. */
  XClearWindow(display,trepWin[tmodn]);
  XDrawLine(display,trepWin[tmodn],cgc[tmodn],rhsize2-15,rvsize2,rhsize2+15,
	    rvsize2);
  XDrawLine(display,trepWin[tmodn],cgc[tmodn],rhsize2,rvsize2-15,rhsize2,
	    rvsize2+15);

  /*Similarly to zoom functions (see below), start function to detect
  button-presses.*/
  XtAddEventHandler(trepWidg[tmodn],ButtonPressMask,False,
		    (XtEventHandler) edit_rep_aux, 0);

}
