/*
  rakarrack - a guitar effects software

 Compressor.C  -  Compressor Effect
 Based on artscompressor.cc by Matthias Kretz <kretz@kde.org>
 Stefan Westerfeld <stefan@space.twc.de> 
 
  Copyright (C) 2008 Daniel Vidal & Josep Andreu
  Author: Daniel Vidal & Josep Andreu

 This program is free software; you can redistribute it and/or modify
 it under the terms of version 2 of the GNU General Public License
 as published by the Free Software Foundation.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License (version 2) for more details.

 You should have received a copy of the GNU General Public License
 (version2)  along with this program; if not, write to the Free Software
 Foundation,
 Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

*/

#include <math.h>
#include "Compressor.h"


Compressor::Compressor (REALTYPE * efxoutl_, REALTYPE * efxoutr_)
{
  efxoutl = efxoutl_;
  efxoutr = efxoutr_;


  volume = 0.0;
  volume_db = 0.0;
  tthreshold = -10;
  tratio = 2;
  toutput = 0;
  tatt = 20;
  trel = 120;
  a_out = 1;
  knee = 10;
  gain = 1.0;
  gain_old = 1.0;
}

Compressor::~Compressor ()
{


}


void
Compressor::cleanup ()
{

  gain = 1.0;
  gain_old = 1.0;

}


void
Compressor::Compressor_Change (int np, int value)
{

  switch (np)
    {

    case 1:
      tthreshold = value;
      threshold = dB2rap (tthreshold);
      knee_min = dB2rap (tthreshold - knee);
      knee_max = dB2rap (tthreshold + knee);
      break;
    case 2:
      tratio = value;
      ratio = 1.0 / value;
      break;
    case 3:
      toutput = value;
      if (a_out)
	output = 1.0;
      else
	output = dB2rap (toutput);
      break;
    case 4:
      tatt = value;
      att = LN2 / fmaxf (value / 1000.0 * SAMPLE_RATE, LN2);
      break;
    case 5:
      trel = value;
      rel = LN2 / fmaxf (value / 1000.0 * SAMPLE_RATE, LN2);
      break;
    case 6:
      a_out = value;
      if (a_out)
	output = 1.0;
      else
	output = dB2rap (toutput);
      break;

    }


}

int
Compressor::getpar (int np)
{

  switch (np)

    {

    case 1:
      return (tthreshold);
      break;
    case 2:
      return (tratio);
      break;
    case 3:
      return (toutput);
      break;
    case 4:
      return (tatt);
      break;
    case 5:
      return (trel);
      break;
    case 6:
      return (a_out);
      break;
    }

  return (0);

}


void
Compressor::Compressor_Change_Preset (int npreset)
{

  const int PRESET_SIZE = 6;
  const int NUM_PRESETS = 3;
  int presets[NUM_PRESETS][PRESET_SIZE] = {
    //2:1
    {-10, 2, 2, 20, 120, 0},
    //4:1
    {-10, 4, 4, 20, 120, 0},
    //8:1
    {-10, 8, 8, 20, 120, 0}
  };

  if (npreset >= NUM_PRESETS)
    npreset = NUM_PRESETS - 1;
  for (int n = 0; n < PRESET_SIZE; n++)
    Compressor_Change (n + 1, presets[npreset][n]);


}



void
Compressor::out (float *efxoutl, float *efxoutr)
{

  int i;

  for (i = 0; i < PERIOD; i++)
    {

      float delta = fmaxf (fabsf (efxoutl[i]), fabsf (efxoutr[i])) - volume;


      if (delta > 0.0)
	volume += att * delta;
      else
	volume += rel * delta;


      if (volume <= knee_min)
	{
	  gain = output;

	}
      else if (volume < knee_max)
	{
	  gain =
	    output +
	    ((volume - knee_min) * compress () / (knee_max - knee_min));
	}
      else
	{
	  gain = compress () + output;
	}

      float gain_t = .4 * gain + .6 * gain_old;

      efxoutl[i] *= gain_t;
      efxoutr[i] *= gain_t;
      gain_old = gain;

    }

}

float
Compressor::compress ()
{

  volume_db = rap2dB (volume);

  return (dB2rap ((volume_db - tthreshold) * ratio + tthreshold) / volume);

}
