/***************************************************************************
 *   Copyright (C) 2008-2009 by fra74                                           *
 *   francesco.b74@gmail.com                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "schedastazione.h"
#include "qviaggiatreno.h"
#include "download_viaggiatreno.h"
#include "utils.h"
//
SchedaStazione::SchedaStazione(QViaggiaTreno* parent, const QString& nome, const unsigned int intervalloStandard)
	: SchedaQViaggiaTreno(parent, tsStazione, intervalloStandard)
{
	m_stazione = nome;
	m_codice = "";
	m_stato = statoNuovaScheda;
	m_documentoDom = 0L;
	m_idTabella = m_idNomeSbagliato = m_idNomeAmbiguo = -1;

	//imposta il parser
	m_parser = new ParserStazioneViaggiaTreno(this);
	
	//crea il widget della scheda
	m_idTabella = addWidget(creaWidgetStazione());
	setCurrentIndex(m_idTabella);

	//connessioni
	connect(this, SIGNAL(statoCambiato(quint32)), parent, SLOT(aggiornaStatoScheda(quint32)));
	connect(this, SIGNAL(nomeSchedaCambiato(quint32)), parent, SLOT(aggiornaNomeScheda(quint32)));
	connect(this, SIGNAL(apriSchedaStazione(const QString&)), parent, SLOT(nuovaStazione(const QString&)));
    connect(this, SIGNAL(apriSchedaTreno(const QString&, const QString&)), parent, SLOT(nuovoTreno(const QString&, const QString&)));
	connect(this, SIGNAL(messaggioStatus(const QString&)), parent, SLOT(mostraMessaggioStatusBar(const QString&)));
	connect(this, SIGNAL(downloadStazione(quint32,QString)), qViaggiaTreno()->downloadViaggiaTreno(), SLOT(downloadStazione(quint32,QString)));
	connect(this, SIGNAL(downloadStazioneConCodice(quint32,QString)), qViaggiaTreno()->downloadViaggiaTreno(), SLOT(downloadStazioneCodice(quint32,QString)));

}

//restituisce il nome della stazione
//seguendo la convenzione di ViaggiaTreno il nome della stazione viene restituito in maiuscolo
QString SchedaStazione::nomeStazione() const
{
	return m_stazione.toUpper();
}

//restituisce una stringa con il "titolo" di questa scheda, che viene ad esempio usato come etichetta della TabBar
// se titoloItem == true restituisce un titolo per l'item dell'albero
//seguendo la convenzione di Viaggiatreno il nome della stazione verrà scritto tutto in maiuscolo
QString SchedaStazione::titolo( bool titoloBreve ) const
{
	if (titoloBreve)
		return nomeStazione();
	
	
	return QString::fromUtf8("Stazione di %1").arg(nomeStazione());
}



//questo slot viene richiamato quando scatta il timer
void SchedaStazione::aggiorna()
{
	cambiaStato(statoInAggiornamento);

	//cancella dallo StackedWidget i widget creati per gestire gli errori
	if (m_idNomeSbagliato != -1)
	{
		QWidget *widgetDaRimuovere = widget(m_idNomeSbagliato);
		widgetDaRimuovere->deleteLater();
		m_idNomeSbagliato = -1;
	}

	if (m_idNomeAmbiguo != -1)
	{
		QWidget *widgetDaRimuovere = widget(m_idNomeAmbiguo);
		widgetDaRimuovere->deleteLater();
		m_idNomeAmbiguo = -1;
	}


	if (m_codice.isEmpty())
		emit (downloadStazione(idScheda(), nomeStazione()));
	else
		emit (downloadStazioneConCodice(idScheda(), codiceStazione()));

}

//questo slot viene richiamato quando è finito il download della risposta dal sito di viaggiatreno
void SchedaStazione::downloadFinito(const QString& rispostaVT)
{

	//imposta nel parser la risposta di ViaggiaTreno
	m_parser->impostaRispostaVT(rispostaVT);


	//controlla se è stata trovata la stazione e in caso contrario mostra finestra di errore
	if (m_parser->stazioneNonTrovata())
	{
		ferma();
		m_idNomeSbagliato = addWidget(creaWidgetErroreNome());
		setCurrentIndex(m_idNomeSbagliato);
		cambiaStato(statoErrore);
		return;
	}


	//controlla se ViaggiaTreno ha segnalato che il nome è ambiguo
	//in tale caso prepara un widget per la selezione
	if (m_parser->nomeStazioneAmbiguo())
	{
		ferma();
		m_idNomeAmbiguo = addWidget(new WidgetDisambiguaNome(this, m_parser->listaCodiciStazioni(rispostaVT)));
		setCurrentIndex(m_idNomeAmbiguo);
		cambiaStato(statoErrore);
		return;
	}

    if (!m_parser->analizza(m_arrivi, m_partenze))
	{
		//errore nel parsing
		cambiaStato(statoErrore);
		QMessageBox msgBox;
		msgBox.setIcon(QMessageBox::Warning);
		msgBox.setText(QString::fromUtf8("Errore durante l'analisi della risposta di ViaggiaTreno. Se il problema persiste contattare l'autore"));
        //msgBox.setDetailedText(QString::fromUtf8("Ricevuto messaggio di errore %1 alla riga %2, colonna %3.\nTesto:\n %4").arg(m_parser->errore())
        //					   .arg(m_parser->rigaErrore()).arg(m_parser->colErrore()).arg(m_parser->rispostaVT()));
		msgBox.exec();
		return;
	}

	m_stazione = m_parser->stazione();
	emit(nomeSchedaCambiato(idScheda()));
	m_labelTitolo->setText(titolo());

    riempiTabelle(m_arrivi, m_partenze);

	m_ultimoAgg = QDateTime::currentDateTime();
	m_labelAggiornamento->setText(QString::fromUtf8("Ultimo aggiornamento: %1").arg(m_ultimoAgg.toString(Qt::DefaultLocaleShortDate)));
	cambiaStato(statoMonitoraggioAttivo);
	emit messaggioStatus(QString::fromUtf8("Aggiornati dati stazione di %1").arg(titolo(true)));
}

QWidget* SchedaStazione::creaWidgetStazione()
{
	QWidget *widget = new QWidget;
	//layout principale
	QVBoxLayout *vbox = new QVBoxLayout;

	//costruisce l'etichetta per il titolo	
	m_labelTitolo = new QLabel(widget);
	//l'etichetta sarà centrata
	m_labelTitolo->setAlignment(Qt::AlignCenter);
	m_labelTitolo->setText(titolo());
	//imposta font con dimensioni del 50% maggiori di quella standard e in grassetto
	QFont font = m_labelTitolo->font();
	font.setBold(true);
        font.setPointSize(font.pointSize()*1.5);
	m_labelTitolo->setFont(font);
	//aggiunge l'etichetta al layout
	vbox->addWidget(m_labelTitolo, 0);
	
	//costruisce l'etichetta con ora/data di aggiornamento
	m_labelAggiornamento = new QLabel(widget);
	//l'etichetta sarà centrata
	m_labelAggiornamento->setAlignment(Qt::AlignCenter);
	m_labelAggiornamento->setText(QString::fromUtf8("Ultimo aggiornamento: aggiornamento in corso"));
	vbox->addWidget(m_labelAggiornamento, 0);
	
	QSplitter *splitter = new QSplitter(Qt::Vertical, widget);

	QWidget *widgetArrivi = new QWidget(widget);
	QVBoxLayout *vboxArrivi = new QVBoxLayout;
	QLabel *labelArrivi = new QLabel(widgetArrivi);
	labelArrivi->setAlignment(Qt::AlignCenter);
	labelArrivi->setText(QString::fromUtf8("<b>Arrivi</b>"));
	vboxArrivi->addWidget(labelArrivi);
	m_tabellaArrivi = new TabellaStazione(widgetArrivi);
	QStringList etichetteArrivi;
	m_tabellaArrivi->setColumnCount(8);
	etichetteArrivi<<QString::fromUtf8("Categoria")<<QString::fromUtf8("Numero")<<QString::fromUtf8("Origine")
		<<QString::fromUtf8("Bin. Programmato")<<QString::fromUtf8("Bin. reale")<<QString::fromUtf8("Orario teorico")<<QString::fromUtf8("Ritardo")<<"";
	m_tabellaArrivi->setHorizontalHeaderLabels(etichetteArrivi);
	

	vboxArrivi->addWidget(m_tabellaArrivi, 1);
	widgetArrivi->setLayout(vboxArrivi);
	splitter->addWidget(widgetArrivi);
	
	QWidget *widgetPartenze = new QWidget(widget);
	QVBoxLayout *vboxPartenze = new QVBoxLayout;
	QLabel *labelPartenze = new QLabel(widgetArrivi);
	labelPartenze->setAlignment(Qt::AlignCenter);
	labelPartenze->setText(QString::fromUtf8("<b>Partenze</b>"));
	vboxPartenze->addWidget(labelPartenze);
	m_tabellaPartenze = new TabellaStazione(widgetPartenze);
	QStringList etichettePartenze;
	m_tabellaPartenze->setColumnCount(8);
	etichettePartenze<<QString::fromUtf8("Categoria")<<QString::fromUtf8("Numero")<<QString::fromUtf8("Destinazione")
		<<QString::fromUtf8("Bin. Programmato")<<QString::fromUtf8("Bin. reale")<<QString::fromUtf8("Orario teorico")<<QString::fromUtf8("Ritardo")<<"";
	m_tabellaPartenze->setHorizontalHeaderLabels(etichettePartenze);
    /*
    QPalette paletta = QApplication::palette();
    //paletta.setColor(QPalette::Base, QColor(153,255,255));
    //paletta.setColor(QPalette::AlternateBase, QColor(153,255,153));
    m_tabellaPartenze->setPalette(paletta);
    */

	vboxPartenze->addWidget(m_tabellaPartenze, 1);
	widgetPartenze->setLayout(vboxPartenze);
	splitter->addWidget(widgetPartenze);
	
	vbox->addWidget(splitter, 1);
	widget->setLayout(vbox);

	//connect(m_tabellaArrivi, SIGNAL(itemActivated(QTableWidgetItem *)), this, SLOT(itemAttivato(QTableWidgetItem*)));
	//connect(m_tabellaPartenze, SIGNAL(itemActivated(QTableWidgetItem *)), this, SLOT(itemAttivato(QTableWidgetItem*)));

	connect(m_tabellaArrivi, SIGNAL(itemClicked(QTableWidgetItem *)), this, SLOT(itemAttivato(QTableWidgetItem*)));
	connect(m_tabellaPartenze, SIGNAL(itemClicked(QTableWidgetItem *)), this, SLOT(itemAttivato(QTableWidgetItem*)));

	return widget;
}

QWidget* SchedaStazione::creaWidgetErroreNome()
{
	QWidget* widget = new QWidget(this);
	
	QVBoxLayout * vbox = new QVBoxLayout;
	widget->setLayout(vbox);
	QLabel *labelErrore = new QLabel(this);
	labelErrore->setWordWrap(true);
	labelErrore->setText(QString::fromUtf8("Il nome della stazione ('<b>%1</b>') non è stato riconosciuto"
		" da ViaggiaTreno. Premete il pulsante sottostante per modificare il nome della stazione, "
		"oppure il pulsante di chiusura se desiderate semplicemente chiudere questa scheda").arg(nomeStazione()));
	QPushButton *btnCambiaNome = new QPushButton(QString::fromUtf8("Cambia nome"), this);
	connect(btnCambiaNome, SIGNAL(clicked()), this, SLOT(modificaNomeStazione()));
	vbox->addWidget(labelErrore);
	vbox->addWidget(btnCambiaNome, 0, Qt::AlignCenter);
	vbox->addStretch();
	
	return widget;
}

void SchedaStazione::modificaNomeStazione()
{
	bool ok;
	QString nuovoNome = QInputDialog::getText(this, QString::fromUtf8("Modifica nome stazione"), QString::fromUtf8("Inserire il nuovo nome della stazione:"),
		QLineEdit::Normal, nomeStazione(), &ok);
		
	if (ok & !nuovoNome.isEmpty())
	{
		m_stazione = nuovoNome;
		m_codice = "";
		m_labelTitolo->setText(titolo());
		//assicurarsi che venga visualizzato il widget della stazione e non il widget di errore
		setCurrentIndex(m_idTabella);
		emit(nomeSchedaCambiato(idScheda()));
		aggiorna();
		avvia();
	}
}


void SchedaStazione::riempiTabelle(QList<StazioneVT::DatiTreno> arrivi, QList<StazioneVT::DatiTreno> partenze)
{

	int riga = -1;
	QVector<int> larghezzaTabArrivi(StazioneVT::colUltima+1), larghezzaTabPartenze(StazioneVT::colUltima+1);
	
	//pulisci le tabelle
	m_tabellaArrivi->clearContents();
	m_tabellaPartenze->clearContents();
	
	//imposta il numero corretto di righe
	m_tabellaArrivi->setRowCount(arrivi.count());
	m_tabellaPartenze->setRowCount(partenze.count());
	
	//inizializza il vettore con la larghezza delle varie colonne
	//con i valori della larghezza degli header
	for (int i = 0; i <= StazioneVT::colUltima ; i++)
		larghezzaTabPartenze[i] = m_tabellaPartenze->horizontalHeader()->sectionSizeHint(i);


	//scorre la lista dei treni in partenza
	QListIterator<StazioneVT::DatiTreno> it(partenze);
	while (it.hasNext())
	{
		QTableWidgetItem* item;
        QColor colore;

        riga++;

		StazioneVT::DatiTreno treno = it.next();
        colore = treno.coloreRitardo();

		// imposta le varie colonne nella tabella, elminando contemporaneamente gli spazi bianchi in eccesso
		//le funzioni per ridimensionare la larghezza delle colonne in QTableWidget
		//ad esempio ResizeColumnsToContent o quelle di QHeaderView hanno effetto
		//solo sulla parte visible della tabella.
		//Per ridimensionare correttamente la colonna è necessario procedere "a mano"
		//Di conseguenza mentre si riempie la tabella teniamo conto della larghezza delle varie colonne,
		//in particolare del massimo valore di larghezza per ogni singola colonna
		item = new QTableWidgetItem(treno.categoria().simplified());
        item->setForeground(colore);
        m_tabellaPartenze->setItem(riga, StazioneVT::colCategoria, item);
		larghezzaTabPartenze[StazioneVT::colCategoria] = qMax(larghezzaTabPartenze[StazioneVT::colCategoria], larghezzaItem(item));
		//nel caso delle colonne numero treno e stazione crea un item con un font sottolineato e con il colore dei link, a simulare la presenza
		//di un link, per far capire che cliccando si apre la scheda corrispondente
		item = new QTableWidgetItem(treno.numero().simplified());
		itemSottolineato(item);
        //aggiungi come attributo "nascosto" il codice della stazione di origine
        item->setData(Qt::UserRole, treno.codiceOrigine());
		m_tabellaPartenze->setItem(riga, StazioneVT::colNumero, item);
		larghezzaTabPartenze[StazioneVT::colNumero] = qMax(larghezzaTabPartenze[StazioneVT::colNumero], larghezzaItem(item));
		item = new QTableWidgetItem(treno.stazione().simplified());
		itemSottolineato(item);
        m_tabellaPartenze->setItem(riga, StazioneVT::colStazione, item);
        larghezzaTabPartenze[StazioneVT::colStazione] = qMax(larghezzaTabPartenze[StazioneVT::colStazione], larghezzaItem(item));
		item = new QTableWidgetItem(treno.binarioProgrammato().simplified());
        item->setForeground(colore);
		//allinea il testo centrandolo nella cologna
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaPartenze->setItem(riga, StazioneVT::colBinProgrammato, item);
		larghezzaTabPartenze[StazioneVT::colBinProgrammato] = qMax(larghezzaTabPartenze[StazioneVT::colBinProgrammato], larghezzaItem(item));
		item = new QTableWidgetItem(treno.binarioReale().simplified());
        item->setForeground(colore);
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaPartenze->setItem(riga, StazioneVT::colBinReale, item);
		larghezzaTabPartenze[StazioneVT::colBinReale] = qMax(larghezzaTabPartenze[StazioneVT::colBinReale], larghezzaItem(item));
		item = new QTableWidgetItem(treno.orario().simplified());
        item->setForeground(colore);
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaPartenze->setItem(riga, StazioneVT::colOrario, item);
		larghezzaTabPartenze[StazioneVT::colOrario] = qMax(larghezzaTabPartenze[StazioneVT::colOrario], larghezzaItem(item));
		item = new QTableWidgetItem(treno.ritardo().simplified());
        item->setForeground(colore);
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaPartenze->setItem(riga, StazioneVT::colRitardo, item);
		larghezzaTabPartenze[StazioneVT::colRitardo] = qMax(larghezzaTabPartenze[StazioneVT::colRitardo], larghezzaItem(item));
	}

	//imposta la larghezza delle colonne della tabella con i valori appena memorizzati
	//aggiungendo un margine di 10 punti
	for (int i = 0; i <= StazioneVT::colUltima; i++)
		m_tabellaPartenze->setColumnWidth(i, larghezzaTabPartenze[i]+10);

	//scorre la lista dei treni in arrivo
	it = QListIterator<StazioneVT::DatiTreno>(arrivi);
	riga = -1;

	for (int i = 0; i <= StazioneVT::colUltima ; i++)
		larghezzaTabArrivi[i] = m_tabellaArrivi->horizontalHeader()->sectionSizeHint(i);

	while (it.hasNext())
	{
		QTableWidgetItem* item;

		riga++;
		StazioneVT::DatiTreno treno = it.next();

        QColor colore = treno.coloreRitardo();


		item = new QTableWidgetItem(treno.categoria().simplified());
        item->setForeground(colore);
		m_tabellaArrivi->setItem(riga, StazioneVT::colCategoria, item);
		larghezzaTabArrivi[StazioneVT::colCategoria] = qMax(larghezzaTabArrivi[StazioneVT::colCategoria], larghezzaItem(item));
		item = new QTableWidgetItem(treno.numero().simplified());
		itemSottolineato(item);
        //aggiungi come attributo "nascosto" il codice della stazione di origine
        item->setData(Qt::UserRole, treno.codiceOrigine());
		m_tabellaArrivi->setItem(riga, StazioneVT::colNumero, item);
		larghezzaTabArrivi[StazioneVT::colNumero] = qMax(larghezzaTabArrivi[StazioneVT::colNumero], larghezzaItem(item));
		item = new QTableWidgetItem(treno.stazione().simplified());
		itemSottolineato(item);
		m_tabellaArrivi->setItem(riga, StazioneVT::colStazione, item);
		larghezzaTabArrivi[StazioneVT::colStazione] = qMax(larghezzaTabArrivi[StazioneVT::colStazione], larghezzaItem(item));
		item = new QTableWidgetItem(treno.binarioProgrammato().simplified());
        item->setForeground(colore);
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaArrivi->setItem(riga, StazioneVT::colBinProgrammato, item);
		larghezzaTabArrivi[StazioneVT::colBinProgrammato] = qMax(larghezzaTabArrivi[StazioneVT::colBinProgrammato], larghezzaItem(item));
		item = new QTableWidgetItem(treno.binarioReale().simplified());
        item->setForeground(colore);
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaArrivi->setItem(riga, StazioneVT::colBinReale, item);
		larghezzaTabArrivi[StazioneVT::colBinReale] = qMax(larghezzaTabArrivi[StazioneVT::colBinReale], larghezzaItem(item));
		item = new QTableWidgetItem(treno.orario().simplified());
        item->setForeground(colore);
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaArrivi->setItem(riga, StazioneVT::colOrario, item);
		larghezzaTabArrivi[StazioneVT::colOrario] = qMax(larghezzaTabArrivi[StazioneVT::colOrario], larghezzaItem(item));
		item = new QTableWidgetItem(treno.ritardo().simplified());
        item->setForeground(colore);
		item->setTextAlignment(Qt::AlignCenter);
		m_tabellaArrivi->setItem(riga, StazioneVT::colRitardo, item);
		larghezzaTabArrivi[StazioneVT::colRitardo] = qMax(larghezzaTabArrivi[StazioneVT::colRitardo], larghezzaItem(item));
	}

	for (int i = 0; i <= StazioneVT::colUltima; i++)
		m_tabellaArrivi->setColumnWidth(i, larghezzaTabArrivi[i]+10);
	
}

void SchedaStazione::itemAttivato(QTableWidgetItem* item)
{
	int col = item->column();
	
	if (col == StazioneVT::colNumero)
    {
        emit(apriSchedaTreno(item->text(), item->data(Qt::UserRole).toString()));
    }
	
	if (col == StazioneVT::colStazione)
		emit(apriSchedaStazione(item->text()));
	
}

//slot
//cambia il codice della stazione e riaggiorna la scheda
void SchedaStazione::cambiaCodiceStazione(const QString& nuovoCodice)
{
	m_codice = nuovoCodice;
	setCurrentIndex(m_idTabella);
	aggiorna();
	avvia();	
}


//Salva i dati necessari per poter ripristinare la scheda alla prossima apertura
void SchedaStazione::salvaScheda(QSettings& settings)
{
	settings.setValue("tipo scheda", "stazione");
	settings.setValue("intervallo", intervallo());
	settings.setValue("nome stazione", nomeStazione());
	settings.setValue("codice stazione", codiceStazione());
}
