/***************************************************************************
 *   Copyright (C) 2008-2009 by fra74                                           *
 *   francesco.b74@gmail.com                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QtNetwork>
#include "schedatreno.h"
#include "qviaggiatreno.h"

SchedaTreno::SchedaTreno(QViaggiaTreno *parent, const QString &numero, const unsigned int intervalloStandard) :
		SchedaViaggiaTreno(parent, tsTreno, intervalloStandard)
{
	m_numero = numero;
	m_idTabelle = m_idNumeroSbagliato = -1;
	m_riepilogoScaricato = false;

	//crea il widget con le tabelle
	m_widget = new WidgetDatiTreno(this);
	m_idTabelle = addWidget(m_widget);
	setCurrentIndex(m_idTabelle);

	m_widget->inizializza();
	m_widget->impostaNumeroTreno(numero);

	//connessioni
	connect(this, SIGNAL(statoCambiato(quint32)), parent, SLOT(aggiornaStatoScheda(quint32)));
	connect(this, SIGNAL(nomeSchedaCambiato(quint32)), parent, SLOT(aggiornaNomeScheda(quint32)));
	connect(this, SIGNAL(apriSchedaStazione(const QString&)), parent, SLOT(nuovaStazione(const QString&)));
	connect(this, SIGNAL(messaggioStatus(const QString&)), parent, SLOT(mostraMessaggioStatusBar(const QString&)));
}

//restituisce una stringa contente il numero del treno monitorato
QString SchedaTreno::numero() const
{
	return m_numero;
}

//restituisce il titolo della scheda
//nel caso di una scheda per il monitoraggio di un treno restituisce semplicemente il numero del treno
QString SchedaTreno::titolo( bool /*titoloItem*/) const
{
	return m_numero;
}


//questo slot viene richiamato quando è finito il download della risposta dal sito di viaggiatreno
void SchedaTreno::downloadFinito(QNetworkReply *risposta)
{
	QString rispostaVT = QString::fromUtf8(risposta->readAll());
	//rimuove tutti gli spazi bianchi
	rispostaVT = correggiOutputVT(rispostaVT);

	//iinizializza solo se è stato scaricato il riepilogo
	if (!m_riepilogoScaricato)
	{
		m_widget->inizializza();
		m_orarioTreno.inizializza();
	}
	//cerca nel testo la stringa "non valido" perché VT quando trova un numero non valido visualizza il messaggio
	//"numero treno non valido"
	//se viene trovata visualizza il widget con il messaggio di errore e cambia lo stato
	if (rispostaVT.contains("non valido", Qt::CaseInsensitive))
	{
		m_widget->impostaStato(TrenoNonPrevisto);
		m_ultimoAgg = QDateTime::currentDateTime();
		m_widget->impostaAggiornamento(m_ultimoAgg.toString(Qt::DefaultLocaleShortDate));
		return;
	}

	//è già stato scaricato (e analizzato il riepilogo?)
	//in caso negativo analizza il riepilogo e richiedi lo scaricamento del
	if (!m_riepilogoScaricato)
	{
		analizzaRiepilogo(rispostaVT);
		m_riepilogoScaricato = true;
		risposta->deleteLater();

		if (m_stato == SchedaViaggiaTreno::statoErrore)
			//c'è stato un errore, esci senza scaricare i dettagli
		{
			risposta->deleteLater();
			return;
		}

		//scarica il dettaglio del treno
		QNetworkRequest request;
		QString dati;
		//costruisce i dati da inviare al server sottoforma di richiesta POST
		//dati = QString("dettaglio=visualizza&numeroTreno=%1&tipoRicerca=numero&lang=IT").arg(numero());
		request.setUrl(QUrl(QString("http://mobile.viaggiatreno.it/viaggiatreno/mobile/scheda?dettaglio=visualizza&numeroTreno=%1&tipoRicerca=numero&lang=IT").arg(numero())));
		/*request.setRawHeader("Content-type", "application/x-www-form-urlencoded");
		networkAM()->post(request, dati.toUtf8());
		*/
		networkAM()->get(request);

		return;
	}

	analizzaDettagli(rispostaVT);
	m_riepilogoScaricato = false;

	m_ultimoAgg = QDateTime::currentDateTime();
	m_widget->aggiornaTreno(m_orarioTreno);
	cambiaStato(statoMonitoraggioAttivo);
	emit messaggioStatus(QString::fromUtf8("Aggiornati dati treno %1").arg(titolo(true)));
	m_widget->impostaAggiornamento(m_ultimoAgg.toString(Qt::DefaultLocaleShortDate));

	risposta->deleteLater();

}


//slot
//questo slot viene richiamato quando scatta il timer
void SchedaTreno::aggiorna()
{
	cambiaStato(statoInAggiornamento);

	QNetworkRequest request;
	QString dati;
	//costruisce i dati da inviare al server sottoforma di richiesta POST
	dati = QString("numeroTreno=%1&tipoRicerca=numero&lang=IT").arg(numero());
	request.setUrl(QUrl("http://mobile.viaggiatreno.it/viaggiatreno/mobile/numero"));
	request.setRawHeader("Content-type", "application/x-www-form-urlencoded");
	networkAM()->post(request, dati.toUtf8());

}



void SchedaTreno::analizzaDettagli(const QString& testoVT)
{
	QList<QDomNode> listaNodi;
	QString errore;
	QDomDocument documentoDOM;
	int riga, colonna;

	if (!documentoDOM.setContent(testoVT, &errore, &riga, &colonna))
	{
		//errore nel parsing
		cambiaStato(statoErrore);
		QMessageBox msgBox;
		msgBox.setIcon(QMessageBox::Warning);
		msgBox.setText(QString::fromUtf8("Errore durante l'analisi della risposta di ViaggiaTreno. Se il problema persiste contattare l'autore"));
		msgBox.setDetailedText(QString::fromUtf8("Ricevuto messaggio di errore %1 alla riga %2, colonna %3.\nTesto:\n %4").arg(errore).arg(riga).arg(colonna).arg(testoVT));
		msgBox.exec();
		return;
	}

	QDomElement body = documentoDOM.documentElement().firstChildElement("body");
	//ottiene la lista degli elementi div
	QDomNodeList elementiDiv = body.elementsByTagName("div");
	//i dati delle fermate si trovano dal secondo al penultimo elemento div
	//ma siccome abbiamo già scritto i dati per la stazione di partenza e di arrivo
	//ci interessano solo gli elementi dal terzo al terz'ultimo
	for (int i = 2; i < elementiDiv.count()-2; i++)
	{
		Treno::Fermata *fermata = new Treno::Fermata();
		QDomElement div = elementiDiv.at(i).toElement();
		//verifica se la fermata è già stata effettuata o no, sfruttando l'attributo class
		if (div.attribute("class") == "corpocentrale")
			fermata->impostaEffetuata(false);
		else
			fermata->impostaEffetuata(true);
		//verifica se per caso la fermata non sia stata soppressa
		if (div.text().contains("soppressa"))
			fermata->impostaSoppressa(true);
		//imposta il nome della fermata
		fermata->impostaNomeFermata(div.firstChildElement("h2").text());
		//imposta l'ora di arriva programmata ed effettiva
		fermata->impostaOraArrivoProgrammata(div.firstChildElement("p").firstChildElement("strong").text());
		if (fermata->effettuata())
			fermata->impostaOraArrivoReale(div.firstChildElement("p").nextSiblingElement("p").firstChildElement("strong").text());
		else
			fermata->impostaOraArrivoStimata(div.firstChildElement("p").nextSiblingElement("p").firstChildElement("strong").text());
		listaNodi.clear();
		for(QDomNode n = div.firstChild(); !n.isNull(); n = n.nextSibling())
		{
			QDomText t = n.toText();
			if (!t.isNull())
				listaNodi.append(n);

		}
		//se ci sono 4 elementi nella lista allora né binario previsto né reale sono scritti in grassetto
		if ( listaNodi.count() == 4)
		{
			fermata->impostaBinarioProgrammato(listaNodi.at(1).toText().data().simplified());
			fermata->impostaBinarioReale(listaNodi.at(3).toText().data().simplified());
		}

		//ce ne sono 3 allora le possibilità sono varie
		//se la fermata è stata effettuata allora il binario reale è in grassetto
		else
			if (fermata->effettuata())
			{
				fermata->impostaBinarioProgrammato(listaNodi.at(1).toText().data().simplified());
				fermata->impostaBinarioReale(div.firstChildElement("strong").text().simplified());
			}
		//oppure la fermata non è stata effettuata
		else
			//è indicato in grassetto il binario reale (evento raro ma succede)
			if (listaNodi.at(2).toText().data().contains("Binario"))
			{
				fermata->impostaBinarioProgrammato(listaNodi.at(1).toText().data().simplified());
				fermata->impostaBinarioReale(div.firstChildElement("strong").text().simplified());
			}
			//è indicato in grassetto il binario programmato
			else
			{
				fermata->impostaBinarioProgrammato(div.firstChildElement("strong").text().simplified());
				fermata->impostaBinarioReale(listaNodi.at(2).toText().data().simplified());
			}

	m_orarioTreno.aggiungiFermata(fermata);
	}

}

void SchedaTreno::analizzaRiepilogo(const QString& testoVT)
{
	QString errore, temp;
	QDomDocument documentoDOM;
	QList<QDomNode> listaNodi;
	int riga, colonna, idx;
	if (!documentoDOM.setContent(testoVT, &errore, &riga, &colonna))

	{
		//errore nel parsing
		cambiaStato(statoErrore);
		QMessageBox msgBox;
		msgBox.setIcon(QMessageBox::Warning);
		msgBox.setText(QString::fromUtf8("Errore durante l'analisi della risposta di ViaggiaTreno. Se il problema persiste contattare l'autore"));
		msgBox.setDetailedText(QString::fromUtf8("Ricevuto messaggio di errore %1 alla riga %2, colonna %3.\nTesto:\n %4").arg(errore).arg(riga).arg(colonna).arg(testoVT));
		msgBox.exec();
		return;
	}

	//inizia l'analisi dell'albero DOM
	m_orarioTreno.inizializza();

	//rintraccia il tag body
	QDomElement body = documentoDOM.documentElement().firstChildElement("body");
	//rintraccia il tag H1 che contiene categoria e numero treno
	temp = body.firstChildElement("h1").text();
	m_orarioTreno.impostaNumeroTreno(temp);

	//ottiene una lista di tutti gli elementi div
	QDomNodeList elementiDiv = body.elementsByTagName("div");

	QDomElement div;

	if (testoVT.contains("arrivato"))
	{
		m_orarioTreno.impostaStatoTreno(TrenoArrivato);
	}

	if (testoVT.contains("ancora partito"))
	{
		m_orarioTreno.impostaStatoTreno(TrenoNonPartito);
	}

	if (testoVT.contains("viaggia "))
	{
		m_orarioTreno.impostaStatoTreno(TrenoInViaggio);
	}

	//rintraccia l'elemento DIV con gli eventuali provvedimenti di limitazione/soppressione
	for (int i = 0; i < elementiDiv.count(); i++)
		if (elementiDiv.at(i).toElement().attribute("style").contains("B02D2D"))
			m_orarioTreno.impostaDato(Treno::dtProvvedimenti, elementiDiv.at(i).toElement().text());

	//rintraccia l'elemento DIV con i dati della stazione di partenza
	for (int i = 0; i < elementiDiv.count(); i++)
			if (elementiDiv.at(i).toElement().text().contains("Partenza"))
				div = elementiDiv.at(i).toElement();

	//estrai i dati sulla stazione di partenza dall'elemento appena ottenuto
	m_orarioTreno.impostaDato(Treno::dtPartenza, div.firstChildElement("h2").text());
	m_orarioTreno.impostaDato(Treno::dtOrarioPartenzaProgrammato, div.firstChildElement("p").firstChildElement("strong").text());
	m_orarioTreno.impostaDato(Treno::dtOrarioPartenzaReale, div.firstChildElement("p").nextSiblingElement("p")
							  .firstChildElement("strong").text());

	//ottiene la lista di tutti i nodi di tipo testo
	for(QDomNode n = div.firstChild(); !n.isNull(); n = n.nextSibling())
	{
		QDomText t = n.toText();
		if (!t.isNull())
			listaNodi.append(n);
	}

	m_orarioTreno.impostaDato(Treno::dtBinarioPartenzaProgrammato, listaNodi.at(1).toText().data());
	if (listaNodi.count() == 4)
		m_orarioTreno.impostaDato(Treno::dtBinarioPartenzaReale, listaNodi.at(3).toText().data());
	else
		m_orarioTreno.impostaDato(Treno::dtBinarioPartenzaReale, div.firstChildElement("strong").text());

	//rintraccia l'elemento DIV con i dati della stazione di arrivo
	for (int i = 0; i < elementiDiv.count(); i++)
		if (elementiDiv.at(i).toElement().text().contains("Arrivo"))
			div = elementiDiv.at(i).toElement();

	//estrai i dati sulla stazione di arrivo dall'elemento appena ottenuto
	m_orarioTreno.impostaDato(Treno::dtArrivo, div.firstChildElement("h2").text());
	m_orarioTreno.impostaDato(Treno::dtOrarioArrivoProgrammato, div.firstChildElement("p").firstChildElement("strong").text());
	m_orarioTreno.impostaDato(Treno::dtOrarioArrivo, div.firstChildElement("p").nextSiblingElement("p")
							  .firstChildElement("strong").text());

	//ottiene la lista di tutti i nodi di tipo testo
	listaNodi.clear();
	for(QDomNode n = div.firstChild(); !n.isNull(); n = n.nextSibling())
	{
		QDomText t = n.toText();
		if (!t.isNull())
			listaNodi.append(n);
	}
	m_orarioTreno.impostaDato(Treno::dtBinarioArrivoProgrammato, listaNodi.at(1).toText().data());
	if (listaNodi.count() == 4)
		m_orarioTreno.impostaDato(Treno::dtBinarioArrivoReale, listaNodi.at(3).toText().data());
	else
		m_orarioTreno.impostaDato(Treno::dtBinarioArrivoReale, div.firstChildElement("strong").text());


	//estra il ritardo attuale per un treno in viaggio
	if (testoVT.contains("viaggia "))
	{
		//rintraccia l'elemento DIV con il ritardo attuale
		idx = -1;
		for (int i = 0; i < elementiDiv.count(); i++)
			if (elementiDiv.at(i).toElement().text().contains("Il treno viaggia"))
				idx = i;


		if (idx != -1)
		{
			QString ritardo, transito;
			div = elementiDiv.at(idx).toElement();
			temp = div.firstChildElement("strong").text();
			idx = temp.indexOf("Ultimo");
			if (idx == -1)
			{
				ritardo = temp;
				transito = "";
			}
			else
			{
				ritardo = temp.left(idx);
				transito = temp.mid(idx);
			}

			//esistono i dati sull'ora e la località dell'ultimo transito
			if (transito != "")
			{
				m_orarioTreno.impostaDato(Treno::dtOrarioTransito, transito.section(' ', -2, -2));
				m_orarioTreno.impostaDato(Treno::dtUltimoRilevamento, transito.section(' ', 3, -5));
			}

			if (ritardo.contains("orario"))
				m_orarioTreno.impostaDato(Treno::dtRitardoTransito, QString::fromUtf8("in orario"));
			else
			{
				temp = ritardo.section(' ', 5, 5);
				if (temp == "1")
					m_orarioTreno.impostaDato(Treno::dtRitardoTransito, QString::fromUtf8("1 minuto in %1").arg(ritardo.section(' ', 8, 8)));
				else
					m_orarioTreno.impostaDato(Treno::dtRitardoTransito, QString::fromUtf8("%1 minuti in %2").arg(temp).arg(ritardo.section(' ', 8, 8)));
			}
		}

	}

	//estra il ritardo attuale per un treno in viaggio
	if (testoVT.contains("arrivato"))
	{
		//rintraccia l'elemento DIV con il ritardo
		idx = -1;
		for (int i = 0; i < elementiDiv.count(); i++)
			if (elementiDiv.at(i).toElement().text().contains("Il treno e' arrivato"))
				idx = i;
		if (idx != -1)
		{
			div = elementiDiv.at(idx).toElement();
			//il treno è arrivato in orario, non c'è ragione di estrarre il ritardo
			if (div.text().contains("orario"))
				m_orarioTreno.impostaDato(Treno::dtRitardoTransito, QString::fromUtf8("In orario"));
			else
			{
				QString minuti = div.text().section(' ', -5, -5);
				QString ritOAnticipo = div.text().section(' ', -2, -2);
				if (minuti == "1")
					temp = QString::fromUtf8("1 minuto in %1").arg(ritOAnticipo);
				else
					temp = QString::fromUtf8("%1 minuti in %2").arg(minuti).arg(ritOAnticipo);

				m_orarioTreno.impostaDato(Treno::dtRitardoTransito, temp);
			}


		}
	}

}

void Treno::DatiTreno::inizializza()
{
	m_statoTreno = DatiSconosciuti;

	m_dati.clear();
	m_fermate.clear();

}

Treno::Fermata::Fermata()
{
	m_effettuata = false;
	m_soppressa = false;

	m_fermata = "";
	m_binProgrammato = "";
	m_binReale = "";
	m_oraArrivoProgrammata = "";
	m_oraArrivoReale = "";
	m_oraArrivoStimata = "";
}

//Salva i dati necessari per poter ripristinare la scheda alla prossima apertura
void SchedaTreno::salvaScheda(QSettings& settings)
{
	settings.setValue("tipo scheda", "treno");
	settings.setValue("numero treno", numero());
	settings.setValue("intervallo", intervallo());
}
