# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


package MenuDisplay;
use Qt;
use Qt::isa qw (Qt::Widget);
use Qt::attributes qw (menu_preview active_button button_pressed);


sub NEW
{
  my $this         = shift;
  my $parent       = shift;
  my $name         = shift;
  my $menu_preview = shift;

  $this -> SUPER::NEW ($parent, $name, &Qt::WStaticContents|&Qt::WNoAutoErase);

  menu_preview   = $menu_preview;
  active_button  = -1;
  button_pressed = 0;
  setMouseTracking (1);
}


sub DESTROY
{
#  print STDERR "MenuDisplay::DESTROY\n";

  button_pressed = 0;
  active_button  = 0;
  menu_preview   = undef;

  SUPER -> DESTROY ();
}


sub preview
{
  my $background = shift;

  my $bgpm = Qt::Pixmap ($background);

  setPaletteBackgroundPixmap ($bgpm);
  setFixedSize ($bgpm -> size ());

  erase ();
  update ();
}


sub mouseMoveEvent
{
  my $ev = shift;

  my $button_geos = menu_preview -> button_geometries ();
  my $nbttns      = @{$button_geos};

  return unless ($nbttns);

  my $pos           = $ev -> pos ();
  my $active_button = active_button;

  return unless ($active_button < 0 ||
		 !$button_geos -> [$active_button] -> contains ($pos));

  # active button changed
  update ($button_geos -> [$active_button]) unless ($active_button < 0);

  my $i;

  for ($i = 0; $i < $nbttns; $i++)
  {
    next if ($i == $active_button);
    last if ($button_geos -> [$i] -> contains ($pos));
  }

  if ($i < $nbttns)
  {
    # new active button found
    active_button = $i;
    update ($button_geos -> [$i]);
  }
  elsif ($active_button >= 0)
  {
    # mouse moved to left or right of active button
    active_button = -1;
  }
}


sub mousePressEvent
{
  my $ev = shift;

  if ($ev -> button () & &Qt::LeftButton)
  {
    button_pressed    = 1;
    my $active_button = active_button;

    update (menu_preview -> button_geometries () -> [$active_button])
      unless ($active_button < 0);
  }
}


sub mouseReleaseEvent
{
  my $ev = shift;

  if ($ev -> button () & &Qt::LeftButton)
  {
    button_pressed    = 0;
    my $active_button = active_button;

    update (menu_preview -> button_geometries () -> [$active_button])
      unless ($active_button < 0);
  }
}


sub paintEvent
{
  my $ev = shift;

  my $rect          = $ev -> rect ();
  my $erased        = $ev -> erased ();
  my $active_button = active_button;
  my $menu_preview  = menu_preview;
  my $base          = $menu_preview -> base_pixmap ();
  my $button_geos   = $menu_preview -> button_geometries ();
  my $p             = Qt::Painter (this);

  if ($erased)
  {
    $p -> drawPixmap (0, 0, $base);
  }
  else
  {
    my $l = $rect -> left ();
    my $t = $rect -> top ();
    my $w = $rect -> width ();
    my $h = $rect -> height ();

    $p -> drawPixmap ($l, $t, $base, $l, $t, $w, $h);
  }

  return if ($active_button < 0);

  my $active_rect = $button_geos -> [$active_button];

  if ($erased || $active_rect -> intersects ($rect))
  {
    my $l  = $active_rect -> left ();
    my $t  = $active_rect -> top ();
    my $w  = $active_rect -> width ();
    my $h  = $active_rect -> height ();
    my $pm = button_pressed ? $menu_preview -> selected_pixmap ()
                            : $menu_preview -> hilited_pixmap ();

    $p -> drawPixmap ($l, $t, $pm, $l, $t, $w, $h);
  }
}


1;
