# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;

package DvdIfoRead;
use Carp;
use LibDvdRead;


sub new
{
  my $proto = shift;
  my $class = ref ($proto) || $proto;
  my $self  = {};

  $self -> {dev} = shift;
  $self -> {dvd} = LibDvdRead::DVDOpen ($self -> {dev});

  return undef unless ($self -> {dvd});

  $self -> {vmg} = LibDvdRead::ifoOpen ($self -> {dvd}, 0);

  unless ($self -> {vmg})
  {
    LibDvdRead::DVDClose ($self -> {dvd});

    return undef;
  }

  $self -> {vts} = [];

  return bless ($self, $class);
}


sub DESTROY
{
  my $self = shift;

  foreach my $vts (@{$self -> {vts}})
  {
    LibDvdRead::ifoClose ($vts) if ($vts);
  }

  LibDvdRead::ifoClose ($self -> {vmg});
  LibDvdRead::DVDClose ($self -> {dvd});
}


sub volume_identifier
{
  my $self = shift;

  return LibDvdRead::DVDVolumeIdentifier ($self -> {dvd});
}


sub disc_identifier
{
  my $self = shift;

  return LibDvdRead::DVDDiscIdentifier ($self -> {dvd});
}


sub nr_of_title_sets
{
  my $self = shift;

  return $self -> {vmg} {vmgi_mat} {vmg_nr_of_title_sets};
}


sub nr_of_titles
{
  my $self = shift;

  return $self -> {vmg} {tt_srpt} {nr_of_srpts};
}


sub title_set_nr
{
  my $self = shift;
  my $ttn  = shift;

  return LibDvdRead::title_info_array_getitem
    ($self -> {vmg} {tt_srpt} {title}, $ttn - 1) -> {title_set_nr};
}


sub title_nr_in_title_set
{
  my $self = shift;
  my $ttn  = shift;

  return LibDvdRead::title_info_array_getitem
    ($self -> {vmg} {tt_srpt} {title}, $ttn - 1) -> {vts_ttn};
}


sub nr_of_angles
{
  my $self = shift;
  my $ttn  = shift;

  return LibDvdRead::title_info_array_getitem
    ($self -> {vmg} {tt_srpt} {title}, $ttn - 1) -> {nr_of_angles};
}


sub nr_of_chapters
{
  my $self = shift;
  my $ttn  = shift;

  return LibDvdRead::title_info_array_getitem
    ($self -> {vmg} {tt_srpt} {title}, $ttn - 1) -> {nr_of_ptts};
}


sub vts
{
  my $self = shift;
  my $tsn  = shift;

  croak "vts: bad title set number: $tsn" unless ($tsn > 0);

  my $vts = $self -> {vts} [$tsn - 1];

  return $vts if ($vts);

  croak "vts: no such title set: $tsn" unless ($tsn <=
					       $self -> nr_of_title_sets ());

  $vts = LibDvdRead::ifoOpen ($self -> {dvd}, $tsn);

  $self -> {vts} [$tsn - 1] = $vts;

  return $vts;
}


sub nr_of_titles_in_title_set
{
  my $self = shift;
  my $tsn  = shift;

  return $self -> vts ($tsn) -> {vts_ptt_srpt} {nr_of_srpts};
}


sub nr_of_program_chains
{
  my $self = shift;
  my $tsn  = shift;

  return $self -> vts ($tsn) -> {vts_pgcit} {nr_of_pgci_srp};
}


sub vts_video_attr
{
  my $self = shift;
  my $tsn  = shift;

  return $self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr};
}


sub mpeg_version
{
  my $self = shift;
  my $tsn  = shift;

  my $mv = $self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {mpeg_version};

  return "MPEG-1" if ($mv == 0);
  return "MPEG-2" if ($mv == 1);
  return "unknown version";		#$translate
}


sub video_format
{
  my $self = shift;
  my $tsn  = shift;

  my $vf = $self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {video_format};

  return "NTSC" if ($vf == 0);
  return "PAL"  if ($vf == 1);
  return "unknown video format";	#$translate
}


sub display_aspect_ratio
{
  my $self = shift;
  my $tsn  = shift;

  my $ar =
    $self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {display_aspect_ratio};

  return "4:3"  if ($ar == 0);
  return "16:9" if ($ar == 3);
  return "unknown aspect ratio";	#$translate
}


sub permitted_display_formats
{
  my $self = shift;
  my $tsn  = shift;

  my $pdf = $self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {permitted_df};

  return ( "wide", "pan&scan", "letterboxed" ) if ($pdf == 0);	#$translate
  return ( "wide", "pan&scan" )                if ($pdf == 1);	#$translate
  return ( "wide", "letterboxed" )             if ($pdf == 2);	#$translate
  return ( "unrestricted" );					#$translate
}


sub ntsc_cc
{
  my $self = shift;
  my $tsn  = shift;

  my @rv = ();

  push (@rv, 1)
    if ($self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {line21_cc_1});
  push (@rv, 2)
    if ($self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {line21_cc_2});

  return @rv;
}


sub constant_bit_rate
{
  my $self = shift;
  my $tsn  = shift;

  return $self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {bit_rate};
}


sub picture_size
{
  my $self = shift;
  my $tsn  = shift;

  my $height = 480;

  $height = 576
    if ($self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {video_format});

  my $ps = $self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {picture_size};

  return ( 720, $height ) if ($ps == 0);
  return ( 704, $height ) if ($ps == 1);
  return ( 352, $height ) if ($ps == 2);
  return ( 352, $height/2 );
}


sub letterboxed
{
  my $self = shift;
  my $tsn  = shift;

  return $self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {letterboxed};
}


sub film_mode
{
  my $self = shift;
  my $tsn  = shift;

  return !$self -> vts ($tsn) -> {vtsi_mat} {vts_video_attr} {film_mode};
}


sub nr_of_audio_streams
{
  my $self = shift;
  my $tsn  = shift;

  return $self -> vts ($tsn) -> {vtsi_mat} {nr_of_vts_audio_streams};
}


sub vts_audio_attr
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  croak "vts_audio_attr: bad audio stream: $stream" unless ($stream > 0);

  my $vts = $self -> vts ($tsn);

  croak "vts_audio_attr: no such audio stream: $stream"
    unless ($stream <= $vts -> {vtsi_mat} {nr_of_vts_audio_streams});

  return LibDvdRead::audio_attr_array_getitem
    ($vts -> {vtsi_mat} {vts_audio_attr}, $stream - 1);
}


sub audio_format
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $af = $self -> vts_audio_attr ($tsn, $stream) -> {audio_format};

  return "Dolby AC-3"      if ($af == 0);
  return "MPEG-1"          if ($af == 2);
  return "MPEG-2 Extended" if ($af == 3);
  return "Linear PCM"      if ($af == 4);
  return "DTS"             if ($af == 6);
  return "unknown audio format";		#$translate
}


sub audio_multichannel_extension
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  return $self -> vts_audio_attr ($tsn, $stream) -> {multichannel_extension};
}


sub audio_application_mode
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $attr = $self -> vts_audio_attr ($tsn, $stream);
  my $am   = $attr -> {application_mode};

  return "unspecified mode" if ($am == 0);	#$translate

  if ($am == 1)
  {
    my $karaoke = $attr -> {app_info} {karaoke};

    return sprintf "%s, %s%s (%d, %d)",
      "karaoke",				#$translate
      $karaoke -> {mode} ? "duet" : "solo",	#$translate
      $karaoke -> {mc_intro} ? (", " .
				"mc intro") :	#$translate
				"",
      $karaoke -> {channel_assignment},
      $karaoke -> {version};
  }

  if ($am == 2)
  {
    return "Dolby surround"
      if ($attr -> {app_info} {surround} {dolby_encoded});

    return "surround sound";	#$translate
  }

  return "unknown mode";	#$translate
}


sub audio_quantization
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $qu = $self -> vts_audio_attr ($tsn, $stream) -> {quantization};

  return ( "16 bps", "20 bps", "24 bps", "DRC") [$qu];
}


sub audio_sample_frequency
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $sf = $self -> vts_audio_attr ($tsn, $stream) -> {sample_frequency};

  return "48 kHz" if ($sf == 0);
  return "96 kHz" if ($sf == 1);
  return "unknown frequency";		#$translate
}


sub audio_channels
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $attr = $self -> vts_audio_attr ($tsn, $stream);
  my $af   = $attr -> {audio_format};
  my $ch   = $attr -> {channels} + 1;

  return "Mono" if ($ch == 1);		#$translate

  return "Stereo" if ($ch == 2);	#$translate

  # AC3, MPEG2Ext or DTS with 6 channels
  return "5.1" if ($af =~ /[036]/ && $ch == 6);

  # MPEG2Ext or DTS with 8 channels
  return "7.1" if ($af =~ /[36]/ && $ch == 8);

  return $ch;
}


sub audio_lang_code
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $attr = $self -> vts_audio_attr ($tsn, $stream);
  my $type = $attr -> {lang_type};

  return "xx"     if ($type == 0);
  return "??" unless ($type == 1);

  my $code = $attr -> {lang_code};
  my $ext  = $attr -> {lang_extension};
  my $lang = chr ($code >> 8) . chr ($code & 0xff);

  return ( $lang ) unless ($ext);
  return ( $lang, chr ($ext) );
}


sub audio_code_extension
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $ce = $self -> vts_audio_attr ($tsn, $stream) -> {code_extension};

  return "unspecified extension"            if ($ce == 0);	#$translate
  return "normal caption"                   if ($ce == 1);	#$translate
  return "audio for visually impaired"      if ($ce == 2);	#$translate
  return "director's comments 1"            if ($ce == 3);	#$translate
  return "director's comments 2"            if ($ce == 4);	#$translate
  return "unknown extension";					#$translate
}


sub nr_of_subtitle_streams
{
  my $self = shift;
  my $tsn  = shift;

  return $self -> vts ($tsn) -> {vtsi_mat} {nr_of_vts_subp_streams};
}


sub vts_subp_attr
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  croak "vts_subp_attr: bad subtitle stream: $stream" unless ($stream > 0);

  my $vts = $self -> vts ($tsn);

  croak "vts_subp_attr: no such subtitle stream: $stream"
    unless ($stream <= $vts -> {vtsi_mat} {nr_of_vts_subp_streams});

  return LibDvdRead::subp_attr_array_getitem
    ($vts -> {vtsi_mat} {vts_subp_attr}, $stream - 1);
}


sub subtitle_code_mode
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $cm = $self -> vts_subp_attr ($tsn, $stream) -> {code_mode};

  return "RLE"      if ($cm == 0);
  return "Extended" if ($cm == 1);
  return "unknown subtitle code mode";	#$translate
}


sub subtitle_lang_code
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $attr = $self -> vts_subp_attr ($tsn, $stream);
  my $type = $attr -> {type};

  return "xx"     if ($type == 0);
  return "??" unless ($type == 1);

  my $code = $attr -> {lang_code};
  my $ext  = $attr -> {lang_extension};
  my $lang = chr ($code >> 8) . chr ($code & 0xff);

  return ( $lang ) unless ($ext);
  return ( $lang, chr ($ext) );
}


sub subtitle_code_extension
{
  my $self   = shift;
  my $tsn    = shift;
  my $stream = shift;

  my $ce = $self -> vts_subp_attr ($tsn, $stream) -> {code_extension};

  return "unspecified extension"                      if ($ce == 0);#$translate
  return "caption with normal size characters"        if ($ce == 1);#$translate
  return "caption with bigger size characters"        if ($ce == 2);#$translate
  return "caption for children"                       if ($ce == 3);#$translate
  return "closed caption with normal size characters" if ($ce == 5);#$translate
  return "closed caption with bigger size characters" if ($ce == 6);#$translate
  return "closed caption for children"                if ($ce == 7);#$translate
  return "forced caption"                             if ($ce == 9);#$translate
  return "director's comments with normal size characters"	    #$translate
    if ($ce == 13);
  return "director's comments with bigger size characters"	    #$translate
    if ($ce == 14);
  return "director's comments for children"          if ($ce == 15);#$translate
  return "reserved subtitle code extension"          if ($ce <  16);#$translate
  return "unknown extension";					    #$translate
}


sub nr_of_ptts
{
  my $self = shift;
  my $tsn  = shift;
  my $ttn  = shift;

  return LibDvdRead::ttu_array_getitem
    ($self -> vts ($tsn) -> {vts_ptt_srpt} {title}, $ttn - 1) -> {nr_of_ptts};
}


sub program_chain_nr
{
  my $self    = shift;
  my $tsn     = shift;
  my $ttn     = shift;
  my $chapter = shift;

  my $ptt = LibDvdRead::ttu_array_getitem
    ($self -> vts ($tsn) -> {vts_ptt_srpt} {title}, $ttn - 1) -> {ptt};
  return LibDvdRead::ptt_info_array_getitem ($ptt, $chapter - 1) -> {pgcn};
}


sub program_nr
{
  my $self    = shift;
  my $tsn     = shift;
  my $ttn     = shift;
  my $chapter = shift;

  my $ptt = LibDvdRead::ttu_array_getitem
    ($self -> vts ($tsn) -> {vts_ptt_srpt} {title}, $ttn - 1) -> {ptt};
  return LibDvdRead::ptt_info_array_getitem ($ptt, $chapter - 1) -> {pgn};
}


sub nr_of_programs
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  return $pgc -> {nr_of_programs};
}


sub nr_of_cells
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  return $pgc -> {nr_of_cells};
}


sub pgc_frame_rate
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  my $frc = $pgc -> {playback_time} {frame_u} >> 6;
  return ( 0, 25.00, 0, 29.97 ) [$frc];
}


sub bcd2bin
{
  my $val = shift;

  return ($val >> 4) * 10 + ($val & 0xf);
}


sub pgc_playback_time
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  my $pbt = $pgc -> {playback_time};
  return ( bcd2bin ($pbt -> {hour}),   bcd2bin ($pbt -> {minute}),
	   bcd2bin ($pbt -> {second}), bcd2bin ($pbt -> {frame_u} & 0x3f) );
}


sub palette
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};

  my @palette = ();

  foreach my $i (0..15)
  {
    push @palette, LibDvdRead::palette_getitem ($pgc -> {palette}, $i);
  }

  return @palette;
}


sub program_start_cell
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;
  my $pgn  = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  return LibDvdRead::pgc_program_map_array_getitem
    ($pgc -> {program_map}, $pgn - 1);
}


sub program_cells
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;
  my $pgn  = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  my $start = LibDvdRead::pgc_program_map_array_getitem
    ($pgc -> {program_map}, $pgn - 1);
  my $end = $pgn < $pgc -> {nr_of_programs} ?
    LibDvdRead::pgc_program_map_array_getitem
    ($pgc -> {program_map}, $pgn) - 1 : $pgc -> {nr_of_cells};

  return ( $start .. $end );
}


sub block_type
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;
  my $cell = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  return LibDvdRead::cell_playback_array_getitem
    ($pgc -> {cell_playback}, $cell - 1) -> {block_type};
}


sub block_mode
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;
  my $cell = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  return LibDvdRead::cell_playback_array_getitem
    ($pgc -> {cell_playback}, $cell - 1) -> {block_mode};
}


sub refine_angle_cells
{
  my $self  = shift;
  my $angle = shift;
  my $tsn   = shift;
  my $pgcn  = shift;
  my $cells = shift;

  my @rv;
  my $next_cell = 0;

  foreach my $cell (@{$cells})
  {
    # skip to beginning of next block if we had an angle block before
    next if ($cell < $next_cell);

    # check if entering an angle block
    if ($self -> block_type ($tsn, $pgcn, $cell) == 1)
    {
      # advance to angle's cell
      $cell += $angle - 1;

      # search last cell of angle block
      for ($next_cell = $cell;
	   $self -> block_mode ($tsn, $pgcn, $next_cell) != 3; $next_cell++)
      {
      }

      # advance to first cell of next block
      $next_cell++;
    }

    push @rv, $cell;
  }

  return @rv;
}


sub cell_frame_rate
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;
  my $cell = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  my $frc = LibDvdRead::cell_playback_array_getitem
    ($pgc -> {cell_playback}, $cell - 1) -> {playback_time} {frame_u} >> 6;
  return ( 0, 25.00, 0, 29.97 ) [$frc];
}


sub cell_playback_time
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;
  my $cell = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  my $pbt = LibDvdRead::cell_playback_array_getitem
    ($pgc -> {cell_playback}, $cell - 1) -> {playback_time};
  return ( bcd2bin ($pbt -> {hour}),   bcd2bin ($pbt -> {minute}),
	   bcd2bin ($pbt -> {second}), bcd2bin ($pbt -> {frame_u} & 0x3f) );
}


sub nr_of_sectors
{
  my $self = shift;
  my $tsn  = shift;
  my $pgcn = shift;
  my $cell = shift;

  my $pgc = LibDvdRead::pgci_srp_array_getitem
    ($self -> vts ($tsn) -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
  my $cpb = LibDvdRead::cell_playback_array_getitem
    ($pgc -> {cell_playback}, $cell - 1);
  return $cpb -> {last_sector} - $cpb -> {first_sector} + 1;
}


sub add_time
{
  my $h1  = shift;
  my $m1  = shift;
  my $s1  = shift;
  my $f1  = shift;
  my $h2  = shift;
  my $m2  = shift;
  my $s2  = shift;
  my $f2  = shift;
  my $fps = shift;

  if ($fps)
  {
    for ($f1 += $f2; $f1 >= $fps; $f1 -= $fps)
    {
      $s1++;
    }
  }

  for ($s1 += $s2; $s1 >= 60; $s1 -= 60)
  {
    $m1++;
  }

  for ($m1 += $m2; $m1 >= 60; $m1 -= 60)
  {
    $h1++;
  }

  $h1 += $h2;

  return ( $h1, $m1, $s1, $f1 );
}


sub title_playback_time
{
  my $self = shift;
  my $tsn  = shift;
  my $ttn  = shift;

  my @play_time = ( 0, 0, 0, 0 );
  my $vts = $self -> vts ($tsn);
  my $nr_of_ptts = LibDvdRead::ttu_array_getitem
    ($vts -> {vts_ptt_srpt} {title}, $ttn - 1) -> {nr_of_ptts};
  my $ptt = LibDvdRead::ttu_array_getitem
    ($vts -> {vts_ptt_srpt} {title}, $ttn - 1) -> {ptt};

  foreach my $pttn (1 .. $nr_of_ptts)
  {
    my $pgcn = LibDvdRead::ptt_info_array_getitem ($ptt, $pttn - 1) -> {pgcn};
    my $pgn  = LibDvdRead::ptt_info_array_getitem ($ptt, $pttn - 1) -> {pgn};
    my $pgc  = LibDvdRead::pgci_srp_array_getitem
      ($vts -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
    my $start = LibDvdRead::pgc_program_map_array_getitem
      ($pgc -> {program_map}, $pgn - 1);
    my $end = $pgn < $pgc -> {nr_of_programs} ?
      LibDvdRead::pgc_program_map_array_getitem
      ($pgc -> {program_map}, $pgn) - 1 : $pgc -> {nr_of_cells};

    foreach my $cell ($start .. $end)
    {
      my @pbt;
      my $pbt = LibDvdRead::cell_playback_array_getitem
	($pgc -> {cell_playback}, $cell - 1) -> {playback_time};
      push @pbt, bcd2bin ($pbt -> {hour});
      push @pbt, bcd2bin ($pbt -> {minute});
      push @pbt, bcd2bin ($pbt -> {second});
      push @pbt, bcd2bin ($pbt -> {frame_u} & 0x3f);
      my $fps    = ( 0, 25.00, 0, 29.97 ) [$pbt -> {frame_u} >> 6];
      @play_time = add_time (@play_time, @pbt, $fps);
    }
  }

  return @play_time;
}


sub title_size
{
  my $self = shift;
  my $tsn  = shift;
  my $ttn  = shift;

  my $nr_of_sectors = 0;
  my $vts = $self -> vts ($tsn);
  my $nr_of_ptts = LibDvdRead::ttu_array_getitem
    ($vts -> {vts_ptt_srpt} {title}, $ttn - 1) -> {nr_of_ptts};
  my $ptt = LibDvdRead::ttu_array_getitem
    ($vts -> {vts_ptt_srpt} {title}, $ttn - 1) -> {ptt};

  foreach my $pttn (1 .. $nr_of_ptts)
  {
    my $pgcn = LibDvdRead::ptt_info_array_getitem ($ptt, $pttn - 1) -> {pgcn};
    my $pgn  = LibDvdRead::ptt_info_array_getitem ($ptt, $pttn - 1) -> {pgn};
    my $pgc  = LibDvdRead::pgci_srp_array_getitem
      ($vts -> {vts_pgcit} {pgci_srp}, $pgcn - 1) -> {pgc};
    my $start = LibDvdRead::pgc_program_map_array_getitem
      ($pgc -> {program_map}, $pgn - 1);
    my $end = $pgn < $pgc -> {nr_of_programs} ?
      LibDvdRead::pgc_program_map_array_getitem
      ($pgc -> {program_map}, $pgn) - 1 : $pgc -> {nr_of_cells};

    foreach my $cell ($start .. $end)
    {
      my $cpb = LibDvdRead::cell_playback_array_getitem
	($pgc -> {cell_playback}, $cell - 1);
      my $pbt = $cpb -> {playback_time};

      $nr_of_sectors += $cpb -> {last_sector} - $cpb -> {first_sector} + 1
	if (($pbt -> {frame_u} & 0x3f) || $pbt -> {second} ||
	    $pbt  -> {minute}          || $pbt -> {hour});
    }
  }

  return $nr_of_sectors;
}


1;
