/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "QtChatLogViewer.h"
#include "ui_ChatLogViewer.h"

#include <model/config/Config.h>
#include <model/config/ConfigManager.h>
#include <model/contactlist/ContactProfile.h>
#include <model/profile/UserProfile.h>
#include <model/profile/UserProfileHandler.h>

#include <control/CQuteCom.h>
#include <control/profile/CUserProfile.h>
#include <control/profile/CUserProfileHandler.h>

#include <presentation/qt/QtQuteCom.h>
#include <presentation/qt/QtNoQuteComAlert.h>
#include <presentation/qt/QtToolBar.h>
#include <presentation/qt/contactlist/QtContactList.h>
#include <presentation/qt/contactlist/QtContactListManager.h>
#include <presentation/qt/filetransfer/QtFileTransfer.h>
#include <presentation/qt/profile/QtProfileDetails.h>
#include <presentation/qt/webservices/sms/QtSms.h>

#include <imwrapper/IMContact.h>
#include <imwrapper/IMContactSet.h>

#include <cutil/global.h>
#include <qtutil/Object.h>
#include <qtutil/SafeConnect.h>
#include <qtutil/StringListConvert.h>
#include <util/Logger.h>
#include <util/SafeDelete.h>
#include <util/StringList.h>

#include <QtCore/QDate>
#include <QtCore/QDir>
#include <QtCore/QFileInfo>
#include <QtCore/QLocale>
#include <QtCore/QTime>

#include <QtGui/QDialog>

QtChatLogViewer::QtChatLogViewer(QWidget * parent, QtQuteCom & qtQuteCom, QString log)
	: QMainWindow(parent),
	_qtQuteCom(qtQuteCom),_log(log) {

	_ui = new Ui::ChatLogViewer();
	_ui->setupUi(this);
	connect(_ui->historyLog,SIGNAL(loadFinished(bool)),this,SLOT(loadFinished(bool)));

	setupMenuBarActions();
	setupToolBarActions();
	
	#if defined(OS_MACOSX)
	setUnifiedTitleAndToolBarOnMac(true);
#endif
	
	SAFE_CONNECT(qtQuteCom.getQtContactList(), SIGNAL(contactChangedEventSignal(QString )), SLOT(contactChangedEventSlot(QString)));
	
	updateToolBarActions();
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	_ui->historyLog->setTranslate(false);
	_ui->historyLog->setTheme(QString::fromUtf8(config.getChatTheme().c_str()),QString::fromUtf8(config.getChatThemeVariant().c_str()));

	_ui->actionSearchQuteComContact->setVisible(false);
	_ui->menuQuteCom->setTitle(tr("Account"));
}

void QtChatLogViewer::loadFinished(bool)
{
	QtContactList * qtContactList = _qtQuteCom.getQtContactList();
	CContactList & cContactList = qtContactList->getCContactList();

	UserProfile * userProfile = _qtQuteCom.getCQuteCom().getCUserProfileHandler().getUserProfileHandler().getCurrentUserProfile();
	if (userProfile) {
		// Setup user avatar
		QPixmap pixmap;
		std::string myData = userProfile->getIcon().getData();
		pixmap.loadFromData((uchar *)myData.c_str(), myData.size());
		_ui->historyLog->setAvatarPixmap("self", pixmap);
	
		// Read history
		std::string userlogin;
		HistoryMementoCollection hmc;
		StringList cuuidList;
		userProfile->getHistory().loadChatLog(std::string(_log.toUtf8()), &hmc, &userlogin, &cuuidList);
		_cuuidList = StringListConvert::toQStringList(cuuidList);
		
		bool isQuteComAccountConnected = userProfile->hasQuteComAccount();

		QString selfContactName = QString::fromUtf8(userlogin.c_str());

		// Set remote user avatars
		Q_FOREACH(QString uuid, _cuuidList) {
			ContactProfile profile = cContactList.getContactProfile(std::string(uuid.toUtf8()));

			std::string data = profile.getIcon().getData();
			if (data.size() != 0) {
				QPixmap pixmap;
				pixmap.loadFromData((uchar *)data.c_str(), data.size());
				_ui->historyLog->setAvatarPixmap(uuid, pixmap);
			}
		}
		
		// parses HistoryMementoCollection to find messages and inserts them in historyLog
		_ui->historyLog->setProtocol (EnumIMProtocol::IMProtocolAll);
		Config & config = ConfigManager::getInstance().getCurrentConfig();
		QString qutecomSuffix = "@" + QString::fromUtf8( config.getQuteComRealm().c_str() );
		QDate previousDate;
		for (HistoryMap::iterator it = hmc.begin(); it != hmc.end(); it++) {
		
			HistoryMemento * memento = it->second;

			// Read date and time
			Date mementoDate = memento->getDate();
			QDate date(mementoDate.getYear(), mementoDate.getMonth(), mementoDate.getDay());
			Time mementoTime = memento->getTime();
			QTime time = QTime(mementoTime.getHour(), mementoTime.getMinute());

			// Insert date line if necessary
			if (date != previousDate) {
				QString txt = QLocale::system().toString(date);
				_ui->historyLog->insertStatusMessage(txt, time);
				previousDate = date;
			}
			
			// Insert message
			QString contactName = QString::fromUtf8(memento->getPeer().c_str());
			if (isQuteComAccountConnected) {
				contactName.remove(qutecomSuffix);
			}
			contactName.remove("sip:");
			
			QString contactId;
			if (contactName == selfContactName) {
				contactId = "self";
			} else {
				contactId = _cuuidList[0];
				// FIXME: History format is broken: it does not store the uuid
				// associated with a peer. Using the first uid will work for
				// all one-to-one chat, but will fail for chat conference.
			}

			_ui->historyLog->insertMessage(
				contactId,
				contactName,
				QString::fromUtf8(memento->getData().c_str()),
				time
			);
		}
	}
}

QtChatLogViewer::~QtChatLogViewer() {
	OWSAFE_DELETE(_ui);
}

void QtChatLogViewer::copyQAction(QObject * actionParent, QAction * action) {
	QAction * tmp = actionParent->findChild<QAction *>(action->objectName());
	if (!tmp) {
		LOG_FATAL("QAction is null, cannot copy it");
	}
	action->setIcon(tmp->icon());
	action->setIconText(tmp->iconText());
	action->setText(tmp->text());
	action->setToolTip(tmp->toolTip());
	action->setShortcut(tmp->shortcut());
}

void QtChatLogViewer::setupMenuBarActions() {
	QtToolBar * qtToolBar = &_qtQuteCom.getQtToolBar();
	QWidget * toolBar = qtToolBar->getWidget();

	// setup "QuteCom" menu
	copyQAction(toolBar, _ui->actionShowQuteComAccount);
	SAFE_CONNECT_RECEIVER(_ui->actionShowQuteComAccount, SIGNAL(triggered()), qtToolBar, SLOT(showQuteComAccount()));
	copyQAction(toolBar, _ui->actionEditMyProfile);
	SAFE_CONNECT_RECEIVER(_ui->actionEditMyProfile, SIGNAL(triggered()), qtToolBar, SLOT(editMyProfile()));
	copyQAction(toolBar, _ui->actionQuteComServices);
	SAFE_CONNECT_RECEIVER(_ui->actionQuteComServices, SIGNAL(triggered()), qtToolBar, SLOT(showQuteComServices()));
	copyQAction(toolBar, _ui->actionClose);
	SAFE_CONNECT(_ui->actionClose, SIGNAL(triggered()), SLOT(close()));
	////

	// setup "contact" menubar
	copyQAction(toolBar, _ui->actionAddContact);
	SAFE_CONNECT_RECEIVER(_ui->actionAddContact, SIGNAL(triggered()), qtToolBar, SLOT(addContact()));
	copyQAction(toolBar, _ui->actionSearchQuteComContact);
	SAFE_CONNECT_RECEIVER(_ui->actionSearchQuteComContact, SIGNAL(triggered()), qtToolBar, SLOT(searchQuteComContact()));
	////

	// setup "actions" menu
	copyQAction(toolBar, _ui->actionCreateConferenceCall);
	SAFE_CONNECT_RECEIVER(_ui->actionCreateConferenceCall, SIGNAL(triggered()), qtToolBar, SLOT(createConferenceCall()));

#if defined (DISABLE_SMS)
	_ui->actionSendSms->setVisible(false);
#else
	copyQAction(toolBar, _ui->actionSendSms);
	SAFE_CONNECT_RECEIVER(_ui->actionSendSms, SIGNAL(triggered()), qtToolBar, SLOT(sendSms()));
#endif
	
	SAFE_CONNECT(_ui->actionSaveHistoryAs, SIGNAL(triggered()), SLOT(saveChatHistory()));
	SAFE_CONNECT(_ui->actionActionRestartChat, SIGNAL(triggered()), SLOT(restartChat()));
	////

#if defined (DISABLE_FILETRANSFER)
	_ui->actionSendFile->setVisible(false);
	_ui->actionShowFileTransfer->setVisible(false);
#else
  	// setup "tools" menu
  	copyQAction(toolBar, _ui->actionShowFileTransfer);
  	SAFE_CONNECT_RECEIVER(_ui->actionShowFileTransfer, SIGNAL(triggered()), qtToolBar, SLOT(showFileTransferWindow()));
  	////
#endif

	// setup "help" menu
	copyQAction(toolBar, _ui->actionShowQuteComForum);
	SAFE_CONNECT_RECEIVER(_ui->actionShowQuteComForum, SIGNAL(triggered()), qtToolBar, SLOT(showQuteComForum()));
	copyQAction(toolBar, _ui->actionQuteComFAQ);
	SAFE_CONNECT_RECEIVER(_ui->actionQuteComFAQ, SIGNAL(triggered()), qtToolBar, SLOT(showQuteComFAQ()));
	////
}

void QtChatLogViewer::setupToolBarActions() {
	SAFE_CONNECT(_ui->actionCallContact, SIGNAL(triggered()), SLOT(callContact()));
	SAFE_CONNECT(_ui->actionSendSms, SIGNAL(triggered()), SLOT(sendSmsToContact()));
	SAFE_CONNECT(_ui->actionSendFile, SIGNAL(triggered()), SLOT(sendFileToContact()));
	SAFE_CONNECT(_ui->actionCreateChatConf, SIGNAL(triggered()), SLOT(createChatConference()));
	SAFE_CONNECT(_ui->actionContactInfo, SIGNAL(triggered()), SLOT(showContactInfo()));
}

bool QtChatLogViewer::canDoFileTransfer(const ContactProfile & contactProfile) {
	if (!contactProfile.getFirstQuteComId().empty() && contactProfile.isAvailable()) {
		IMContact imContact = contactProfile.getFirstAvailableQuteComIMContact();
		if ( (imContact.getPresenceState() != EnumPresenceState::PresenceStateOffline) &&
				(imContact.getPresenceState() != EnumPresenceState::PresenceStateUnknown) &&
				(imContact.getPresenceState() != EnumPresenceState::PresenceStateUnavailable)) {
				return true;
		}
	}
	return false;
}

void QtChatLogViewer::updateToolBarActions() {

	QtContactList * qtContactList = _qtQuteCom.getQtContactList();
	ContactProfile contactProfile;
	
	//call'n chat
	if (_cuuidList.size() <= 0) {
		//no contact to call
		_ui->actionCallContact->setEnabled(false);
		_ui->actionActionRestartChat->setEnabled(false);
	} else	if (_cuuidList.size() > 1) {
		//TO DO
		_ui->actionCallContact->setEnabled(false);
		_ui->actionActionRestartChat->setEnabled(false);
	} else {
		std::string cuuid(_cuuidList[0].toUtf8());
		contactProfile = qtContactList->getCContactList().getContactProfile(cuuid);
		_ui->actionCallContact->setEnabled(contactProfile.hasCall() && contactProfile.isAvailable());
		
		_ui->actionActionRestartChat->setEnabled(contactProfile.hasIM());
	}
	////
	
	// depending on sip account type ( SMS, File Transfert )
	bool sendSMS = false;
	bool sendFile = false;
	UserProfile * userProfile = _qtQuteCom.getCQuteCom().getCUserProfileHandler().getUserProfileHandler().getCurrentUserProfile();
	if (userProfile->hasQuteComAccount()) {
		for (QStringList::const_iterator it = _cuuidList.begin(); it != _cuuidList.end(); it++) {
			contactProfile = qtContactList->getCContactList().getContactProfile(std::string(it->toUtf8()));
			sendSMS |= !contactProfile.getMobilePhone().empty();
			sendFile |= canDoFileTransfer(contactProfile);
		}
	} else if (userProfile->hasSipAccount()) {
		sendSMS = true; // a pop up
		//sendFile = false; //useless
	}
	_ui->actionSendSms->setEnabled(sendSMS);
	_ui->actionSendFile->setEnabled(sendFile);
	////

	if (_cuuidList.size() <= 0) {
		_ui->actionContactInfo->setEnabled(false);
		_ui->actionCreateChatConf->setEnabled(false);
	} else {
		_ui->actionContactInfo->setEnabled(true);
		_ui->actionCreateChatConf->setEnabled(false);
	}

}

void QtChatLogViewer::sendSmsToContact() {
	QtContactList * qtContactList = _qtQuteCom.getQtContactList();
	ContactProfile contactProfile;
	UserProfile * userProfile = _qtQuteCom.getCQuteCom().getCUserProfileHandler().getUserProfileHandler().getCurrentUserProfile();
	
	if (userProfile->hasQuteComAccount()) {
		QtSms * sms = _qtQuteCom.getQtSms();
		if (sms) {
			sms->clear();
			for (QStringList::const_iterator it = _cuuidList.begin(); it != _cuuidList.end(); it++) {
				contactProfile = qtContactList->getCContactList().getContactProfile(std::string(it->toUtf8()));
				sms->addPhoneNumber(QString::fromUtf8(contactProfile.getMobilePhone().c_str()));
			}
			sms->getWidget()->show();
		}
	} else {
		QtNoQuteComAlert alert(this, _qtQuteCom);
		alert.getQDialog()->exec();
	}
}

void QtChatLogViewer::sendFileToContact() {

	QtFileTransfer * qtFileTransfer = _qtQuteCom.getFileTransfer();
	if (!qtFileTransfer) {
		return;
	}

	// gets a filename
	QString filename = qtFileTransfer->getChosenFile();
	if (filename.isEmpty()) {
		return;
	}
	QFileInfo fileInfo(filename);
	Config & config = ConfigManager::getInstance().getCurrentConfig();
	config.set(Config::FILETRANSFER_LASTUPLOADEDFILE_FOLDER_KEY, std::string(fileInfo.dir().absolutePath().toUtf8()));
	////
	
	// sends this file to all available contacts
	QtContactList * qtContactList = _qtQuteCom.getQtContactList();
	CContactList & cContactList = qtContactList->getCContactList();
	IMContactSet contactSet;
	ContactProfile contactProfile;
	
	for (QStringList::const_iterator it = _cuuidList.begin(); it != _cuuidList.end(); it++) {
		contactProfile = qtContactList->getCContactList().getContactProfile(std::string(it->toUtf8()));
		
		if (contactProfile.hasAvailableQuteComId()) {
			contactSet.insert(contactProfile.getFirstAvailableQuteComIMContact());
		}
	}
	qtFileTransfer->createSendFileSession(contactSet, filename, cContactList);
	////
}

void QtChatLogViewer::callContact() {
	if ((_cuuidList.size() > 0) && (!_cuuidList[0].isEmpty())) {
		QtContactListManager::getInstance()->startCall(_cuuidList[0]);
		if (_qtQuteCom.getWidget()->isMinimized()) {
			_qtQuteCom.getWidget()->showNormal();
		}
	}
}

void QtChatLogViewer::showContactInfo() {

	QtContactList * qtContactList = _qtQuteCom.getQtContactList();
	CContactList & cContactList = qtContactList->getCContactList();
	ContactProfile contactProfile;

	for (QStringList::const_iterator it = _cuuidList.begin(); it != _cuuidList.end(); it++) {
		contactProfile = cContactList.getContactProfile(std::string(it->toUtf8()));

		QtProfileDetails qtProfileDetails(
				*_qtQuteCom.getCQuteCom().getCUserProfileHandler().getCUserProfile(),
		contactProfile, this, tr("Edit Contact"));
		if (qtProfileDetails.show()) {
			_qtQuteCom.getCQuteCom().getCUserProfileHandler().getCUserProfile()->getCContactList().updateContact(contactProfile);
		}
	}
}

void QtChatLogViewer::saveChatHistory() {
	_ui->historyLog->saveHistoryAsHtml();
}

void QtChatLogViewer::restartChat() {

	// starts a "normal" chat
	if (_cuuidList.size() > 0) {
		_qtQuteCom.getCQuteCom().getCUserProfileHandler().getCUserProfile()->startIM(std::string(_cuuidList[0].toUtf8()));
	}
	////
	
	//// TO DO allow to start a multi-chat 
	// launches a confchat
	//if (_cuuidList.size() > 1) {
	//	for (QStringList::const_iterator it = _cuuidList.begin(); it != _cuuidList.end(); it++) {
	//		
	//	}
	//}
	//////
}

void QtChatLogViewer::createChatConference() {

	//// TO DO allow to start a multi-chat 

	//QtChatWidget * widget = getActiveTabWidget();
	//if (widget) {
	//	widget->showInviteDialog();
	//}
	
	//if (canDoMultiChat()) {
	//	QtChatRoomInviteDlg dlg(*_imChatSession,
	//		_cChatHandler.getCUserProfile().getCContactList(), this);
	//	dlg.exec();
	//}
}

void QtChatLogViewer::contactChangedEventSlot(QString contactid) {

	bool updateNeeded = false;

	for (QStringList::const_iterator it = _cuuidList.begin(); it != _cuuidList.end(); it++) {
		if ((*it) == contactid) {
				updateNeeded = true;
		} 
	}
	
	if (updateNeeded && isVisible()) {
		updateToolBarActions();
	}
}
