/***************************************************************************
                          qimageioext.cpp  -  description
                             -------------------
    begin                : Mon Mar 26 2001
    copyright            : (C) 2001 by M. Herder
    email                : http://quiteinsane.sf.net/contact.html
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "qimageioext.h"

#include <qcolor.h>
#include <qcstring.h>
#include <qfile.h>
#include <qimage.h>
#include <qstring.h>
#include <qtextstream.h>
#include <stdlib.h>
#include <ctype.h>

/*****************************************************************************
  PBM/PGM/PPM (ASCII and RAW) image read/write functions
 *****************************************************************************/
//Why do we need this?
//Because the relevant function is buggy in all versions of
//Qt2.2.x, at least up to version 2.2.4
//This leads to random crashes when one tries to save an 8 bit
//image in PGM format.
//There's also a problem with saving 1 bit images in PBM format
//in the original code, which leads to a conversion
//to 8 bit and therefore to bloated files in the wrong format.
//This reader can also read images in 16bit raw format. It
//converts the data to 8bit/channel.
//This code is based on the functions
//static void read/write_pbm_image( QImageIO *iio ) in qimage.cpp

static void qis_get_dots_per_meter(QIODevice *d,int* dots_m_x,int* dots_m_y)
{
  QString qs;
  bool ok;
  QTextStream ts(d);
  qs = ts .readLine();
  for(int i=0;i<3;i++)
  {
    qs = ts .readLine();
    if(qs.contains("#DOTS_PER_METER_X"))
    {
      qs = qs.right(qs.length()-17);
      qs = qs.stripWhiteSpace();
      *dots_m_x = qs.toInt(&ok);
      if(!ok)
        *dots_m_x = 0;
    }
    else if(qs.contains("#DOTS_PER_METER_Y"))
    {
      qs = qs.right(qs.length()-17);
      qs = qs.stripWhiteSpace();
      *dots_m_y = qs.toInt(&ok);
      if(!ok)
        *dots_m_y = 0;
    }
  }
  //if one value is invalid, set both values to 0
  if((*dots_m_x == 0) || (*dots_m_y == 0))
  {
    *dots_m_x = 0;
    *dots_m_y = 0;
  }
  d->reset();
}

static int qis_read_pbm_int( QIODevice *d )
{
  int	  c;
  int	  val = -1;
  bool  digit;
  const int buflen = 100;
  char  buf[buflen];

  while(TRUE)
  {
    if( (c=d->getch()) == EOF )		// end of file
      break;
    digit = isdigit(c);
    if ( val != -1 )
    {
      if ( digit )
      {
  	     val = 10*val + c - '0';
  	     continue;
      }
      else
      {
  	     if ( c == '#' )			// comment
  	       d->readLine( buf, buflen );
  	     break;
      }
    }
    if ( digit )				// first digit
	   val = c - '0';
    else if ( isspace(c) )
	   continue;
	  else if ( c == '#' )
	   d->readLine( buf, buflen );
	  else
	   break;
  }
  return val;
}

void qis_read_pbm_image( QImageIO *iio )	// read PBM image data
{
  const int	buflen = 300;
  char	buf[buflen];
  QIODevice  *d = iio->ioDevice();
  int		w, h, nbits, mcc, y;
  int		pbm_bpl;
  int red,green,blue;
  char	type;
  bool	raw;
  bool is_16_bit = false;
  QImage	image;
  int dotsmx;
  int dotsmy;
  QString params;
  int word_size;
  bool big_endian;

  qSysInfo (&word_size,&big_endian);
  qis_get_dots_per_meter(d,&dotsmx,&dotsmy);

  if ( d->readBlock( buf, 3 ) != 3 )			// read P[1-6]<white-space>
  {
    iio->setStatus(1);
  	return;
  }
  if ( !(buf[0] == 'P' && isdigit(buf[1]) && isspace(buf[2])) )
  {
    iio->setStatus(1);
  	return;
  }
  switch ( (type=buf[1]) )
  {
  	case '1':				// ascii PBM
  	case '4':				// raw PBM
	    nbits = 1;
	    break;
	  case '2':				// ascii PGM
	  case '5':				// raw PGM
	    nbits = 8;
	    break;
	  case '3':				// ascii PPM
	  case '6':				// raw PPM
	    nbits = 32;
	    break;
	  default:
      iio->setStatus(1);
    	return;
  }
  raw = type >= '4';
  w = qis_read_pbm_int( d );			// get image width
  h = qis_read_pbm_int( d );			// get image height
  if ( nbits == 1 )
	  mcc = 0;				// ignore max color component
  else
	  mcc = qis_read_pbm_int( d );		// get max color component
  if ( w <= 0 || w > 32767 || h <= 0 || h > 32767 || mcc < 0)
  {
    iio->setStatus(1);
  	return;
  }

  int maxc = mcc;
  if ( maxc > 255 )
	  maxc = 255;
  if(mcc == 65535)
    is_16_bit = true;
  image.create( w, h, nbits, 0,
          		  nbits == 1 ? QImage::BigEndian :  QImage::IgnoreEndian );
  if ( image.isNull() )
  {
    iio->setStatus(1);
  	return;
  }

  //set resolution
  image.setDotsPerMeterX(dotsmx);
  image.setDotsPerMeterY(dotsmy);

  pbm_bpl = (nbits*w+7)/8;			// bytes per scanline in PBM

  if ( raw )
  {				// read raw data
	  if ( nbits == 32 )
    {			// type 6
      if(is_16_bit)
        pbm_bpl = 6*w;
      else
  	    pbm_bpl = 3*w;
	    uchar *buf24 = new uchar[pbm_bpl], *b;
	    QRgb  *p;
	    QRgb  *end;
	    for ( y=0; y<h; y++ )
      {
		    if ( d->readBlock( (char *)buf24, pbm_bpl ) != pbm_bpl )
        {
		      delete[] buf24;
          iio->setStatus(1);
          return;
		    }
	    	p = (QRgb *)image.scanLine( y );
    		end = p + w;
    		b = buf24;
    		while ( p < end )
        {
          if(is_16_bit)
          {
            if(!big_endian)
            {
              red = ((int(b[0]) << 8) + b[1]) / 256;
              green = ((int(b[2]) << 8) + b[3]) / 256;
              blue = ((int(b[4]) << 8) + b[5]) / 256;
            }
            else
            {
              red = ((int(b[1]) << 8) + b[0]) / 256;
              green = ((int(b[3]) << 8) + b[2]) / 256;
              blue = ((int(b[5]) << 8) + b[4]) / 256;
            }
		        *p++ = qRgb(red,green,blue);
		        b += 6;
          }
          else
          {
		        *p++ = qRgb(b[0],b[1],b[2]);
		        b += 3;
          }
		    }
	    }
	    delete[] buf24;
	  }
    else
    {				// type 4,5
      uchar* buf16 = 0;
      uchar* b;
      uchar* p;
      uchar* end;
      if(is_16_bit && type == '5')
      {
        pbm_bpl = 2*w;
   	    buf16 = new uchar[pbm_bpl];
        b = buf16;
      }
      for ( y=0; y<h; y++ )
      {
        if(is_16_bit && type == '5')
        {
		      if(d->readBlock((char*)buf16,pbm_bpl ) != pbm_bpl)
          {
		        delete[] buf16;
            iio->setStatus(1);
  	        return;
		      }
	    	  p = (uchar *)image.scanLine( y );
    		  end = p + w;
    		  b = buf16;
    		  while ( p < end )
          {
            if(is_16_bit)
            {
              if(!big_endian)
                red = ((int(b[0]) << 8) + b[1]) / 256;
              else
                red = ((int(b[1]) << 8) + b[0]) / 256;
		          *p++ = (uchar)red;
		          b += 2;
            }
          }
        }
        else
        {
		      if ( d->readBlock( (char *)image.scanLine(y), pbm_bpl )
			         != pbm_bpl )
          {
            iio->setStatus(1);
          	return;
          }
	      }
      }
      if(buf16)
        delete [] buf16;
	  }
  }
  else
  {					// read ascii data
	  register uchar *p;
	  int n;
	  for ( y=0; y<h; y++ )
    {
	    p = image.scanLine( y );
	    n = pbm_bpl;
	    if ( nbits == 1 )
      {
		    int b;
		    while ( n-- )
        {
		      b = 0;
		      for ( int i=0; i<8; i++ )
			      b = (b << 1) | (qis_read_pbm_int(d) & 1);
		      *p++ = b;
		    }
	    }
      else if ( nbits == 8 )
      {
		    if ( mcc == maxc )
        {
		      while ( n-- )
          {
			      *p++ = qis_read_pbm_int( d );
		      }
		    }
        else
        {
		      while ( n-- )
          {
			      *p++ = qis_read_pbm_int( d ) * maxc / mcc;
		      }
		    }
	    }
      else
      {				// 32 bits
		    n /= 4;
		    int r, g, b;
		    if ( mcc == maxc )
        {
		      while ( n-- )
          {
       			r = qis_read_pbm_int( d );
       			g = qis_read_pbm_int( d );
       			b = qis_read_pbm_int( d );
       			*((QRgb*)p) = qRgb( r, g, b );
       			p += 4;
		      }
		    }
        else
        {
		      while ( n-- )
          {
       			r = qis_read_pbm_int( d ) * maxc / mcc;
       			g = qis_read_pbm_int( d ) * maxc / mcc;
       			b = qis_read_pbm_int( d ) * maxc / mcc;
       			*((QRgb*)p) = qRgb( r, g, b );
       			p += 4;
  		    }
	     	}
	    }
	  }
  }

  if ( nbits == 1 )
  {				// bitmap
	  image.setNumColors( 2 );
	  image.setColor( 0, qRgb(255,255,255) ); // white
	  image.setColor( 1, qRgb(0,0,0) );	// black
  }
  else if ( nbits == 8 )
  {			// graymap
	  image.setNumColors( maxc+1 );
	  for ( int i=0; i<=maxc; i++ )
	    image.setColor( i, qRgb(i*255/maxc,i*255/maxc,i*255/maxc) );
  }

  iio->setImage( image );
  iio->setStatus( 0 );			// image ok
}
//Format PNM is supported directly; i. e. the image is
//converted to the correct sub-format.

void qis_write_pbm_image( QImageIO *iio )
{
  QIODevice* out = iio->ioDevice();
  QCString str;

  QImage  image  = iio->image();
  QCString format = iio->format();
  format = format.left(3);			// ignore RAW part
  bool gray = false;
  bool detached = false;
  if( format == "PBM" )
  {
    if(image.depth() > 1)
    {
      image.detach();
      image = image.convertDepth(1);
      detached = true;
    }
  }
  else if(format == "PGM")
  {
    gray = true;
    if(image.depth() < 8)
    {
      image.detach();
      image = image.convertDepth(8);
      detached = true;
    }
  }
  else if(format == "PPM")
  {
    gray = false;
    if(image.depth() < 8)
    {
      image.detach();
      image = image.convertDepth(8);
      detached = true;
    }
  }
  else if (format == "PNM")
  {
    if((image.depth() == 32) || (image.depth() == 8))
    {
      if(image.isGrayscale())
      {
        format = "PGM";
        gray = true;
      }
      else
      {
        format = "PPM";
      }
    }
    else
    {
      format = "PBM";
      if(image.depth() > 1)
      {
        image.detach();
	      image = image.convertDepth(1);
        detached = true;
      }
    }
  }
  else
  {
    //error
    iio->setStatus(1);
    return;
  }

  if ( image.depth() == 1 && image.numColors() == 2 )
  {
	  if ( qGray(image.color(0)) < qGray(image.color(1)) )
    {
	    // 0=dark/black, 1=light/white - invert
      if(!detached)
	      image.detach();
	    for ( int y=0; y<image.height(); y++ )
      {
		    Q_UINT8* p = image.scanLine(y);
		    Q_UINT8* end = p + image.bytesPerLine();
		    while ( p < end )
		      *p++ ^= 0xff;
	    }
	  }
  }

  int w = image.width();
  int h = image.height();

  str.sprintf("P\n%d %d\n", w, h);

  switch (image.depth())
  {
	  case 1:
    {
	    str.insert(1, '4');
	    if ((unsigned int)out->writeBlock(str, str.length()) != str.length())
      {
		    iio->setStatus(1);
		    return;
	    }
	    w = w = (w+7)/8;
	    for (int y=0; y<h; y++)
      {
		    Q_UINT8* line = image.scanLine(y);
		    if ( w != (int)out->writeBlock((char*)line,w) )
        {
		      iio->setStatus(1);
		      return;
		    }
	    }
	  }
	  break;

	  case 8:
    {
	    str.insert(1, gray ? '5' : '6');
	    str.append("255\n");
	    if ((unsigned int)out->writeBlock(str, str.length()) != str.length())
      {
		    iio->setStatus(1);
		    return;
	    }
      Q_UINT8* buf;
      int bpl;
      if(gray)
        bpl = w;
      else
        bpl = w*3;
      buf = new Q_UINT8 [bpl];
      if(buf)
      {
  	    for(int y=0; y<h; y++)
        {
          for(int x=0;x<w;x++)
          {
        		QRgb  rgb = image.pixel(x,y);
        		if(gray)
            {
              buf[x] = (Q_UINT8) qGray(rgb);
      		  }
            else
            {
           		buf[x*3] = (Q_UINT8) qRed(rgb);
           		buf[x*3+1] = (Q_UINT8) qGreen(rgb);
           		buf[x*3+2] = (Q_UINT8) qBlue(rgb);
     		    }
       		}
    	  	if ( bpl != (int)out->writeBlock((char*)buf, bpl) )
          {
            delete [] buf;
    		    iio->setStatus(1);
    		    return;
    		  }
  	    }
  	    delete [] buf;
      }
      else
      {
        iio->setStatus(1);
        return;
		  }
	  }
	  break;

  	case 32:
    {
      str.insert(1, gray ? '5' : '6');
      str.append("255\n");
      if ((uint)out->writeBlock(str, str.length()) != str.length())
      {
    		iio->setStatus(1);
  		  return;
  	  }
      QRgb rgb;
      Q_UINT8* buf;
      int bpl;
      if(gray)
        bpl = w;
      else
        bpl = w*3;
      buf = new Q_UINT8 [bpl];
      if(buf)
      {
        for(int y=0; y<h; y++)
        {
          for(int x=0;x<w;x++)
          {
        		rgb = image.pixel(x,y);
        		if(gray)
            {
              buf[x] = (Q_UINT8)qGray(rgb);
      		  }
            else
            {
           		buf[x*3] = (Q_UINT8) qRed(rgb);
           		buf[x*3+1] = (Q_UINT8) qGreen(rgb);
           		buf[x*3+2] = (Q_UINT8) qBlue(rgb);
     		    }
       		}
    	  	if ( bpl != (int)out->writeBlock((char*)buf, bpl) )
          {
            delete [] buf;
    		    iio->setStatus(1);
    		    return;
    		  }
    	  }
    	  delete [] buf;
      }
      else
      {
        iio->setStatus(1);
        return;
		  }
  	}
  }
  iio->setStatus(0);
}

