/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PQListView ARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include <QDirModel>
#include <QListView>
#include <QLineEdit>
#include <QVBoxLayout>
#include <QAction>
#include <QToolBar>
#include <QHBoxLayout>
#include <QPushButton>
#include <QModelIndex>
#include <QDrag>
#include <QMimeData>
#include <QMessageBox>
#include <QCompleter>

#include "navigator.h"
#include "config.h"


Navigator::Navigator(QWidget *parent) : BaseWidget(parent) 
{
	widget_type=NAVIGATOR;
	
	octave_connection=NULL;
	copy_ok=false; cut_ok=false;
	
	setWindowTitle(tr("Navigator"));
	setWindowIcon( QIcon(QString( ICON_PATH )+"/new_folder.png"));
	createActions();
	//createMenuBar();
	createToolBar();
	createLayout();
	createConnections();
	createContextMenu();

	centralWidget()->setLayout(layout);
	//show();
}

void Navigator::createActions() 
{
	newFolder = new QAction(QIcon(QString(ICON_PATH) + "/new_folder"), tr("&New folder"), this);
	exit = new QAction(QIcon(QString(ICON_PATH) + "/exit.png"), tr("salir"), this);

	cut = new QAction(QIcon(QString(ICON_PATH) + "/editcut"), tr("Cu&t"), this);
	copy = new QAction(QIcon(QString(ICON_PATH) + "/editcopy"), tr("&Copy"), this);
	paste = new QAction(QIcon(QString(ICON_PATH) + "/editpaste"), tr("&Paste"), this);

	back = new QAction(QIcon(QString(ICON_PATH) + "/back.png"), tr("&Back"), this);
	forward = new QAction(QIcon(QString(ICON_PATH) + "/forward.png"), tr("&Forward"), this);
	up = new QAction(QIcon(QString(ICON_PATH) + "/up.png"), tr("Up"), this);
	remove=new QAction(QIcon(QString(ICON_PATH) + "/button_cancel.png"), tr("Delete file"), this);
	rename=new QAction(tr("Rename"), this);
	/* */
}

/*void Navigator::createMenuBar() 
{
	menubar = new QMenuBar(this);

	QMenu *menu = menubar->addMenu("&File");
	menu->addAction(newFolder);
	menu->addAction(exit);

	menu = menubar->addMenu("&Edit");
	menu->addAction(cut);
	menu->addAction(copy);
	menu->addAction(paste);

	menu = menubar->addMenu("&Navigator");
	menu->addAction(back);
	menu->addAction(forward);
	menu->addAction(up);
}*/

void Navigator::setSession(Session *session)
{
	//Sets octave_connection to NULL to not change dir at creation
	//Uses last pwd in Octave.
	OctaveConnection *oc=octave_connection;
	octave_connection=NULL;
	
	BaseWidget::setSession(session);
	setProject(session->getProjectName());
	connect(session, SIGNAL(projectChanged(QString)), this, SLOT(setProject(QString)) );
	
	//Restore octave_connection
	octave_connection=oc;
}

void Navigator::createToolBar() 
{
	toolbar = addToolBar(tr("Navigation tools"));
	toolbar->setObjectName("Navigation tools");
	//toolbar->setMinimumHeight(40);
  	//toolbar->setMaximumHeight(40);

	toolbar->addAction(back);
	toolbar->addAction(forward);
	toolbar->addAction(up);
	toolbar->addSeparator();
	
	toolbar->addAction(cut);
	toolbar->addAction(copy);
	toolbar->addAction(paste);
	toolbar->addSeparator();
	
	toolbar->addAction(newFolder);

}

void Navigator::createLayout() 
{
	layout = new QVBoxLayout(centralWidget());
	layout->setSpacing(0);
	layout->setMargin(0);
	//layout->setMenuBar(menubar);
	//layout->setStretchFactor(toolbar, 0);
	//layout->addWidget(toolbar);

	url = new QComboBox(this);
	url->setEditable(true);
	url->setDuplicatesEnabled(false);
	url->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLengthWithIcon);
	QCompleter *completer = new QCompleter(this);
	completer->setModel(new QDirModel(completer));
	url->setCompleter(completer);
	
	go = new QPushButton(tr("Go"), this);
	go->setFixedSize(url->height(), url->height());

	QHBoxLayout *urlBox = new QHBoxLayout();
	urlBox->addWidget(url);
	urlBox->addSpacing(10);
	urlBox->addWidget(go);
	urlBox->addSpacing(10);

	layout->addLayout(urlBox);
	
	filter_comboBox=new QComboBox(this);
	filter_comboBox->setEditable(true);
	filter_comboBox->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLengthWithIcon);
	filter_comboBox->addItem("*.m");
	filter_comboBox->addItem("*");
	filter_comboBox->addItem("*.txt");
	filter_comboBox->setSizePolicy(QSizePolicy::MinimumExpanding,QSizePolicy::MinimumExpanding);
	
	QHBoxLayout *filterBox = new QHBoxLayout();
	QLabel *filters_label=new QLabel(tr("Filters: "));
	filterBox->addWidget(filters_label);
	filterBox->addWidget(filter_comboBox);
	layout->addLayout(filterBox);
	
	//treemodel = new QDirModel();
	//treemodel->setFilter(QDir::Dirs);

	listmodel = new QDirModel();
	listmodel->setResolveSymlinks(false);
	listmodel->setFilter(QDir::AllDirs|QDir::AllEntries|QDir::NoDotAndDotDot);
	listmodel->setNameFilters(QStringList() << "*.m");

	//QSplitter * splitter = new QSplitter(this);
	//splitter->setMinimumSize(600,300);
	
	//QString path = QDir::currentPath();

	//treeview = new QTreeView(this);
	//treeview->setModel(treemodel);
	//treeview->setRootIndex(treemodel->index(path));

	QString paths=get_config("navigator_start_path"), path;
	if(!paths.isEmpty())
	{
		QStringList values=paths.split("\n");
		url->addItems(values);
	}
	QModelIndex root = listmodel->index(path);
	
	if(!root.isValid())
	{
		path = QDir::currentPath();
		root = listmodel->index(path);
	}

	listview = new DragDropListView(this);
	listview->setModel(listmodel);
	listview->setRootIndex(root);
	listview->setSelectionMode(QAbstractItemView::ExtendedSelection);
	listview->setContextMenuPolicy(Qt::CustomContextMenu);
	listview->setMouseTracking(true);
	//listview->setCursor(Qt::OpenHandCursor);

	url->lineEdit()->setText(path);

	back->setEnabled(false);
	forward->setEnabled(false);

	if (!(listmodel->parent(root)).isValid())
		up->setEnabled(false);

	//splitter->addWidget(treeview);
	//splitter->setStretchFactor(splitter->indexOf(treeview), 1);
	//splitter->addWidget(listview);
	//splitter->setStretchFactor(splitter->indexOf(listview), 1);

	layout->addWidget(listview, 1);

	//statusbar = new QStatusBar(this);
	//layout->addWidget(statusbar);
}

QString Navigator::getNavigatorCurrentPath()
{
	QModelIndex root = listview->rootIndex();
	QString rootPath = listmodel->filePath(root);
	return rootPath;
}

Navigator::~Navigator()
{
	QMap<QString,QString> configuration;
	QString values=getNavigatorCurrentPath();
	QStringList repetidos; repetidos << values;
	for(int i=0;i<url->count() && i<10;i++)
	{
		QDir dir(url->itemText(i));
		if(!repetidos.contains(url->itemText(i)) && dir.exists())
		{
			values+="\n"+url->itemText(i);
			repetidos << url->itemText(i);
		}
	}
	configuration["navigator_start_path"]=values;
	set_config(configuration);
	
	if(project_name.isEmpty()) Projects::saveNavigatorPath("Empty", getNavigatorCurrentPath());
	else Projects::saveNavigatorPath(project_name, getNavigatorCurrentPath());
}

void Navigator::setProject(QString name)
{
	project_name=name;
	
	setNavigatorCurrentPath(Projects::navigatorPath(name));
}

void Navigator::createConnections()
{
	connect(listview, SIGNAL(clicked(const QModelIndex &)), this, SLOT(expanded(const QModelIndex &)));
	connect(listview, SIGNAL(entered(const QModelIndex &)), this, SLOT(change_cursor_cb(const QModelIndex &)));
	//connect(listview, SIGNAL(doubleClicked(const QModelIndex &)), this, SLOT(drag_start(const QModelIndex &)));
	connect(back, SIGNAL(triggered(bool)), this, SLOT(goBack(bool)));
	connect(forward, SIGNAL(triggered(bool)), this, SLOT(goForward(bool)));
	connect(up, SIGNAL(triggered(bool)), this, SLOT(goUp(bool)));
	//connect(url->lineEdit(), SIGNAL(returnPressed()), this, SLOT(urlButtonPressed()));
	connect(url, SIGNAL(activated(const QString & )), this, SLOT(urlButtonPressed(const QString &)));
	connect(go, SIGNAL(clicked(bool)), this, SLOT(urlButtonPressed(bool)));
	connect(filter_comboBox, SIGNAL(activated(const QString & )), this, SLOT(filter_cb(const QString &)));
	connect(newFolder, SIGNAL( triggered() ), this, SLOT( mkdir() ) );
	connect(listview, SIGNAL( customContextMenuRequested ( const QPoint & )  ), this, SLOT( contextMenu_cb( const QPoint & ) ) );
	connect(copy, SIGNAL( triggered() ), this, SLOT( copy_cb() ) );
	connect(cut, SIGNAL( triggered() ), this, SLOT( cut_cb() ) );
	connect(remove, SIGNAL( triggered() ), this, SLOT( remove_cb() ) );
	connect(paste, SIGNAL( triggered() ), this, SLOT( paste_cb() ) );
	connect(rename, SIGNAL( triggered() ), this, SLOT( rename_cb() ) );
}


void Navigator::setOctaveConnection(OctaveConnection *oc)
{
	octave_connection=oc;
}


void Navigator::setUrl(const QModelIndex &index)
{
	QString path = listmodel->filePath(index);
	url->lineEdit()->setText(path);
}

void Navigator::setUpActionState(const QModelIndex &index)
{
	// Disables up Action if index has no parent. If has Enables it
	if (!(listmodel->parent(index)).isValid())
		up->setEnabled(false);
	else 
		up->setEnabled(true);
}

void Navigator::expanded(const QModelIndex &index)
{
	setUpActionState(index);

	if (listmodel->isDir(index)) {
		QModelIndex root = listview->rootIndex();
		QString rootPath = listmodel->filePath(root);
		
		// Add current dir to back Action record
		backHistory.append(rootPath);
		
		// Enables back Action
		back->setEnabled(true);
		
		// Clean forward Action record
		int size = forwardHistory.size();
		
		if ( size > 0) {
			forwardHistory.remove(0, size);
		}

		// Disables forward Action
		forward->setEnabled(false);
		
		// Change current URL to index`s URL
		setUrl(index);
		
		// Change listview current dir
		listview->setRootIndex(index);
		listmodel->refresh(index);
		
		if(octave_connection!=NULL)
		{
			QFileInfo dirinfo = listmodel->fileInfo(index);

			octave_connection->command_enter("cd \""+dirinfo.absoluteFilePath()+"\"", true);
		}

	} else {
		//It´s a file... Do something!!
		//drag_start(index);
	}
}

/*
void Navigator::drag_start( const QModelIndex & index )
{
	QDrag *drag = new QDrag(this);
	QMimeData *mimeData = new QMimeData;

	mimeData->setText(listmodel->filePath(index));
	
	//printf("%s\n",listmodel->filePath(index).toLocal8Bit().data());
	
	drag->setMimeData(mimeData);
	drag->setPixmap( (QString(ICON_PATH) + "/filenew.png") );
	//drag->setDragCursor(QCursor(Qt::ClosedHandCursor).pixmap(), Qt::CopyAction);

	//Qt::DropAction dropAction =
	drag->start();
}
*/

void Navigator::goBack(bool /*checked*/)
{
	int size = backHistory.size();

	if (size > 0) {
		QString lastPath = backHistory.last();
		QModelIndex last = listmodel->index(lastPath);
		QModelIndex root = listview->rootIndex();
		QString rootPath = listmodel->filePath(root);
		
		// Disables back Action if there is no more dirs saved
		if (--size == 0)
			back->setEnabled(false);
		
		// Add current dir to forward Action record
		forwardHistory.append(rootPath);

		// Enables forward Action
		forward->setEnabled(true);
	
		// Change current URL to last`s URL
		setUrl(last);

		// Change listview current dir
		listview->setRootIndex(last);
		listmodel->refresh(last);
		
		listview->scrollTo(listmodel->index(rootPath));
		
		setUpActionState(last);

		// Remove last dir from back Action record
		backHistory.remove(size);
	}
}

void Navigator::goForward(bool /*checked*/)
{
	int size = forwardHistory.size();

	if (size > 0) {
		QString lastPath = forwardHistory.last();
		QModelIndex last = listmodel->index(lastPath);
		QModelIndex root = listview->rootIndex();
		QString rootPath = listmodel->filePath(root);		

		// Disables forward Action if there is no more dirs saved
		if (--size == 0)
			forward->setEnabled(false);
		
		// Add current dir to back Action record
		backHistory.append(rootPath);

		// Enables back Action
		back->setEnabled(true);
	
		// Change current URL to last`s URL
		setUrl(last);

		// Change listview current dir
		listview->setRootIndex(last);
		listmodel->refresh(last);

		setUpActionState(last);

		// Remove last dir from forward Action record
		forwardHistory.remove(size);
	}
}

void Navigator::goUp(bool /*checked*/)
{
	QModelIndex root = listview->rootIndex();
	QModelIndex parent = listmodel->parent(root);

	emit expanded(parent);
}

void Navigator::urlButtonPressed(const QString & )
{
	go->click();
}

void Navigator::urlButtonPressed(bool /*checked*/)
{
	setNavigatorCurrentPath( url->lineEdit()->text() );
}

void Navigator::setNavigatorCurrentPath(QString path)
{
	if( path.isEmpty() ) return;
	
	QModelIndex next = listmodel->index(path);
	
	if (next.isValid()) {
		emit expanded(next);
	}
}

#include <QInputDialog>

void Navigator::mkdir()
{
	QString dir_name=QInputDialog::getText(this, tr("New Folder"), tr("Name:") );
	
	if( !dir_name.isEmpty() )
	{
		QModelIndex index = listview->rootIndex();
		QDir dir(listmodel->filePath(index));
		//printf("%s/%s\n",listmodel->filePath(index).toLocal8Bit().data(), dir_name.toLocal8Bit().data() );
		//listmodel->mkdir(index, dir_name);
		dir.mkdir(dir_name);
		listmodel->refresh(index);
	}
}

void Navigator::contextMenu_cb( const QPoint & /*pos*/)
{
	context_menu->popup(QCursor::pos () );
}

void Navigator::createContextMenu()
{
	context_menu=new QMenu(this);
	
	QList<QAction *> actions;
	actions.append(newFolder);
	actions.append(copy);
	actions.append(cut);
	actions.append(paste);
	actions.append(remove);
	actions.append(rename);
	
	context_menu->addActions(actions);
}

/**Returns selected files.
*/
static QList<QFileInfo> selected_files(QListView *listview, QDirModel *listmodel)
{
	QList<QFileInfo> list;
	QModelIndexList indexes=listview->selectionModel()->selectedIndexes();
	for(int i=0; i<indexes.size(); i++)
	{
		QModelIndex index=indexes.at(i);
		list.append(listmodel->fileInfo(index));
	}
	return list;
}

void Navigator::copy_cb()
{
	file_list=selected_files(listview, listmodel);
	copy_ok=true;
	cut_ok=false;
}

void Navigator::cut_cb()
{
	file_list=selected_files(listview, listmodel);
	copy_ok=false;
	cut_ok=true;
}

void Navigator::paste_cb()
{
	if(file_list.isEmpty()) return;
	
	FileActions *f=new FileActions(this);
	connect(f, SIGNAL(process_end(QThread*)), this, SLOT(process_end(QThread*)) );
	
	if(copy_ok)
	{
		QModelIndex index = listview->rootIndex();

		f->copy(file_list, listmodel->filePath(index));
		
		//copy_ok=false;
		
		//file_list.clear();
	}
	else if(cut_ok)
	{
		QModelIndex index = listview->rootIndex();

		f->cut(file_list, listmodel->filePath(index));
		
		cut_ok=false;
		
		file_list.clear();
	}
}

void Navigator::remove_cb()
{
	int ok=QMessageBox::question(NULL, tr("Remove file"), tr("Remove selected files?"), QMessageBox::Yes, QMessageBox::No);
	
	if(ok!=QMessageBox::Yes) return;
	
	file_list=selected_files(listview, listmodel);
	copy_ok=false;
	cut_ok=false;
	
	if(file_list.isEmpty()) return;
	
	FileActions *f=new FileActions(this);
	connect(f, SIGNAL(process_end(QThread*)), this, SLOT(process_end(QThread*)) );
	f->remove(file_list);
	
	file_list.clear();
}

void Navigator::rename_cb()
{
	QList<QFileInfo> file_list=selected_files(listview, listmodel);
	
	if( file_list.isEmpty() ) return;
	
	QString name=QInputDialog::getText(this, tr("Rename"), tr("Name:") );
	
	if( !name.isEmpty() )
	{
		QFileInfo fileinfo=file_list.at(0);
		
		QFile::rename(fileinfo.absoluteFilePath(), fileinfo.absolutePath()+"/"+name);
		
		listmodel->refresh();
	}
}

void Navigator::process_end(QThread *object)
{
	object->wait();
	listmodel->refresh();
	delete object;
}

void Navigator::filter_cb(const QString &s)
{
	listmodel->setNameFilters(QStringList() << s);
}

void Navigator::change_cursor_cb(const QModelIndex &index)
{
	if(listmodel->isDir(index))
		listview->setCursor(Qt::ArrowCursor);
	else
		listview->setCursor(Qt::OpenHandCursor);
}

FileActions::FileActions(QObject *parent):QThread(parent)
{
	window=new QWidget;
	window->setWindowTitle(tr("Processing files..."));
	
	QVBoxLayout *layout=new QVBoxLayout;
	
	label=new QLabel(window);
	layout->addWidget(label);
	label->setText(tr("Starting process..."));
	label->show();
	
	cancel_button=new QPushButton(window);
	cancel_button->setText(tr("Cancel"));
	layout->addWidget(cancel_button);
	cancel_button->show();
	
	window->setLayout(layout);
	
	connect(this, SIGNAL(file_name(QString)), this, SLOT(set_label_text(QString)) );
	connect(cancel_button, SIGNAL(clicked()), this, SLOT(stop()) );
	
	copy_ok=move_ok=remove_ok=stop_ok=false;
}

FileActions::~FileActions()
{
	delete window;
}

void FileActions::stop()
{
	stop_ok=true;
}

void FileActions::copy(QList<QFileInfo> list, QString path)
{
	this->list=list;
	this->path=path;
	
	copy_ok=true;
	
	window->show();
	
	start();
}

void FileActions::cut(QList<QFileInfo> list, QString path)
{
	this->list=list;
	this->path=path;
	
	move_ok=true;
	
	window->show();
	
	start();
}

void FileActions::remove(QList<QFileInfo> list)
{
	this->list=list;
	remove_ok=true;
	
	window->show();
	
	start();
}

void FileActions::run()
{
	for(int i=0;i<list.size() && !stop_ok;i++)
	{
		QFileInfo file=list.at(i);
		
		if(copy_ok) copy(file,path);
		else if(move_ok) cut(file,path);
		else if(remove_ok) remove(file);
	}
	
	emit process_end(this);
	copy_ok=move_ok=remove_ok=stop_ok=false;
}

void FileActions::set_label_text(QString text)
{
	label->setText("Copy: File "+text);
}

void FileActions::copy(QFileInfo file, QString path)
{
	QString file_path=file.absoluteFilePath();
	emit file_name(file_path);
	#if defined Q_OS_WIN32
		system(("copy \""+file_path+"\" \""+path+"\"").toLocal8Bit().data() );
	#else
		system(("cp -fRp \""+file_path+"\" \""+path+"\"").toLocal8Bit().data() );
	#endif
}

void FileActions::cut(QFileInfo file, QString path)
{
	QString file_path=file.absoluteFilePath();
	emit file_name(file_path);
	#if defined Q_OS_WIN32
		system(("move \""+file_path+"\" \""+path+"\"").toLocal8Bit().data() );
	#else
		system(("mv -f \""+file_path+"\" \""+path+"\"").toLocal8Bit().data() );
	#endif
}

void FileActions::remove(QFileInfo file)
{
	QString file_path=file.absoluteFilePath();
	emit file_name(file_path);
	#if defined Q_OS_WIN32
		system(("del \""+file_path+"\"").toLocal8Bit().data() );
	#else
		system(("rm -fR \""+file_path+"\"").toLocal8Bit().data() );
	#endif
}

DragDropListView::DragDropListView(QWidget *parent):QListView(parent)
{
}

void DragDropListView::mousePressEvent(QMouseEvent *event)
{
	QModelIndex index=indexAt( event->pos() );
	if (event->button() == Qt::LeftButton &&  index.isValid() )
	{
		QDirModel *listmodel=(QDirModel *)model();
		if( listmodel->isDir(index) )
		{
			QListView::mousePressEvent(event);
			return;
		}
		
		QDrag *drag = new QDrag(this);
		QMimeData *mimeData = new QMimeData;
		
		mimeData->setText(listmodel->filePath(index));
		
		//printf("%s\n",listmodel->filePath(index).toLocal8Bit().data());
		
		drag->setMimeData(mimeData);
		drag->setPixmap( (QString(ICON_PATH) + "/filenew.png") );
		//drag->setDragCursor(QCursor(Qt::ClosedHandCursor).pixmap(), Qt::CopyAction);
	
		/*Qt::DropAction dropAction = */drag->start();
	}
}

BaseWidget *Navigator::copyBaseWidget(QWidget * parent )
{
	Navigator *bw=new Navigator(parent);
	bw->setSession(session);
	bw->setOctaveConnection(octave_connection);
	
	return bw;
}


