/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include <QBoxLayout>
#include <QToolBar>
#include <QIcon>
#include <QWorkspace>
#include <QFileDialog>
#include <QTextStream>
#include <QMessageBox>
#include <QTextCursor>
#include "editor.h"
#include "config.h"
#include "search_dialog.h"

Editor::Editor(QWidget *parent): BaseWidget(parent), search_dialog(this)
{
  connect(&search_dialog, SIGNAL(search_signal()), this, SLOT(search()));
  connect(&search_dialog, SIGNAL(replace_signal()), this, SLOT(replace()));
  // Title
  setWindowTitle(tr("Editor"));
  setWindowIcon(QIcon(QString(ICON_PATH) + "/xedit.png"));

  // Layout
  QVBoxLayout *vLayout = new QVBoxLayout;
  setLayout(vLayout);

  resize(500, 400);

  // Toolbar
  toolBar = new QToolBar(this);
  toolBar->setOrientation(Qt::Horizontal);
  toolBar->setMinimumHeight(40);
  toolBar->setMaximumHeight(40);
  connect(toolBar, SIGNAL(actionTriggered(QAction*)),
	  this, SLOT(toolbar_action(QAction*)));
  vLayout->addWidget(toolBar);
  toolBar->show();

  // Toolbar buttons
  actionOpen = toolBar->addAction(QIcon(QString(ICON_PATH) + "/fileopen"),
				  tr("Open"));
  actionSave = toolBar->addAction(QIcon(QString(ICON_PATH) + "/filesave"),
				  tr("Save"));
  actionSaveAs = toolBar->addAction(QIcon(QString(ICON_PATH) + "/filesaveas.png"),
				    tr("Save as"));
  toolBar->addSeparator();

  actionRun = toolBar->addAction(QIcon(QString(ICON_PATH) + "/run.png"),
				 tr("Run"));
  toolBar->addSeparator();

  actionUndo = toolBar->addAction(QIcon(QString(ICON_PATH) + "/undo.png"),
				  tr("Undo"));
  actionCut = toolBar->addAction(QIcon(QString(ICON_PATH) + "/editcut"),
				 tr("Cut"));
  actionCopy = toolBar->addAction(QIcon(QString(ICON_PATH) + "/editcopy"),
				  tr("Copy"));
  actionPaste = toolBar->addAction(QIcon(QString(ICON_PATH) + "/editpaste"),
				   tr("Paste"));

  toolBar->addSeparator();

  actionSearch = toolBar->addAction(QIcon(QString(ICON_PATH) + "/search.png"),
				    tr("Search and replace"));

  // Editbox
  textEdit = new QTextEdit(this);
  textEdit->setUndoRedoEnabled(true);
  vLayout->addWidget(textEdit);
  syntax = new Syntax(textEdit);
  syntax->load(QString(CONFIG_PATH) + "/octave.xml");
  textEdit->setTabStopWidth(32);
  textEdit->show();
}

void Editor::setOctaveConnection(OctaveConnection *oc)
{
  octave_connection = oc;
}

void Editor::toolbar_action(QAction *action)
{
  QStringList filters;

  filters << "Octave (*.m; *.M)"
	  << "Plain text (*.txt)"
	  << "All files (*)";

  if(action == actionOpen){
    // Open
    QFileDialog openDialog(this, Qt::Dialog);

    openDialog.setAcceptMode(QFileDialog::AcceptOpen);
    openDialog.setDefaultSuffix("m");
    openDialog.setFilters(filters);

    if(openDialog.exec() == QDialog::Accepted)
    {
      FILE *fl;

      path = openDialog.selectedFiles().first();
      fl = fopen(path.toLocal8Bit().constData(), "rt");
      if(fl)
      {
	QTextStream *stream = new QTextStream(fl);
	textEdit->document()->setPlainText(stream->readAll());
	delete stream;
	fclose(fl);
	setWindowTitle(path.split("/").last());
      }else{
	//QMessageBox msg(QMessageBox::Critical, tr("Error"),
	//		path + " can not be opened");
	//msg.exec();
	QMessageBox::critical(NULL, tr("Error"), path + " can not be opened");
	path = "";
      }
    }
  }else if(action == actionSave && !path.isEmpty()){  
    // Save
    FILE *fl;
    fl = fopen(path.toLocal8Bit().constData(), "wt");

    if(fl)
    {
      QTextStream *stream = new QTextStream(fl);

      (*stream) << textEdit->document()->toPlainText();

      delete stream;
      fclose(fl);
    }else{
      //QMessageBox msg(QMessageBox::Critical, tr("Error"),"Can not be saved");
      //msg.exec();
      QMessageBox::critical(NULL, tr("Error"), "Can not be saved");
    }

  }else if(action == actionSaveAs || (action == actionSave && path.isEmpty())){
    // Save as
    QFileDialog saveDialog(this, Qt::Dialog);

    saveDialog.setAcceptMode(QFileDialog::AcceptSave);
    saveDialog.setDefaultSuffix("m");
    saveDialog.setFilters(filters);

    if(saveDialog.exec() == QDialog::Accepted)
    {
      FILE *fl;

      path = saveDialog.selectedFiles().first();
      fl = fopen(path.toLocal8Bit().constData(), "wt");
      if(fl)
      {
	QTextStream *stream = new QTextStream(fl);
	(*stream) << textEdit->document()->toPlainText();
	delete stream;
	fclose(fl);
	setWindowTitle(path.split("/").last());
      }else{
	//QMessageBox msg(QMessageBox::Critical, tr("Error"), path + "can not be saved");
	//msg.exec();
	QMessageBox::critical(NULL, tr("Error"), path + "can not be saved");
	path = "";
      }
    }
  }else if(action == actionRun){
    // Run
    octave_connection->command_enter(textEdit->document()->toPlainText());
  }else if(action == actionUndo){
    // Undo
    textEdit->document()->undo();
  }else if(action == actionCut){
    // Cut
    textEdit->cut();
  }else if(action == actionCopy){
    // Copy
    textEdit->copy();
  }else if(action == actionPaste){
    // Paste
    textEdit->paste();
  }else if(action == actionSearch){
    search_dialog.show();
  }else{
    printf("Unhandled action\n");
  }
}

void Editor::search()
{
  QString search, replace;
  QTextCursor cursor;

  // Strings
  search = search_dialog.searchString();
  replace = search_dialog.replaceString();

  // Flags
  QTextDocument::FindFlags flags;
  if(search_dialog.caseSensitive())
    flags |= QTextDocument::FindCaseSensitively;
  if(search_dialog.wholeWords())
    flags |= QTextDocument::FindWholeWords;

  // Search
  cursor = textEdit->textCursor();
  if(search_dialog.searchStringIsRegExp())
  {
    // Search string is a regular expression
    QRegExp searchReg(search);
    
    //cursor = textEdit->document()->find(searchReg, cursor, flags);
    cursor = textEdit->document()->find(search, cursor, flags);
  }else{
    // Search string is not a regular expression
    cursor = textEdit->document()->find(search, cursor, flags);
  }

  textEdit->setTextCursor(cursor);
}

void Editor::replace()
{
  QTextCursor cursor = textEdit->textCursor();

  if(!cursor.hasSelection())
    cursor.insertText(search_dialog.replaceString());

  search();
}
