/****************************************************************************
**
** Copyright (C) 2012 Digia Plc and/or its subsidiary(-ies).
** Contact: http://www.qt-project.org/legal
**
** This file is part of Qt Creator.
**
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Digia.  For licensing terms and
** conditions see http://qt.digia.com/licensing.  For further information
** use the contact form at http://qt.digia.com/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Digia gives you certain additional
** rights.  These rights are described in the Digia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
****************************************************************************/

#include <AST.h>
#include <ASTVisitor.h>
#include <ASTPatternBuilder.h>
#include <ASTMatcher.h>
#include <Control.h>
#include <Scope.h>
#include <TranslationUnit.h>
#include <Literals.h>
#include <Symbols.h>
#include <Names.h>
#include <CoreTypes.h>
#include <CppDocument.h>
#include <SymbolVisitor.h>
#include <Overview.h>

#include <QFile>
#include <QList>
#include <QCoreApplication>
#include <QStringList>
#include <QFileInfo>
#include <QTime>
#include <QDebug>

#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <iostream>
#ifdef __GNUC__
#  include <cxxabi.h>
#endif

using namespace CPlusPlus;

class ASTDump: protected ASTVisitor
{
public:
    ASTDump(TranslationUnit *unit)
        : ASTVisitor(unit) {}

    void operator()(AST *ast) {
        QByteArray basename = translationUnit()->fileName();
        int dotIdx = basename.lastIndexOf('.');
        if (dotIdx != -1)
            basename.truncate(dotIdx);
        basename.append(".ast.dot");
        out.open(basename.constData());

        out << "digraph AST { ordering=out;" << std::endl;
        // std::cout << "rankdir = \"LR\";" << std::endl;

        generateTokens();
        accept(ast);

        typedef QPair<QByteArray, QByteArray> Pair;

        foreach (const Pair &conn, _connections)
            out << conn.first.constData() << " -> " << conn.second.constData() << std::endl;

        alignTerminals();

        out << "}" << std::endl;
        out.close();
        std::cout << basename.constData() << std::endl;
    }

    // the following file can be generated by using:
    //    generate-ast <path to cpp stuff> <path to dumpers.inc>
#include "dumpers.inc"

protected:
    void alignTerminals() {
        out<<"{ rank=same;" << std::endl;
        foreach (const QByteArray &terminalShape, _terminalShapes) {
            out << "  " << std::string(terminalShape) << ";" << std::endl;
        }
        out<<"}"<<std::endl;
    }

    static QByteArray name(AST *ast) {
#ifdef __GNUC__
        QByteArray name = abi::__cxa_demangle(typeid(*ast).name(), 0, 0, 0) + 11;
        name.truncate(name.length() - 3);
#else
        QByteArray name = typeid(*ast).name();
#endif
        return name;
    }

    QByteArray terminalId(unsigned token)
    { return 't' + QByteArray::number(token); }

    void terminal(unsigned token, AST *node) {
        _connections.append(qMakePair(_id[node], terminalId(token)));
    }

    void generateTokens() {
        for (unsigned token = 1; token < translationUnit()->tokenCount(); ++token) {
            if (translationUnit()->tokenKind(token) == T_EOF_SYMBOL)
                break;

            QByteArray t;

            t.append(terminalId(token));
            t.append(" [shape=rect label = \"");
            t.append(spell(token));
            t.append("\"]");

            if (token > 1) {
                t.append("; ");
                t.append(terminalId(token - 1));
                t.append(" -> ");
                t.append(terminalId(token));
                t.append(" [arrowhead=\"vee\" color=\"transparent\"]");
            }

            _terminalShapes.append(t);
        }
    }

    virtual void nonterminal(AST *ast) {
        accept(ast);
    }

    virtual void node(AST *ast) {
        out << _id[ast].constData() << " [label=\"" << name(ast).constData() << "\"];" << std::endl;
    }

    virtual bool preVisit(AST *ast) {
        static int count = 1;
        const QByteArray id = 'n' + QByteArray::number(count++);
        _id[ast] = id;


        if (! _stack.isEmpty())
            _connections.append(qMakePair(_id[_stack.last()], id));

        _stack.append(ast);

        node(ast);

        return true;
    }

    virtual void postVisit(AST *) {
        _stack.removeLast();
    }

private:
    QHash<AST *, QByteArray> _id;
    QList<QPair<QByteArray, QByteArray> > _connections;
    QList<AST *> _stack;
    QList<QByteArray> _terminalShapes;
    std::ofstream out;
};

class SymbolDump: protected SymbolVisitor
{
public:
    SymbolDump(TranslationUnit *unit)
        : translationUnit(unit)
    {
        o.setShowArgumentNames(true);
        o.setShowFunctionSignatures(true);
        o.setShowReturnTypes(true);
    }

    void operator()(Symbol *s) {
        QByteArray basename = translationUnit->fileName();
        int dotIdx = basename.lastIndexOf('.');
        if (dotIdx != -1)
            basename.truncate(dotIdx);
        basename.append(".symbols.dot");
        out.open(basename.constData());

        out << "digraph Symbols { ordering=out;" << std::endl;
        // std::cout << "rankdir = \"LR\";" << std::endl;
        accept(s);

        for (int i = 0; i < _connections.size(); ++i) {
            QPair<Symbol*,Symbol*> connection = _connections.at(i);
            QByteArray from = _id.value(connection.first);
            if (from.isEmpty())
                from = name(connection.first);
            QByteArray to = _id.value(connection.second);
            if (to.isEmpty())
                to = name(connection.second);
            out << from.constData() << " -> " << to.constData() << ";" << std::endl;
        }

        out << "}" << std::endl;
        out.close();
        std::cout << basename.constData() << std::endl;
    }

protected:
    QByteArray name(Symbol *s) {
#ifdef __GNUC__
        QByteArray result = abi::__cxa_demangle(typeid(*s).name(), 0, 0, 0) + 11;
#else
        QByteArray result = typeid(*s).name();
#endif
        if (s->identifier()) {
            result.append("\\nid: ");
            result.append(s->identifier()->chars());
        }
        if (s->isDeprecated())
            result.append("\\n(deprecated)");

        return result;
    }

    virtual bool preVisit(Symbol *s) {
        static int count = 0;
        QByteArray nodeId("s");
        nodeId.append(QByteArray::number(++count));
        _id[s] = nodeId;

        if (!_stack.isEmpty())
            _connections.append(qMakePair(_stack.last(), s));

        _stack.append(s);

        return true;
    }

    virtual void postVisit(Symbol *) {
        _stack.removeLast();
    }

    virtual void simpleNode(Symbol *symbol) {
        out << _id[symbol].constData() << " [label=\"" << name(symbol).constData() << "\"];" << std::endl;
    }

    virtual bool visit(Class *symbol) {
        const char *id = _id.value(symbol).constData();
        out << id << " [label=\"";
        if (symbol->isClass()) {
            out << "class";
        } else if (symbol->isStruct()) {
            out << "struct";
        } else if (symbol->isUnion()) {
            out << "union";
        } else {
            out << "UNKNOWN";
        }

        out << "\\nid: ";
        if (symbol->identifier()) {
            out << symbol->identifier()->chars();
        } else {
            out << "NO ID";
        }
        if (symbol->isDeprecated())
            out << "\\n(deprecated)";
        out << "\"];" << std::endl;

        return true;
    }

    virtual bool visit(UsingNamespaceDirective *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(UsingDeclaration *symbol) { simpleNode(symbol); return true; }

    virtual bool visit(Declaration *symbol) {
        out << _id[symbol].constData() << " [label=\"";
        out << "Declaration\\n";
        out << qPrintable(o(symbol->name()));
        out << ": ";
        out << qPrintable(o(symbol->type()));
        if (symbol->isDeprecated())
            out << "\\n(deprecated)";
        if (Function *funTy = symbol->type()->asFunctionType()) {
            if (funTy->isPureVirtual())
                out << "\\n(pure virtual)";
            else if (funTy->isVirtual())
                out << "\\n(virtual)";

            if (funTy->isSignal())
                out << "\\n(signal)";
            if (funTy->isSlot())
                out << "\\n(slot)";
            if (funTy->isInvokable())
                out << "\\n(invokable)";
        }
        out << "\"];" << std::endl;

        return true;
    }

    virtual bool visit(Argument *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(TypenameArgument *symbol) { simpleNode(symbol); return true; }

    virtual bool visit(BaseClass *symbol) {
        out << _id[symbol].constData() << " [label=\"BaseClass\\n";
        out << qPrintable(o(symbol->name()));
        if (symbol->isDeprecated())
            out << "\\n(deprecated)";
        out << "\"];" << std::endl;

        return true;
    }

    virtual bool visit(Enum *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(Function *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(Namespace *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(Block *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ForwardClassDeclaration *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ObjCBaseClass *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ObjCBaseProtocol *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ObjCClass *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ObjCForwardClassDeclaration *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ObjCProtocol *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ObjCForwardProtocolDeclaration *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ObjCMethod *symbol) { simpleNode(symbol); return true; }
    virtual bool visit(ObjCPropertyDeclaration *symbol) { simpleNode(symbol); return true; }

private:
    TranslationUnit *translationUnit;
    QHash<Symbol *, QByteArray> _id;
    QList<QPair<Symbol *,Symbol*> >_connections;
    QList<Symbol *> _stack;
    std::ofstream out;
    Overview o;
};

int main(int argc, char *argv[])
{
    QCoreApplication app(argc, argv);

    QStringList files = app.arguments();
    files.removeFirst();

    foreach (const QString &fileName, files) {
        QFile file(fileName);
        if (! file.open(QFile::ReadOnly)) {
            std::cerr << "Cannot open \"" << qPrintable(fileName)
                      << "\", skipping it." << std::endl;
            continue;
        }

        const QByteArray source = file.readAll();
        file.close();

        Document::Ptr doc = Document::create(fileName);
        doc->control()->setDiagnosticClient(0);
        doc->setUtf8Source(source);
        doc->parse();

        doc->check();

        ASTDump dump(doc->translationUnit());
        dump(doc->translationUnit()->ast());

        SymbolDump dump2(doc->translationUnit());
        dump2(doc->globalNamespace());
    }

    return EXIT_SUCCESS;
}
