/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#include "singlemessagewgt.h"
#include "globals.h"

#include <QScrollBar>
#include <QGridLayout>
#include <QDateTime>

#include "inputtextwgt.h"
#include "chattextwgt.h"
#include "message.h"

SingleMessageWgt::SingleMessageWgt(const QString & title, const QString & message, const QHostAddress & destAddr, bool isIncoming, QWidget *parent) : QDialog    (parent)
{
  init(title, message, destAddr, isIncoming);
}

SingleMessageWgt::SingleMessageWgt(Message* msg, bool isIncoming, QWidget * parent) : QDialog    (parent)
{
  QDateTime dt;
  dt.setTime_t(msg->receiveTime());

  init(tr("Message from ") + msg->userName() + QString("[") + dt.toString("hh:mm:ss") + "] ",
       msg->msg(),
       QHostAddress(msg->srcIp()),
       isIncoming);
}

//\*****************************************************************************
SingleMessageWgt::~SingleMessageWgt()
{
  qDebug("[SingleMessageWgt::~SingleMessageWgt]\n");
}
//\*****************************************************************************
void SingleMessageWgt::retranslate()
{
  m_closeBtn       ->setText(tr("&Close"       ));
  m_sendBtn        ->setText(tr("&Send"        ));
  m_replyBtn       ->setText(tr("&Reply"       ));
  m_replyQuotedBtn ->setText(tr("Reply &Quoted"));
  }
//\*****************************************************************************
void SingleMessageWgt::slot_reply()
{
   m_sendBtn ->show();
   m_inputText->show();

   m_replyQuotedBtn->hide();
   m_replyBtn      ->hide();

   setWindowTitle(tr("Replying to ") + m_title + "(" + m_destAddr.toString() + ")");

   m_inputText->setFocus();
}
//\*****************************************************************************
void SingleMessageWgt::slot_replyQuoted()
{
  QString str = QString(">") + m_messageIn->toPlainText();
  m_quot = true;

  str.replace("\n", "\n>");
  if(str[str.size() - 1] == ('>') && str.size() > 1)
     str.resize(str.size() - 1);
  else
    str.append("\n");

  m_messageIn->text()->setFontItalic(true);
  m_messageIn->text()->setPlainText(str);

  m_sendBtn ->show();
  m_inputText->show();

  m_replyQuotedBtn->hide();
  m_replyBtn      ->hide();

  setWindowTitle(tr("Replying to ") + m_title + "(" + m_destAddr.toString() + ")");

  m_inputText->setFocus();
}
//\*****************************************************************************
void SingleMessageWgt::slot_selfDestroy()
{
  hide();
  this->~SingleMessageWgt();
}
//\*****************************************************************************
void SingleMessageWgt::slot_sendMessage()
{
  qDebug("[SingleMessageWgt::slot_sendMessage]: addr = '%s'", m_destAddr.toString().toLocal8Bit().data());

  if(m_quot)
    emit singleMessage(m_messageIn->toPlainText() + m_inputText->toPlainText(), m_destAddr);
  else
    emit singleMessage(m_inputText->toPlainText(), m_destAddr);

  slot_selfDestroy();
}
//\*****************************************************************************
void SingleMessageWgt::init(const QString & title, const QString & message, const QHostAddress & destAddr, bool isIncoming)
{
  m_isIncoming = isIncoming;
  m_quot       = false;
  m_destAddr   = destAddr;
  m_title      = title;

  QGridLayout*  grid;

  grid             = new QGridLayout(this);
  m_closeBtn       = new QPushButton(this);
  m_sendBtn        = new QPushButton(this);
  m_replyBtn       = new QPushButton(this);
  m_replyQuotedBtn = new QPushButton(this);
  m_messageIn      = new ChatTextWgt (this);
  m_inputText      = new InputTextWgt(this);

  m_messageIn->setMinimumWidth(400);
  m_inputText->setMinimumWidth(400);

  if(m_isIncoming)
  {
    grid->addWidget(m_messageIn     , 0, 0, 1, 3);
    grid->addWidget(m_replyBtn      , 1, 0, 1, 1);
    grid->addWidget(m_replyQuotedBtn, 1, 1, 1, 1);
    grid->addWidget(m_closeBtn      , 1, 2, 1, 1);
    grid->addWidget(m_sendBtn       , 1, 0, 1, 2);
    grid->addWidget(m_inputText     , 2, 0, 1, 3);

    m_sendBtn  ->hide();
    m_inputText->hide();

    m_messageIn->setMsg(message);
  }
  else
  {
    grid->addWidget(m_inputText     , 0, 0, 1, 2);
    grid->addWidget(m_sendBtn       , 1, 0, 1, 1);
    grid->addWidget(m_closeBtn      , 1, 1, 1, 1);

    m_replyBtn      ->hide();
    m_replyQuotedBtn->hide();
    m_messageIn     ->hide();

    m_inputText->setFocus();
  }

  grid->setMargin(2);

  m_messageIn->setFocus();

  setOrientation(Qt::Vertical);
  resize(400, 150);

  connect(m_replyBtn      , SIGNAL(clicked()) , this, SLOT(slot_reply      ()));
  connect(m_replyQuotedBtn, SIGNAL(clicked()) , this, SLOT(slot_replyQuoted()));
  connect(m_closeBtn      , SIGNAL(clicked()) , this, SLOT(slot_selfDestroy()));
  connect(m_sendBtn       , SIGNAL(clicked()) , this, SLOT(slot_sendMessage()));
  connect(m_inputText     , SIGNAL(wantSend()), this, SLOT(slot_sendMessage()));

  setWindowTitle(m_title + "(" + destAddr.toString() + ")");

  qDebug("[SingleMessageWgt::SingleMessageWgt]: addr = '%s'", m_destAddr.toString().toLocal8Bit().data());

  retranslate();

  this->show();
}
