/***************************************************************************
 $RCSfile: timport.cpp,v $
                             -------------------
    cvs         : $Id: timport.cpp 433 2008-01-12 08:58:57Z martin $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "app.h"
#include "timport.h"
#include "standingorder.h"

#include <gwenhywfar/debug.h>
#include <gwenhywfar/gui.h>




TransactionImporter::DayData::DayData(int year, int month, int d)
:_year(year)
,_month(month)
,_day(d)
,_transactions(0){
  _transactions=new std::list<RefPointer<Transaction> >();


}


TransactionImporter::DayData::DayData(const TransactionImporter::DayData &d)
:_year(d._year)
,_month(d._month)
,_day(d._day)
,_transactions(0){
  _transactions=new std::list<RefPointer<Transaction> >();
}



TransactionImporter::DayData::~DayData(){
  DBG_DEBUG(0, "Deleting day");
  if (_transactions) {
    delete _transactions;
  }
}



void TransactionImporter::DayData::addTransaction(RefPointer<Transaction> t){
  assert(_transactions);
  _transactions->push_back(t);
}



void
TransactionImporter::DayData::replaceTransactions(std::list<RefPointer<Transaction> > *tl){
  delete _transactions;
  _transactions=tl;
}





TransactionImporter::AccountData::AccountData(Account *a)
:_account(a) {
}



TransactionImporter::AccountData::AccountData
(const TransactionImporter::AccountData &a)
:_account(a._account) {

}



TransactionImporter::AccountData::~AccountData(){
  std::list<DayData*>::iterator it;

  DBG_DEBUG(0, "Deleting AccountData");
  for (it=_newDayList.begin();
       it!=_newDayList.end();
       it++)
    delete *it;

}



const std::string &TransactionImporter::AccountData::accountNumber(){
  return _account->getAccountNumber();
}



const std::string &TransactionImporter::AccountData::bankCode(){
  return _account->getBankCode();
}



const std::string &TransactionImporter::AccountData::accountName(){
  return _account->getAccountName();
}



TransactionImporter::DayData*
TransactionImporter::AccountData::getNewDayData(int year,
                                                int month,
                                                int day){
  std::list<DayData*>::iterator it;

  for (it=_newDayList.begin(); it!=_newDayList.end(); it++) {
    if ((*it)->getYear()==year &&
        (*it)->getMonth()==month &&
        (*it)->getDay()==day)
      return *it;
  }
  _newDayList.push_back(new DayData(year, month, day));
  return _newDayList.back();
}








TransactionImporter::TransactionImporter(App *kb,
					 AB_IMEXPORTER_CONTEXT *ctx,
                                         uint32_t flags)
:_app(kb)
,_context(ctx)
,_flags(flags){

#if 0
  GWEN_DB_NODE *db;

  db=GWEN_DB_Group_new("context");
  AB_ImExporterContext_toDb(ctx, db);
  GWEN_DB_WriteFile(db, "/tmp/context.ctx", GWEN_DB_FLAGS_DEFAULT);
  GWEN_DB_Group_free(db);
#endif

}



TransactionImporter::~TransactionImporter(){
  std::list<AccountData*>::iterator it;

  for (it=_accountDataList.begin();
       it!=_accountDataList.end();
       it++)
    delete *it;
}



Account*
TransactionImporter::askUserForAccount(const std::string &bankCode,
				       const std::string &accountNumber,
				       const std::string &accountName){
  return 0;
}



GWEN_TIME *TransactionImporter::askUserForDate(const Transaction *t,
                                               const GWEN_TIME *ti){
  return 0;
}



Payee *TransactionImporter::askUserForPayee(RefPointer<Transaction> t,
                                            bool &aborted,
                                            bool &askNoMore){
  aborted=false;
  askNoMore=true;
  return 0;
}



Category *TransactionImporter::askUserForCategory(RefPointer<Transaction> t,
                                                  bool &aborted,
                                                  bool &askNoMore) {
  aborted=false;
  askNoMore=true;
  return 0;
}



TransactionImporter::DupeCheckResult
TransactionImporter::askUserForDuplicate(Account *a,
                                         RefPointer<Transaction> tnew,
                                         RefPointer<Transaction> told,
                                         const std::list<RefPointer<Transaction> > &dnl,
                                         const std::list<RefPointer<Transaction> > &dol,
                                         bool &askNoMore){
  return DupeCheck_Error;
}



bool
TransactionImporter::presentTransactionsToUser(std::list<RefPointer<Transaction> > &toAdd,
					       std::list<RefPointer<Transaction> > &toDel) {
  return true;
}




bool TransactionImporter::sortNewTransactions() {
  AB_IMEXPORTER_ACCOUNTINFO *ai;

  DBG_INFO(0, "Sorting transactions...");
  ai=AB_ImExporterContext_GetFirstAccountInfo(_context);
  if (!ai) {
    DBG_INFO(0, "No accounts");
  }
  while(ai) {
    AccountData *currentAccount;
    const AB_TRANSACTION *t;
    const GWEN_TIME *lastDate;
    DayData *currentDay;

    DBG_DEBUG(0, "Checking account (%s/%s)",
              AB_ImExporterAccountInfo_GetBankCode(ai),
              AB_ImExporterAccountInfo_GetAccountNumber(ai));
    currentAccount=
      getAccountData(AB_ImExporterAccountInfo_GetBankCode(ai),
		     AB_ImExporterAccountInfo_GetAccountNumber(ai),
		     AB_ImExporterAccountInfo_GetAccountName(ai));
    if (!currentAccount) {
      // TODO: ask user whether he wants to abort
      return false;
    }

    lastDate=0;
    currentDay=0;
    t=AB_ImExporterAccountInfo_GetFirstTransaction(ai);
    while(t) {
      Transaction *tr;
      const GWEN_TIME *ti;
      int d, month, year;

      tr=new Transaction(t);

      if (tr) {
        tr->gatherInfoFromPurpose();
	tr->setLocalBankCode(currentAccount->bankCode());
	tr->setLocalAccountNumber(currentAccount->accountNumber());

        ti=tr->getDate();
        if (!ti)
          ti=tr->getValutaDate();
        if (!ti)
          // still no date, ask the user about the date
          ti=askUserForDate(tr, lastDate);
        if (!ti) {
          DBG_ERROR(0, "No date, skip transaction");
          // TODO: Ask user for abort
        }
        else {
          lastDate=ti;
          if (GWEN_Time_GetBrokenDownUtcDate(ti, &d, &month, &year)) {
            DBG_ERROR(0, "Unable to break down date");
            // TODO: Ask user for abort
            return false;
          }
          else {
            // get corresponding day
            if (currentDay)
              if (currentDay->getDay()!=d ||
                  currentDay->getMonth()+1!=month ||
                  currentDay->getYear()!=year)
                currentDay=0;
            if (currentDay==0)
              currentDay=currentAccount->getNewDayData(year, month+1, d);
            assert(currentDay);
  
            // add transaction to that day
            currentDay->addTransaction(tr);
          }
        }
      }

      t=AB_ImExporterAccountInfo_GetNextTransaction(ai);
    } /* while */

    ai=AB_ImExporterContext_GetNextAccountInfo(_context);
  } /* while accounts */


  return true;
}







TransactionImporter::DupeCheckResult
TransactionImporter::checkForDuplicate(Account *a,
                                       RefPointer<Transaction> tnew,
                                       RefPointer<Transaction> told,
                                       std::list<RefPointer<Transaction> > &dnl,
                                       std::list<RefPointer<Transaction> > &dol,
                                       bool &dontAskAgain,
                                       TransactionImporter::DupeCheckResult lastRv){
  std::list<RefPointer<Transaction> >::iterator it;
  int newCount;
  int oldCount;

  /* count the occurrences of the new transaction in both the old and the
   * new list
   */
  DBG_DEBUG(0, "Equal, checking further");
  newCount=oldCount=0;

  if (_flags & TRANSACTION_IMPORTER_FLAGS_ASK_ALL_DUPES) {
    if (_flags & TRANSACTION_IMPORTER_FLAGS_FUZZY) {
      for (it=dol.begin(); it!=dol.end(); it++) {
        if ((*it).ref().matchFuzzy(tnew.ref(),
                                   _app->getSensitiveFuzzyThreshold())) {
	  oldCount++;
	}
      }
    } // if fuzzy
    else {
      for (it=dol.begin(); it!=dol.end(); it++) {
	if (tnew.ref().getHash()==(*it).ref().getHash()) {
	  oldCount++;
	}
      }
    } // if !fuzzy

    if (oldCount) {
      if (dontAskAgain)
	return lastRv;
      else
	return askUserForDuplicate(a, tnew, told, dnl, dol, dontAskAgain);
    }
    else {
      DBG_DEBUG(0, "Adding new");
      return DupeCheck_AddNew;
    }
  } // if askAllDupes
  else {
    for (it=dnl.begin(); it!=dnl.end(); it++) {
      if (tnew.ref().getHash()==(*it).ref().getHash())
	newCount++;
    }
    for (it=dol.begin(); it!=dol.end(); it++) {
      if (tnew.ref().getHash()==(*it).ref().getHash())
	oldCount++;
    }

    if (newCount>oldCount) {
      if (_flags & TRANSACTION_IMPORTER_FLAGS_COMPLETE_DAYS) {
	/* always complete days, and the new list contains more of these
	 * exact same transactions, so simply add the new one
	 */
	DBG_DEBUG(0, "Adding new");
	return DupeCheck_AddNew;
      }
      else {
	// not sure whether complete days are received, ask the user
	if (dontAskAgain) {
	  DBG_DEBUG(0, "Returning previous value (%d)", lastRv);
	  return lastRv;
	}
	else {
	  DBG_DEBUG(0, "Asking user");
	  return askUserForDuplicate(a, tnew, told, dnl, dol, dontAskAgain);
	}
      }
    }
    else {
      if (newCount<oldCount) {
	// the old list already contains more than the new one, don't add
	if (_flags & TRANSACTION_IMPORTER_FLAGS_COMPLETE_DAYS) {
	  DBG_WARN(0, "Transactions missing in new list, should not happen");
	}
      }

      return DupeCheck_DismissNew;
    }
  }
}



bool
TransactionImporter::_checkDay(Account *a,
                               TransactionImporter::DayData *d,
                               bool &askNoMore,
                               TransactionImporter::DupeCheckResult &lastRv){
  std::list<RefPointer<Transaction> >::iterator it;

  // make sure all transactions have a date or valuta date
  for (it=d->getTransactions()->begin();
       it!=d->getTransactions()->end();
       it++) {
    const GWEN_TIME *ti;
    uint32_t id;

    // assign unique id if there was none
    id=(*it).ref().getTransactionId();
    if (id==0) {
      DBG_DEBUG(0, "Setting new id");
      (*it).ref().setTransactionId(_app->getNextUniqueId());
    }
    else {
      DBG_DEBUG(0, "id is %08x", id);
    }

    ti=(*it).ref().getDate();
    if (!ti)
      ti=(*it).ref().getValutaDate();
    if (!ti) {
      GWEN_TIME *nti;

      DBG_DEBUG(0, "Setting new date");
      // set UTC date, 12 o'clock (in this case: UTC date==local date)
      nti=GWEN_Time_new(d->getYear(), d->getMonth()-1, d->getDay(),
                        12, 0, 0, 1);
      assert(nti);
      (*it).ref().setDate(nti);
      GWEN_Time_free(nti);
    } // if no time
  } // for


  if (!(_flags & TRANSACTION_IMPORTER_FLAGS_OVERWRITE_DAYS)) {
    AH_STORAGE *st;
    uint32_t hdl;
  
    st=a->getStorage();
    assert(st);

    DBG_DEBUG(0, "Opening day");
    hdl=AH_Storage_OpenDay(st, d->getYear(), d->getMonth(), d->getDay(), 1);
    if (hdl) {
      GWEN_DB_NODE *dbT;
      std::list<RefPointer<Transaction> > oldTransactions;

      // there is data for this day, check it
      dbT=AH_Storage_GetFirstTransaction(st, hdl);
      while(dbT) {
        oldTransactions.push_back(new Transaction(dbT));
        dbT=AH_Storage_GetNextTransaction(st, hdl);
      } // while transactions

      // check for duplicates
      for (it=d->getTransactions()->begin();
           it!=d->getTransactions()->end();
           it++) {
        std::list<RefPointer<Transaction> >::iterator itOld;
	bool handled;

        DBG_VERBOUS(0, "checking transaction");
        handled=false;

	// find duplicates
	for (itOld=oldTransactions.begin();
	     itOld!=oldTransactions.end();
	     itOld++) {
	  bool isEqual;

	  if (_flags & TRANSACTION_IMPORTER_FLAGS_FUZZY) {
            isEqual=(*it).ref().matchFuzzy((*itOld).ref(),
                                           _app->getSensitiveFuzzyThreshold());
            DBG_DEBUG(0, "Fuzzy matching (%d)", isEqual);
	  }
          else {
            isEqual=((*it).ref().getHash()==(*itOld).ref().getHash());
            DBG_DEBUG(0, "Hash matching (%d)", isEqual);
	  }

	  if (isEqual) {
	    DupeCheckResult rv;

            DBG_DEBUG(0, "Duplicate candidate found");

	    // duplicate candidate
	    rv=checkForDuplicate(a,
				 *it, *itOld,
				 *(d->getTransactions()),
				 oldTransactions,
				 askNoMore,
                                 lastRv);
            DBG_DEBUG(0, "Result of check: %d", rv);
	    lastRv=rv;
	    if (rv==DupeCheck_Error) {
	      DBG_ERROR(0, "User aborted");
	      AH_Storage_AbandonDay(st, hdl);
	      return false;
	    }
	    else if (rv==DupeCheck_AddNew) {
	      DBG_INFO(0, "Adding duplicate");
	      _toAdd.push_back(*it);
	    }
	    else if (rv==DupeCheck_DismissNew) {
	      // dismiss new transaction
	      DBG_INFO(0, "Dismissing duplicate");
	    }
	    else if (rv==DupeCheck_RemoveOld) {
	      // dismissing old
	      DBG_INFO(0, "Removing old transaction");
              _toDel.push_back(*itOld);
	    }
	    else if (rv==DupeCheck_ReplaceOld) {
	      // replace old
	      DBG_INFO(0, "Replacing old transaction");
	      _toDel.push_back(*itOld);
	      _toAdd.push_back(*it);
	    }

	    handled=true;
	    break;
          } // if is equal
	}
	if (!handled)
	  _toAdd.push_back(*it);
      } // for new transactions

      DBG_DEBUG(0, "Closing day");
      AH_Storage_CloseDay(st, hdl);
    } // if old data for this day exists
  } // if not in overwrite mode

  return true;
}



bool
TransactionImporter::_checkTransactions() {
  std::list<AccountData*>::iterator it;
  DupeCheckResult lastRv=DupeCheck_Error;
  bool askNoMore=false;

  if (!sortNewTransactions()) {
    DBG_INFO(0, "Error sorting transactions");
    return false;
  }

  for (it=_accountDataList.begin();
       it!=_accountDataList.end();
       it++) {
    std::list<DayData*>::iterator dit;

    for (dit=(*it)->getNewDayList().begin();
         dit!=(*it)->getNewDayList().end();
         dit++) {
      if (!_checkDay((*it)->account(), *dit,
		     askNoMore, lastRv)) {
	DBG_INFO(0, "Error checking day");
	return false;
      }
    } // for days
  } // for accounts

  return true;
}



bool TransactionImporter::_assignCatsAndPayees() {
  std::list<RefPointer<Transaction> >::iterator it;
  bool bAskUserForPayee;
  bool bAskUserForCategory;
  unsigned int cnt;
  uint32_t pid;

  bAskUserForPayee=_app->optionAutoAskForPayee();
  bAskUserForCategory=_app->optionAutoAskForCategory();


  pid=GWEN_Gui_ProgressStart(GWEN_GUI_PROGRESS_DELAY |
			     GWEN_GUI_PROGRESS_ALLOW_EMBED |
			     GWEN_GUI_PROGRESS_SHOW_PROGRESS |
			     GWEN_GUI_PROGRESS_SHOW_ABORT,
			     _app->tr("Assigning categories and payees").c_str(),
			     _app->tr("transaction(s)").c_str(),
			     _toAdd.size(),
			     0);
  cnt=0;

  /* assign payees and categories */
  for (it=_toAdd.begin(); it!=_toAdd.end(); it++) {
    DBG_DEBUG(0, "Handling transaction");
  
    if ((*it).ref().getPayee().empty()) {
      Payee *payee=0;

      if (_app->optionAutoAssignPayee()) {
	payee=_app->findPayeeByTransactionMatch((*it).ptr());
	if (!payee) {
	  if (bAskUserForPayee) {
	    bool aborted=false;
	    bool askNoMore=false;
  
	    payee=askUserForPayee((*it), aborted, askNoMore);
	    if (!payee && aborted) {
	      DBG_ERROR(0, "User aborted import");
	      GWEN_Gui_ProgressEnd(pid);
	      return false;
	    }
	    if (askNoMore)
	      bAskUserForPayee=false;
	  }
	}
	if (payee) {
	  DBG_DEBUG(0, "Assigning payee %s",
                    payee->id().c_str());
	  (*it).ref().setPayee(payee->id());
	}
      }
    }

    if ((*it).ref().getCategory().empty()) {
      Category *category=0;
  
      if (_app->optionAutoAssignCategory()) {
	category=_app->findCategoryByTransactionMatch((*it).ptr());
	if (!category) {
	  if (bAskUserForCategory) {
	    bool aborted=false;
	    bool askNoMore=false;
  
	    category=askUserForCategory((*it), aborted, askNoMore);
	    if (!category && aborted) {
	      DBG_ERROR(0, "User aborted import");
	      GWEN_Gui_ProgressEnd(pid);
	      return false;
	    }
	    if (askNoMore)
	      bAskUserForCategory=false;
	  }
	}
      }
      if (category) {
	DBG_DEBUG(0, "Assigning category %s",
		  category->getName().c_str());
	(*it).ref().setCategory(category->getId());
      }
    }
  
    cnt++;
    if (GWEN_ERROR_USER_ABORTED==GWEN_Gui_ProgressAdvance(pid, cnt)) {
      GWEN_Gui_ProgressEnd(pid);
      return false;
    }
  } /* for */
  GWEN_Gui_ProgressEnd(pid);
  return true;
}



bool
TransactionImporter::_adjustPCValues(std::list<RefPointer<Transaction> > &tl,
				     bool add){
  std::list<RefPointer<Transaction> >::iterator it;
  unsigned int cnt;
  uint32_t pid;

  pid=GWEN_Gui_ProgressStart(GWEN_GUI_PROGRESS_DELAY |
			     GWEN_GUI_PROGRESS_ALLOW_EMBED |
			     GWEN_GUI_PROGRESS_SHOW_PROGRESS |
			     GWEN_GUI_PROGRESS_SHOW_ABORT,
			     _app->tr("Adjusting amounts of categories and payees").c_str(),
			     _app->tr("transaction(s)").c_str(),
			     tl.size(),
			     0);
  cnt=0;
  for (it=tl.begin(); it!=tl.end(); it++) {
    const AB_VALUE *v;

    v=(*it).ref().getValue();
    if (v) {
      std::string spayee;
      std::string scat;

      spayee=(*it).ref().getPayee();
      if (!spayee.empty()) {
	Payee *payee;

	payee=_app->findPayeeById(spayee.c_str());
	if (payee) {
	  if (add) {
	    /* add values */
	    if (AB_Value_IsNegative(v)) {
	      AB_VALUE *nv;
  
	      nv=AB_Value_dup(v);
	      AB_Value_Negate(nv);
	      payee->addOutValue(nv);
	      AB_Value_free(nv);
	    }
	    else
	      payee->addInValue(v);
	  }
	  else {
	    /* subtract values */
	    if (AB_Value_IsNegative(v)) {
	      payee->addOutValue(v);
	    }
	    else {
	      AB_VALUE *nv;

	      nv=AB_Value_dup(v);
	      AB_Value_Negate(nv);
	      payee->addInValue(nv);
	      AB_Value_free(nv);
	    }
	  }
	} // if payee
      } // if spayee

      scat=(*it).ref().getCategory();
      if (!scat.empty()) {
	Category *cat;

	cat=_app->findCategoryById(scat.c_str());
	if (cat) {
	  if (add)
	    cat->addValue(v);
          else {
	    /* subtract value */
	    AB_VALUE *nv;

	    nv=AB_Value_dup(v);
	    AB_Value_Negate(nv);
	    cat->addValue(nv);
	    AB_Value_free(nv);
	  }
	}
      }
    } // if v
    cnt++;

    if (GWEN_ERROR_USER_ABORTED==GWEN_Gui_ProgressAdvance(pid, cnt)) {
      GWEN_Gui_ProgressEnd(pid);
      return false;
    }
  } // for
  GWEN_Gui_ProgressEnd(pid);

  return true;
}



bool TransactionImporter::importTransactions(){
  std::list<RefPointer<Transaction> >::iterator it;
  unsigned int cnt;
  uint32_t pid;

  if (!_checkTransactions())
    return false;

  if (!_assignCatsAndPayees())
    return false;

  if (!_adjustPCValues(_toDel, false))
    return false;

  if (!_adjustPCValues(_toAdd, true))
    return false;

  if ((_toAdd.size() || _toDel.size()) &&
      !presentTransactionsToUser(_toAdd, _toDel))
    return false;

  /* add new transactions */
  pid=GWEN_Gui_ProgressStart(GWEN_GUI_PROGRESS_DELAY |
			     GWEN_GUI_PROGRESS_ALLOW_EMBED |
			     GWEN_GUI_PROGRESS_SHOW_PROGRESS |
			     GWEN_GUI_PROGRESS_SHOW_ABORT,
			     _app->tr("Adding transactions to accounts").c_str(),
			     _app->tr("transaction(s)").c_str(),
			     _toAdd.size(),
			     0);
  cnt=0;
  for (it=_toAdd.begin(); it!=_toAdd.end(); it++) {
    Account *a;

    a=_app->findAccount((*it).ref().getLocalBankCode().c_str(),
			(*it).ref().getLocalAccountNumber().c_str());
    assert(a);
    a->addTransaction(*it);
    cnt++;
    if (GWEN_ERROR_USER_ABORTED==GWEN_Gui_ProgressAdvance(pid, cnt)) {
      GWEN_Gui_ProgressEnd(pid);
      return false;
    }
  }
  GWEN_Gui_ProgressEnd(pid);

  /* del old transactions */
  if (_toDel.size()) {
    pid=GWEN_Gui_ProgressStart(GWEN_GUI_PROGRESS_DELAY |
			       GWEN_GUI_PROGRESS_ALLOW_EMBED |
			       GWEN_GUI_PROGRESS_SHOW_PROGRESS |
			       GWEN_GUI_PROGRESS_SHOW_ABORT,
			       _app->tr("Deleting transactions from accounts").c_str(),
			       _app->tr("transaction(s)").c_str(),
			       _toDel.size(),
			       0);
    cnt=0;
    for (it=_toDel.begin(); it!=_toDel.end(); it++) {
      Account *a;
  
      a=_app->findAccount((*it).ref().getLocalBankCode().c_str(),
			  (*it).ref().getLocalAccountNumber().c_str());
      assert(a);
      a->removeTransaction((*it).ptr());
      cnt++;
      if (GWEN_ERROR_USER_ABORTED==GWEN_Gui_ProgressAdvance(pid, cnt)) {
	GWEN_Gui_ProgressEnd(pid);
	return false;
      }
    }
    GWEN_Gui_ProgressEnd(pid);
  }

  return true;

}



AB_ACCOUNT_STATUS*
TransactionImporter::_getBestBalance(AB_IMEXPORTER_ACCOUNTINFO *ai,
				     bool noted) {
  AB_ACCOUNT_STATUS *ast;
  AB_ACCOUNT_STATUS *best=0;

  ast=AB_ImExporterAccountInfo_GetFirstAccountStatus(ai);
  while(ast) {
    if ((noted && AB_AccountStatus_GetNotedBalance(ast)) ||
	(!noted && AB_AccountStatus_GetBookedBalance(ast))) {
      if (!best)
	best=ast;
      else {
	const GWEN_TIME *tiBest;
	const GWEN_TIME *ti;

	tiBest=AB_AccountStatus_GetTime(best);
	ti=AB_AccountStatus_GetTime(ast);

	if (!tiBest) {
	  best=ast;
	}
	else {
	  if (ti) {
	    double d;

	    /* we have two times, compare them */
	    d=GWEN_Time_Diff(ti, tiBest);
	    if (d>0)
	      /* newer */
	      best=ast;
	  }
	}
      }
    } /* if noted balance */
    ast=AB_ImExporterAccountInfo_GetNextAccountStatus(ai);
  } /* while */

  return best;
}



bool TransactionImporter::importAccounts(){
  AB_IMEXPORTER_ACCOUNTINFO *ai;

  ai=AB_ImExporterContext_GetFirstAccountInfo(_context);
  while(ai) {
    TransactionImporter::AccountData *ad;

    ad=getAccountData(AB_ImExporterAccountInfo_GetBankCode(ai),
		      AB_ImExporterAccountInfo_GetAccountNumber(ai),
		      AB_ImExporterAccountInfo_GetAccountName(ai));
    if (!ad) {
      /* account not found, TODO: ask the user to add it */
      DBG_ERROR(0, "Account \"%s/%s\" not found",
		AB_ImExporterAccountInfo_GetBankCode(ai),
		AB_ImExporterAccountInfo_GetAccountNumber(ai));
    }
    else {
      Account *a;
      AB_ACCOUNT_STATUS *bestBooked;
      AB_ACCOUNT_STATUS *bestNoted;

      a=ad->account();
      assert(a);

      /* combine best booked and best noted balances */
      bestBooked=_getBestBalance(ai, false);
      bestNoted=_getBestBalance(ai, true);

      if (bestBooked || bestNoted) {
	AB_ACCOUNT_STATUS *finalStatus;

	finalStatus=AB_AccountStatus_new();

	if (bestBooked) {
	  const AB_BALANCE *bal;

	  bal=AB_AccountStatus_GetBookedBalance(bestBooked);
	  if (bal)
	    AB_AccountStatus_SetBookedBalance(finalStatus, bal);
	  if (AB_AccountStatus_GetBankLine(finalStatus)==NULL) {
	    const AB_VALUE *v=AB_AccountStatus_GetBankLine(bestBooked);
	    AB_AccountStatus_SetBankLine(finalStatus, v);
	  }
	  if (AB_AccountStatus_GetDisposable(finalStatus)==NULL) {
	    const AB_VALUE *v=AB_AccountStatus_GetDisposable(bestBooked);
	    AB_AccountStatus_SetDisposable(finalStatus, v);
	  }
	  if (AB_AccountStatus_GetDisposed(finalStatus)==NULL) {
	    const AB_VALUE *v=AB_AccountStatus_GetDisposed(bestBooked);
	    AB_AccountStatus_SetDisposed(finalStatus, v);
	  }
	  AB_AccountStatus_SetTime(finalStatus,
				   AB_AccountStatus_GetTime(bestBooked));
	}

	if (bestNoted) {
	  const AB_BALANCE *bal;

	  bal=AB_AccountStatus_GetNotedBalance(bestNoted);
	  if (bal)
	    AB_AccountStatus_SetNotedBalance(finalStatus, bal);
	  if (AB_AccountStatus_GetBankLine(finalStatus)==NULL) {
	    const AB_VALUE *v=AB_AccountStatus_GetBankLine(bestNoted);
	    AB_AccountStatus_SetBankLine(finalStatus, v);
	  }
	  if (AB_AccountStatus_GetDisposable(finalStatus)==NULL) {
	    const AB_VALUE *v=AB_AccountStatus_GetDisposable(bestNoted);
	    AB_AccountStatus_SetDisposable(finalStatus, v);
	  }
	  if (AB_AccountStatus_GetDisposed(finalStatus)==NULL) {
	    const AB_VALUE *v=AB_AccountStatus_GetDisposed(bestNoted);
	    AB_AccountStatus_SetDisposed(finalStatus, v);
	  }
					 
	  if (AB_AccountStatus_GetTime(finalStatus)==NULL)
	    AB_AccountStatus_SetTime(finalStatus,
				     AB_AccountStatus_GetTime(bestNoted));
	}
	a->setAccountStatus(finalStatus);
      }

      /* set account name if it is empty in DB */
      if (a->getAccountName().empty()) {
	const char *s;

	s=AB_ImExporterAccountInfo_GetAccountName(ai);
	if (s)
          a->setAccountName(s);
      }
    }
    ai=AB_ImExporterContext_GetNextAccountInfo(_context);
  } /* while accounts */

  return true;
}



bool TransactionImporter::importStandingOrders(){
  AB_IMEXPORTER_ACCOUNTINFO *ai;

  ai=AB_ImExporterContext_GetFirstAccountInfo(_context);
  while(ai) {
    Account *a;

    a=_app->findAccount(AB_ImExporterAccountInfo_GetBankCode(ai),
                        AB_ImExporterAccountInfo_GetAccountNumber(ai));
    if (!a) {
      /* account not found, TODO: ask the user to add it */
    }
    else {
      const AB_TRANSACTION *t;

      t=AB_ImExporterAccountInfo_GetFirstStandingOrder(ai);
      while(t) {
        RefPointer<StandingOrder> sto;

	// TODO: search for existing standing order
        sto=new StandingOrder(t);
        assert(sto.isValid());
        a->addStandingOrder(sto);
        t=AB_ImExporterAccountInfo_GetNextStandingOrder(ai);
      }
    }
    ai=AB_ImExporterContext_GetNextAccountInfo(_context);
  } /* while accounts */

  return true;
}



bool TransactionImporter::importDatedTransfers(){
  AB_IMEXPORTER_ACCOUNTINFO *ai;

  ai=AB_ImExporterContext_GetFirstAccountInfo(_context);
  while(ai) {
    Account *a;

    a=_app->findAccount(AB_ImExporterAccountInfo_GetBankCode(ai),
                        AB_ImExporterAccountInfo_GetAccountNumber(ai));
    if (!a) {
      /* account not found, TODO: ask the user to add it */
      DBG_ERROR(0, "Account \"%s/%s\" not found",
                AB_ImExporterAccountInfo_GetBankCode(ai),
                AB_ImExporterAccountInfo_GetAccountNumber(ai));
    }
    else {
      const AB_TRANSACTION *t;

      t=AB_ImExporterAccountInfo_GetFirstDatedTransfer(ai);
      while(t) {
	RefPointer<Transfer> sto;

	// TODO: search for existing standing order
	sto=new Transfer(t);
	assert(sto.isValid());
	a->addDatedTransfer(sto);
        t=AB_ImExporterAccountInfo_GetNextDatedTransfer(ai);
      }
    }
    ai=AB_ImExporterContext_GetNextAccountInfo(_context);
  } /* while accounts */

  return true;
}



bool TransactionImporter::importContext(){
  bool rv;
  std::list<AccountData*>::iterator ait;

  rv=importTransactions();
  rv&=importAccounts();
  rv&=importStandingOrders();
  rv&=importDatedTransfers();

  /* update views */
  for (ait=_accountDataList.begin(); ait!=_accountDataList.end(); ait++) {
    _app->appInfoTransactionsChanged((*ait)->account());
  }
  return rv;
}



TransactionImporter::AccountData*
TransactionImporter::getAccountData(const char *s1,
				    const char *s2,
				    const char *s3) {
  std::list<AccountData*>::iterator it;
  std::string bankCode;
  std::string accountNumber;
  std::string accountName;
  std::string alias;
  Account *a;

  DBG_DEBUG(0, "Checking for account \"%s/%s\"", s1, s2);

  if (s1)
    bankCode=s1;
  if (s2)
    accountNumber=s2;
  if (s3)
    accountName=s3;

  alias=bankCode+":"+accountNumber+":"+accountName;

  a=0;
  for (it=_accountDataList.begin();
       it!=_accountDataList.end();
       it++) {
    if (((*it)->bankCode()==bankCode &&
	 (*it)->accountNumber()==accountNumber)
	||
	(bankCode.empty() &&
	 accountNumber.empty() &&
	 !accountName.empty() &&
	 (*it)->accountName()==accountName))
      return *it;
  } // for

  /* find by alias */
  for (it=_accountDataList.begin();
       it!=_accountDataList.end();
       it++) {
    std::list<std::string>::const_iterator sit;

    for (sit=(*it)->aliases().begin();
	 sit!=(*it)->aliases().end();
	 sit++) {
      if (*sit==alias)
	return *it;
    }
  } // for

  if (!bankCode.empty() && !accountNumber.empty())
    a=_app->findAccount(bankCode.c_str(), accountNumber.c_str());
  else if (bankCode.empty() && !accountNumber.empty())
    a=_app->findAccount("*", accountNumber.c_str());
  else
    a=_app->findAccountByName(accountName.c_str());
  if (!a) {
    // TODO: check for aliases
    // TODO: ask the user for alias
    a=askUserForAccount(bankCode, accountNumber, accountName);
    if (!a) {
      DBG_NOTICE(0, "Account not found");
      return NULL;
    }

    for (it=_accountDataList.begin();
         it!=_accountDataList.end();
         it++) {
      if ((*it)->bankCode()==a->getBankCode() &&
	  (*it)->accountNumber()==a->getAccountNumber()) {
	(*it)->addAlias(alias);
	return *it;
      }
    } // for
  }

  DBG_DEBUG(0, "Adding account data");

  AccountData *ad=new AccountData(a);
  ad->addAlias(alias);
  _accountDataList.push_back(ad);
  return _accountDataList.back();
}









