# This script generates the module Scientific.IO.PDBSpaceGroups
#
# The space group information is taken from cctbx, which must be
# available to run this script.
#
# Written by Konrad Hinsen
# last revision: 2007-10-18
#

from cctbx.sgtbx import space_group_info
from Scientific import N

def format_rational(r):
    s = str(r)
    if '/' in s:
        return './'.join(s.split('/')) + '.'
    else:
        return s + '.'

def space_group_table_entry(number, labels, sgi):
    group = sgi.group()
    print "transformations = []"
    for symmetry_transformation in group:
        rot =  symmetry_transformation.as_rational().r
        trans =  symmetry_transformation.as_rational().t
        print 'rot = N.array([' + \
              ','.join([str(x) for x in rot]) + '])'
        print 'rot.shape = (3, 3)'
        print 'trans = Vector(' + \
              ','.join([format_rational(x) for x in trans]) + ')'
        print 'transformations.append((rot, trans))'
    print 'sg = SpaceGroup(%d, %s, transformations)' % (number, str(labels))
    print "_space_group_table[%d] = sg" % number
    for l in labels:
        print "_space_group_table[%s] = sg" % repr(l)
    print

# This list was obtained from the sginfo utility. Probably the list can
# be obtained from cctbx as well somewhow, but I don't know how.
space_group_labels = [
['P1', 'P 1', '1', 'C1^1'],
['P-1', '-P 1', '2', 'Ci^1'],
['P2:b', 'P121', 'P 2y', '3:b', 'C2^1'],
['P2:c', 'P112', 'P 2', '3:c', 'C2^1'],
['P2:a', 'P211', 'P 2x', '3:a', 'C2^1'],
['P21:b', 'P1211', 'P 2yb', '4:b', 'C2^2'],
['P21:c', 'P1121', 'P 2c', '4:c', 'C2^2'],
['P21:a', 'P2111', 'P 2xa', '4:a', 'C2^2'],
['C2:b1', 'C121', 'C 2y', '5:b1', 'C2^3'],
['C2:b2', 'A121', 'A 2y', '5:b2', 'C2^3'],
['C2:b3', 'I121', 'I 2y', '5:b3', 'C2^3'],
['C2:c1', 'A112', 'A 2', '5:c1', 'C2^3'],
['C2:c2', 'B112', 'B2', 'B 2', '5:c2', 'C2^3'],
['C2:c3', 'I112', 'I 2', '5:c3', 'C2^3'],
['C2:a1', 'B211', 'B 2x', '5:a1', 'C2^3'],
['C2:a2', 'C211', 'C 2x', '5:a2', 'C2^3'],
['C2:a3', 'I211', 'I 2x', '5:a3', 'C2^3'],
['Pm:b', 'P1m1', 'P -2y', '6:b', 'Cs^1'],
['Pm:c', 'P11m', 'P -2', '6:c', 'Cs^1'],
['Pm:a', 'Pm11', 'P -2x', '6:a', 'Cs^1'],
['Pc:b1', 'P1c1', 'P -2yc', '7:b1', 'Cs^2'],
['Pc:b2', 'P1n1', 'P -2yac', '7:b2', 'Cs^2'],
['Pc:b3', 'P1a1', 'P -2ya', '7:b3', 'Cs^2'],
['Pc:c1', 'P11a', 'P -2a', '7:c1', 'Cs^2'],
['Pc:c2', 'P11n', 'P -2ab', '7:c2', 'Cs^2'],
['Pc:c3', 'P11b', 'Pb', 'P -2b', '7:c3', 'Cs^2'],
['Pc:a1', 'Pb11', 'P -2xb', '7:a1', 'Cs^2'],
['Pc:a2', 'Pn11', 'P -2xbc', '7:a2', 'Cs^2'],
['Pc:a3', 'Pc11', 'P -2xc', '7:a3', 'Cs^2'],
['Cm:b1', 'C1m1', 'C -2y', '8:b1', 'Cs^3'],
['Cm:b2', 'A1m1', 'A -2y', '8:b2', 'Cs^3'],
['Cm:b3', 'I1m1', 'I -2y', '8:b3', 'Cs^3'],
['Cm:c1', 'A11m', 'A -2', '8:c1', 'Cs^3'],
['Cm:c2', 'B11m', 'Bm', 'B -2', '8:c2', 'Cs^3'],
['Cm:c3', 'I11m', 'I -2', '8:c3', 'Cs^3'],
['Cm:a1', 'Bm11', 'B -2x', '8:a1', 'Cs^3'],
['Cm:a2', 'Cm11', 'C -2x', '8:a2', 'Cs^3'],
['Cm:a3', 'Im11', 'I -2x', '8:a3', 'Cs^3'],
['Cc:b1', 'C1c1', 'C -2yc', '9:b1', 'Cs^4'],
['Cc:b2', 'A1n1', 'A -2yac', '9:b2', 'Cs^4'],
['Cc:b3', 'I1a1', 'I -2ya', '9:b3', 'Cs^4'],
['Cc:-b1', 'A1a1', 'A -2ya', '9:-b1', 'Cs^4'],
['Cc:-b2', 'C1n1', 'C -2ybc', '9:-b2', 'Cs^4'],
['Cc:-b3', 'I1c1', 'I -2yc', '9:-b3', 'Cs^4'],
['Cc:c1', 'A11a', 'A -2a', '9:c1', 'Cs^4'],
['Cc:c2', 'B11n', 'B -2bc', '9:c2', 'Cs^4'],
['Cc:c3', 'I11b', 'I -2b', '9:c3', 'Cs^4'],
['Cc:-c1', 'B11b', 'Bb', 'B -2b', '9:-c1', 'Cs^4'],
['Cc:-c2', 'A11n', 'A -2ac', '9:-c2', 'Cs^4'],
['Cc:-c3', 'I11a', 'I -2a', '9:-c3', 'Cs^4'],
['Cc:a1', 'Bb11', 'B -2xb', '9:a1', 'Cs^4'],
['Cc:a2', 'Cn11', 'C -2xbc', '9:a2', 'Cs^4'],
['Cc:a3', 'Ic11', 'I -2xc', '9:a3', 'Cs^4'],
['Cc:-a1', 'Cc11', 'C -2xc', '9:-a1', 'Cs^4'],
['Cc:-a2', 'Bn11', 'B -2xbc', '9:-a2', 'Cs^4'],
['Cc:-a3', 'Ib11', 'I -2xb', '9:-a3', 'Cs^4'],
['P2/m:b', 'P12/m1', '-P 2y', '10:b', 'C2h^1'],
['P2/m:c', 'P112/m', '-P 2', '10:c', 'C2h^1'],
['P2/m:a', 'P2/m11', '-P 2x', '10:a', 'C2h^1'],
['P21/m:b', 'P121/m1', '-P 2yb', '11:b', 'C2h^2'],
['P21/m:c', 'P1121/m', '-P 2c', '11:c', 'C2h^2'],
['P21/m:a', 'P21/m11', '-P 2xa', '11:a', 'C2h^2'],
['C2/m:b1', 'C12/m1', '-C 2y', '12:b1', 'C2h^3'],
['C2/m:b2', 'A12/m1', '-A 2y', '12:b2', 'C2h^3'],
['C2/m:b3', 'I12/m1', '-I 2y', '12:b3', 'C2h^3'],
['C2/m:c1', 'A112/m', '-A 2', '12:c1', 'C2h^3'],
['C2/m:c2', 'B112/m', 'B2/m', '-B 2', '12:c2', 'C2h^3'],
['C2/m:c3', 'I112/m', '-I 2', '12:c3', 'C2h^3'],
['C2/m:a1', 'B2/m11', '-B 2x', '12:a1', 'C2h^3'],
['C2/m:a2', 'C2/m11', '-C 2x', '12:a2', 'C2h^3'],
['C2/m:a3', 'I2/m11', '-I 2x', '12:a3', 'C2h^3'],
['P2/c:b1', 'P12/c1', '-P 2yc', '13:b1', 'C2h^4'],
['P2/c:b2', 'P12/n1', '-P 2yac', '13:b2', 'C2h^4'],
['P2/c:b3', 'P12/a1', '-P 2ya', '13:b3', 'C2h^4'],
['P2/c:c1', 'P112/a', '-P 2a', '13:c1', 'C2h^4'],
['P2/c:c2', 'P112/n', '-P 2ab', '13:c2', 'C2h^4'],
['P2/c:c3', 'P112/b', 'P2/b', '-P 2b', '13:c3', 'C2h^4'],
['P2/c:a1', 'P2/b11', '-P 2xb', '13:a1', 'C2h^4'],
['P2/c:a2', 'P2/n11', '-P 2xbc', '13:a2', 'C2h^4'],
['P2/c:a3', 'P2/c11', '-P 2xc', '13:a3', 'C2h^4'],
['P21/c:b1', 'P121/c1', '-P 2ybc', '14:b1', 'C2h^5'],
['P21/c:b2', 'P121/n1', '-P 2yn', '14:b2', 'C2h^5'],
['P21/c:b3', 'P121/a1', '-P 2yab', '14:b3', 'C2h^5'],
['P21/c:c1', 'P1121/a', '-P 2ac', '14:c1', 'C2h^5'],
['P21/c:c2', 'P1121/n', '-P 2n', '14:c2', 'C2h^5'],
['P21/c:c3', 'P1121/b', 'P21/b', '-P 2bc', '14:c3', 'C2h^5'],
['P21/c:a1', 'P21/b11', '-P 2xab', '14:a1', 'C2h^5'],
['P21/c:a2', 'P21/n11', '-P 2xn', '14:a2', 'C2h^5'],
['P21/c:a3', 'P21/c11', '-P 2xac', '14:a3', 'C2h^5'],
['C2/c:b1', 'C12/c1', '-C 2yc', '15:b1', 'C2h^6'],
['C2/c:b2', 'A12/n1', '-A 2yac', '15:b2', 'C2h^6'],
['C2/c:b3', 'I12/a1', '-I 2ya', '15:b3', 'C2h^6'],
['C2/c:-b1', 'A12/a1', '-A 2ya', '15:-b1', 'C2h^6'],
['C2/c:-b2', 'C12/n1', '-C 2ybc', '15:-b2', 'C2h^6'],
['C2/c:-b3', 'I12/c1', '-I 2yc', '15:-b3', 'C2h^6'],
['C2/c:c1', 'A112/a', '-A 2a', '15:c1', 'C2h^6'],
['C2/c:c2', 'B112/n', '-B 2bc', '15:c2', 'C2h^6'],
['C2/c:c3', 'I112/b', '-I 2b', '15:c3', 'C2h^6'],
['C2/c:-c1', 'B112/b', 'B2/b', '-B 2b', '15:-c1', 'C2h^6'],
['C2/c:-c2', 'A112/n', '-A 2ac', '15:-c2', 'C2h^6'],
['C2/c:-c3', 'I112/a', '-I 2a', '15:-c3', 'C2h^6'],
['C2/c:a1', 'B2/b11', '-B 2xb', '15:a1', 'C2h^6'],
['C2/c:a2', 'C2/n11', '-C 2xbc', '15:a2', 'C2h^6'],
['C2/c:a3', 'I2/c11', '-I 2xc', '15:a3', 'C2h^6'],
['C2/c:-a1', 'C2/c11', '-C 2xc', '15:-a1', 'C2h^6'],
['C2/c:-a2', 'B2/n11', '-B 2xbc', '15:-a2', 'C2h^6'],
['C2/c:-a3', 'I2/b11', '-I 2xb', '15:-a3', 'C2h^6'],
['P222', 'P 2 2', '16', 'D2^1'],
['P2221', 'P 2c 2', '17', 'D2^2'],
['P2122', 'P 2a 2a', '17:cab', 'D2^2'],
['P2212', 'P 2 2b', '17:bca', 'D2^2'],
['P21212', 'P 2 2ab', '18', 'D2^3'],
['P22121', 'P 2bc 2', '18:cab', 'D2^3'],
['P21221', 'P 2ac 2ac', '18:bca', 'D2^3'],
['P212121', 'P 2ac 2ab', '19', 'D2^4'],
['C2221', 'C 2c 2', '20', 'D2^5'],
['A2122', 'A 2a 2a', '20:cab', 'D2^5'],
['B2212', 'B 2 2b', '20:bca', 'D2^5'],
['C222', 'C 2 2', '21', 'D2^6'],
['A222', 'A 2 2', '21:cab', 'D2^6'],
['B222', 'B 2 2', '21:bca', 'D2^6'],
['F222', 'F 2 2', '22', 'D2^7'],
['I222', 'I 2 2', '23', 'D2^8'],
['I212121', 'I 2b 2c', '24', 'D2^9'],
['Pmm2', 'P 2 -2', '25', 'C2v^1'],
['P2mm', 'P -2 2', '25:cab', 'C2v^1'],
['Pm2m', 'P -2 -2', '25:bca', 'C2v^1'],
['Pmc21', 'P 2c -2', '26', 'C2v^2'],
['Pcm21', 'P 2c -2c', '26:ba-c', 'C2v^2'],
['P21ma', 'P -2a 2a', '26:cab', 'C2v^2'],
['P21am', 'P -2 2a', '26:-cba', 'C2v^2'],
['Pb21m', 'P -2 -2b', '26:bca', 'C2v^2'],
['Pm21b', 'P -2b -2', '26:a-cb', 'C2v^2'],
['Pcc2', 'P 2 -2c', '27', 'C2v^3'],
['P2aa', 'P -2a 2', '27:cab', 'C2v^3'],
['Pb2b', 'P -2b -2b', '27:bca', 'C2v^3'],
['Pma2', 'P 2 -2a', '28', 'C2v^4'],
['Pbm2', 'P 2 -2b', '28:ba-c', 'C2v^4'],
['P2mb', 'P -2b 2', '28:cab', 'C2v^4'],
['P2cm', 'P -2c 2', '28:-cba', 'C2v^4'],
['Pc2m', 'P -2c -2c', '28:bca', 'C2v^4'],
['Pm2a', 'P -2a -2a', '28:a-cb', 'C2v^4'],
['Pca21', 'P 2c -2ac', '29', 'C2v^5'],
['Pbc21', 'P 2c -2b', '29:ba-c', 'C2v^5'],
['P21ab', 'P -2b 2a', '29:cab', 'C2v^5'],
['P21ca', 'P -2ac 2a', '29:-cba', 'C2v^5'],
['Pc21b', 'P -2bc -2c', '29:bca', 'C2v^5'],
['Pb21a', 'P -2a -2ab', '29:a-cb', 'C2v^5'],
['Pnc2', 'P 2 -2bc', '30', 'C2v^6'],
['Pcn2', 'P 2 -2ac', '30:ba-c', 'C2v^6'],
['P2na', 'P -2ac 2', '30:cab', 'C2v^6'],
['P2an', 'P -2ab 2', '30:-cba', 'C2v^6'],
['Pb2n', 'P -2ab -2ab', '30:bca', 'C2v^6'],
['Pn2b', 'P -2bc -2bc', '30:a-cb', 'C2v^6'],
['Pmn21', 'P 2ac -2', '31', 'C2v^7'],
['Pnm21', 'P 2bc -2bc', '31:ba-c', 'C2v^7'],
['P21mn', 'P -2ab 2ab', '31:cab', 'C2v^7'],
['P21nm', 'P -2 2ac', '31:-cba', 'C2v^7'],
['Pn21m', 'P -2 -2bc', '31:bca', 'C2v^7'],
['Pm21n', 'P -2ab -2', '31:a-cb', 'C2v^7'],
['Pba2', 'P 2 -2ab', '32', 'C2v^8'],
['P2cb', 'P -2bc 2', '32:cab', 'C2v^8'],
['Pc2a', 'P -2ac -2ac', '32:bca', 'C2v^8'],
['Pna21', 'P 2c -2n', '33', 'C2v^9'],
['Pbn21', 'P 2c -2ab', '33:ba-c', 'C2v^9'],
['P21nb', 'P -2bc 2a', '33:cab', 'C2v^9'],
['P21cn', 'P -2n 2a', '33:-cba', 'C2v^9'],
['Pc21n', 'P -2n -2ac', '33:bca', 'C2v^9'],
['Pn21a', 'P -2ac -2n', '33:a-cb', 'C2v^9'],
['Pnn2', 'P 2 -2n', '34', 'C2v^10'],
['P2nn', 'P -2n 2', '34:cab', 'C2v^10'],
['Pn2n', 'P -2n -2n', '34:bca', 'C2v^10'],
['Cmm2', 'C 2 -2', '35', 'C2v^11'],
['A2mm', 'A -2 2', '35:cab', 'C2v^11'],
['Bm2m', 'B -2 -2', '35:bca', 'C2v^11'],
['Cmc21', 'C 2c -2', '36', 'C2v^12'],
['Ccm21', 'C 2c -2c', '36:ba-c', 'C2v^12'],
['A21ma', 'A -2a 2a', '36:cab', 'C2v^12'],
['A21am', 'A -2 2a', '36:-cba', 'C2v^12'],
['Bb21m', 'B -2 -2b', '36:bca', 'C2v^12'],
['Bm21b', 'B -2b -2', '36:a-cb', 'C2v^12'],
['Ccc2', 'C 2 -2c', '37', 'C2v^13'],
['A2aa', 'A -2a 2', '37:cab', 'C2v^13'],
['Bb2b', 'B -2b -2b', '37:bca', 'C2v^13'],
['Amm2', 'A 2 -2', '38', 'C2v^14'],
['Bmm2', 'B 2 -2', '38:ba-c', 'C2v^14'],
['B2mm', 'B -2 2', '38:cab', 'C2v^14'],
['C2mm', 'C -2 2', '38:-cba', 'C2v^14'],
['Cm2m', 'C -2 -2', '38:bca', 'C2v^14'],
['Am2m', 'A -2 -2', '38:a-cb', 'C2v^14'],
['Abm2', 'A 2 -2c', '39', 'C2v^15'],
['Bma2', 'B 2 -2c', '39:ba-c', 'C2v^15'],
['B2cm', 'B -2c 2', '39:cab', 'C2v^15'],
['C2mb', 'C -2b 2', '39:-cba', 'C2v^15'],
['Cm2a', 'C -2b -2b', '39:bca', 'C2v^15'],
['Ac2m', 'A -2c -2c', '39:a-cb', 'C2v^15'],
['Ama2', 'A 2 -2a', '40', 'C2v^16'],
['Bbm2', 'B 2 -2b', '40:ba-c', 'C2v^16'],
['B2mb', 'B -2b 2', '40:cab', 'C2v^16'],
['C2cm', 'C -2c 2', '40:-cba', 'C2v^16'],
['Cc2m', 'C -2c -2c', '40:bca', 'C2v^16'],
['Am2a', 'A -2a -2a', '40:a-cb', 'C2v^16'],
['Aba2', 'A 2 -2ac', '41', 'C2v^17'],
['Bba2', 'B 2 -2bc', '41:ba-c', 'C2v^17'],
['B2cb', 'B -2bc 2', '41:cab', 'C2v^17'],
['C2cb', 'C -2bc 2', '41:-cba', 'C2v^17'],
['Cc2a', 'C -2bc -2bc', '41:bca', 'C2v^17'],
['Ac2a', 'A -2ac -2ac', '41:a-cb', 'C2v^17'],
['Fmm2', 'F 2 -2', '42', 'C2v^18'],
['F2mm', 'F -2 2', '42:cab', 'C2v^18'],
['Fm2m', 'F -2 -2', '42:bca', 'C2v^18'],
['Fdd2', 'F 2 -2d', '43', 'C2v^19'],
['F2dd', 'F -2d 2', '43:cab', 'C2v^19'],
['Fd2d', 'F -2d -2d', '43:bca', 'C2v^19'],
['Imm2', 'I 2 -2', '44', 'C2v^20'],
['I2mm', 'I -2 2', '44:cab', 'C2v^20'],
['Im2m', 'I -2 -2', '44:bca', 'C2v^20'],
['Iba2', 'I 2 -2c', '45', 'C2v^21'],
['I2cb', 'I -2a 2', '45:cab', 'C2v^21'],
['Ic2a', 'I -2b -2b', '45:bca', 'C2v^21'],
['Ima2', 'I 2 -2a', '46', 'C2v^22'],
['Ibm2', 'I 2 -2b', '46:ba-c', 'C2v^22'],
['I2mb', 'I -2b 2', '46:cab', 'C2v^22'],
['I2cm', 'I -2c 2', '46:-cba', 'C2v^22'],
['Ic2m', 'I -2c -2c', '46:bca', 'C2v^22'],
['Im2a', 'I -2a -2a', '46:a-cb', 'C2v^22'],
['Pmmm', '-P 2 2', '47', 'D2h^1'],
['Pnnn:1', 'P 2 2 -1n', '48:1', 'D2h^2'],
['Pnnn:2', '-P 2ab 2bc', '48:2', 'D2h^2'],
['Pccm', '-P 2 2c', '49', 'D2h^3'],
['Pmaa', '-P 2a 2', '49:cab', 'D2h^3'],
['Pbmb', '-P 2b 2b', '49:bca', 'D2h^3'],
['Pban:1', 'P 2 2 -1ab', '50:1', 'D2h^4'],
['Pban:2', '-P 2ab 2b', '50:2', 'D2h^4'],
['Pncb:1', 'P 2 2 -1bc', '50:1cab', 'D2h^4'],
['Pncb:2', '-P 2b 2bc', '50:2cab', 'D2h^4'],
['Pcna:1', 'P 2 2 -1ac', '50:1bca', 'D2h^4'],
['Pcna:2', '-P 2a 2c', '50:2bca', 'D2h^4'],
['Pmma', '-P 2a 2a', '51', 'D2h^5'],
['Pmmb', '-P 2b 2', '51:ba-c', 'D2h^5'],
['Pbmm', '-P 2 2b', '51:cab', 'D2h^5'],
['Pcmm', '-P 2c 2c', '51:-cba', 'D2h^5'],
['Pmcm', '-P 2c 2', '51:bca', 'D2h^5'],
['Pmam', '-P 2 2a', '51:a-cb', 'D2h^5'],
['Pnna', '-P 2a 2bc', '52', 'D2h^6'],
['Pnnb', '-P 2b 2n', '52:ba-c', 'D2h^6'],
['Pbnn', '-P 2n 2b', '52:cab', 'D2h^6'],
['Pcnn', '-P 2ab 2c', '52:-cba', 'D2h^6'],
['Pncn', '-P 2ab 2n', '52:bca', 'D2h^6'],
['Pnan', '-P 2n 2bc', '52:a-cb', 'D2h^6'],
['Pmna', '-P 2ac 2', '53', 'D2h^7'],
['Pnmb', '-P 2bc 2bc', '53:ba-c', 'D2h^7'],
['Pbmn', '-P 2ab 2ab', '53:cab', 'D2h^7'],
['Pcnm', '-P 2 2ac', '53:-cba', 'D2h^7'],
['Pncm', '-P 2 2bc', '53:bca', 'D2h^7'],
['Pman', '-P 2ab 2', '53:a-cb', 'D2h^7'],
['Pcca', '-P 2a 2ac', '54', 'D2h^8'],
['Pccb', '-P 2b 2c', '54:ba-c', 'D2h^8'],
['Pbaa', '-P 2a 2b', '54:cab', 'D2h^8'],
['Pcaa', '-P 2ac 2c', '54:-cba', 'D2h^8'],
['Pbcb', '-P 2bc 2b', '54:bca', 'D2h^8'],
['Pbab', '-P 2b 2ab', '54:a-cb', 'D2h^8'],
['Pbam', '-P 2 2ab', '55', 'D2h^9'],
['Pmcb', '-P 2bc 2', '55:cab', 'D2h^9'],
['Pcma', '-P 2ac 2ac', '55:bca', 'D2h^9'],
['Pccn', '-P 2ab 2ac', '56', 'D2h^10'],
['Pnaa', '-P 2ac 2bc', '56:cab', 'D2h^10'],
['Pbnb', '-P 2bc 2ab', '56:bca', 'D2h^10'],
['Pbcm', '-P 2c 2b', '57', 'D2h^11'],
['Pcam', '-P 2c 2ac', '57:ba-c', 'D2h^11'],
['Pmca', '-P 2ac 2a', '57:cab', 'D2h^11'],
['Pmab', '-P 2b 2a', '57:-cba', 'D2h^11'],
['Pbma', '-P 2a 2ab', '57:bca', 'D2h^11'],
['Pcmb', '-P 2bc 2c', '57:a-cb', 'D2h^11'],
['Pnnm', '-P 2 2n', '58', 'D2h^12'],
['Pmnn', '-P 2n 2', '58:cab', 'D2h^12'],
['Pnmn', '-P 2n 2n', '58:bca', 'D2h^12'],
['Pmmn:1', 'P 2 2ab -1ab', '59:1', 'D2h^13'],
['Pmmn:2', '-P 2ab 2a', '59:2', 'D2h^13'],
['Pnmm:1', 'P 2bc 2 -1bc', '59:1cab', 'D2h^13'],
['Pnmm:2', '-P 2c 2bc', '59:2cab', 'D2h^13'],
['Pmnm:1', 'P 2ac 2ac -1ac', '59:1bca', 'D2h^13'],
['Pmnm:2', '-P 2c 2a', '59:2bca', 'D2h^13'],
['Pbcn', '-P 2n 2ab', '60', 'D2h^14'],
['Pcan', '-P 2n 2c', '60:ba-c', 'D2h^14'],
['Pnca', '-P 2a 2n', '60:cab', 'D2h^14'],
['Pnab', '-P 2bc 2n', '60:-cba', 'D2h^14'],
['Pbna', '-P 2ac 2b', '60:bca', 'D2h^14'],
['Pcnb', '-P 2b 2ac', '60:a-cb', 'D2h^14'],
['Pbca', '-P 2ac 2ab', '61', 'D2h^15'],
['Pcab', '-P 2bc 2ac', '61:ba-c', 'D2h^15'],
['Pnma', '-P 2ac 2n', '62', 'D2h^16'],
['Pmnb', '-P 2bc 2a', '62:ba-c', 'D2h^16'],
['Pbnm', '-P 2c 2ab', '62:cab', 'D2h^16'],
['Pcmn', '-P 2n 2ac', '62:-cba', 'D2h^16'],
['Pmcn', '-P 2n 2a', '62:bca', 'D2h^16'],
['Pnam', '-P 2c 2n', '62:a-cb', 'D2h^16'],
['Cmcm', '-C 2c 2', '63', 'D2h^17'],
['Ccmm', '-C 2c 2c', '63:ba-c', 'D2h^17'],
['Amma', '-A 2a 2a', '63:cab', 'D2h^17'],
['Amam', '-A 2 2a', '63:-cba', 'D2h^17'],
['Bbmm', '-B 2 2b', '63:bca', 'D2h^17'],
['Bmmb', '-B 2b 2', '63:a-cb', 'D2h^17'],
['Cmca', '-C 2bc 2', '64', 'D2h^18'],
['Ccmb', '-C 2bc 2bc', '64:ba-c', 'D2h^18'],
['Abma', '-A 2ac 2ac', '64:cab', 'D2h^18'],
['Acam', '-A 2 2ac', '64:-cba', 'D2h^18'],
['Bbcm', '-B 2 2bc', '64:bca', 'D2h^18'],
['Bmab', '-B 2bc 2', '64:a-cb', 'D2h^18'],
['Cmmm', '-C 2 2', '65', 'D2h^19'],
['Ammm', '-A 2 2', '65:cab', 'D2h^19'],
['Bmmm', '-B 2 2', '65:bca', 'D2h^19'],
['Cccm', '-C 2 2c', '66', 'D2h^20'],
['Amaa', '-A 2a 2', '66:cab', 'D2h^20'],
['Bbmb', '-B 2b 2b', '66:bca', 'D2h^20'],
['Cmma', '-C 2b 2', '67', 'D2h^21'],
['Cmmb', '-C 2b 2b', '67:ba-c', 'D2h^21'],
['Abmm', '-A 2c 2c', '67:cab', 'D2h^21'],
['Acmm', '-A 2 2c', '67:-cba', 'D2h^21'],
['Bmcm', '-B 2 2c', '67:bca', 'D2h^21'],
['Bmam', '-B 2c 2', '67:a-cb', 'D2h^21'],
['Ccca:1', 'C 2 2 -1bc', '68:1', 'D2h^22'],
['Ccca:2', '-C 2b 2bc', '68:2', 'D2h^22'],
['Cccb:1', 'C 2 2 -1bc', '68:1ba-c', 'D2h^22'],
['Cccb:2', '-C 2b 2c', '68:2ba-c', 'D2h^22'],
['Abaa:1', 'A 2 2 -1ac', '68:1cab', 'D2h^22'],
['Abaa:2', '-A 2a 2c', '68:2cab', 'D2h^22'],
['Acaa:1', 'A 2 2 -1ac', '68:1-cba', 'D2h^22'],
['Acaa:2', '-A 2ac 2c', '68:2-cba', 'D2h^22'],
['Bbcb:1', 'B 2 2 -1bc', '68:1bca', 'D2h^22'],
['Bbcb:2', '-B 2bc 2b', '68:2bca', 'D2h^22'],
['Bbab:1', 'B 2 2 -1bc', '68:1a-cb', 'D2h^22'],
['Bbab:2', '-B 2b 2bc', '68:2a-cb', 'D2h^22'],
['Fmmm', '-F 2 2', '69', 'D2h^23'],
['Fddd:1', 'F 2 2 -1d', '70:1', 'D2h^24'],
['Fddd:2', '-F 2uv 2vw', '70:2', 'D2h^24'],
['Immm', '-I 2 2', '71', 'D2h^25'],
['Ibam', '-I 2 2c', '72', 'D2h^26'],
['Imcb', '-I 2a 2', '72:cab', 'D2h^26'],
['Icma', '-I 2b 2b', '72:bca', 'D2h^26'],
['Ibca', '-I 2b 2c', '73', 'D2h^27'],
['Icab', '-I 2a 2b', '73:ba-c', 'D2h^27'],
['Imma', '-I 2b 2', '74', 'D2h^28'],
['Immb', '-I 2a 2a', '74:ba-c', 'D2h^28'],
['Ibmm', '-I 2c 2c', '74:cab', 'D2h^28'],
['Icmm', '-I 2 2b', '74:-cba', 'D2h^28'],
['Imcm', '-I 2 2a', '74:bca', 'D2h^28'],
['Imam', '-I 2c 2', '74:a-cb', 'D2h^28'],
['P4', 'P 4', '75', 'C4^1'],
['P41', 'P 4w', '76', 'C4^2'],
['P42', 'P 4c', '77', 'C4^3'],
['P43', 'P 4cw', '78', 'C4^4'],
['I4', 'I 4', '79', 'C4^5'],
['I41', 'I 4bw', '80', 'C4^6'],
['P-4', 'P -4', '81', 'S4^1'],
['I-4', 'I -4', '82', 'S4^2'],
['P4/m', '-P 4', '83', 'C4h^1'],
['P42/m', '-P 4c', '84', 'C4h^2'],
['P4/n:1', 'P 4ab -1ab', '85:1', 'C4h^3'],
['P4/n:2', '-P 4a', '85:2', 'C4h^3'],
['P42/n:1', 'P 4n -1n', '86:1', 'C4h^4'],
['P42/n:2', '-P 4bc', '86:2', 'C4h^4'],
['I4/m', '-I 4', '87', 'C4h^5'],
['I41/a:1', 'I 4bw -1bw', '88:1', 'C4h^6'],
['I41/a:2', '-I 4ad', '88:2', 'C4h^6'],
['P422', 'P 4 2', '89', 'D4^1'],
['P4212', 'P 4ab 2ab', '90', 'D4^2'],
['P4122', 'P 4w 2c', '91', 'D4^3'],
['P41212', 'P 4abw 2nw', '92', 'D4^4'],
['P4222', 'P 4c 2', '93', 'D4^5'],
['P42212', 'P 4n 2n', '94', 'D4^6'],
['P4322', 'P 4cw 2c', '95', 'D4^7'],
['P43212', 'P 4nw 2abw', '96', 'D4^8'],
['I422', 'I 4 2', '97', 'D4^9'],
['I4122', 'I 4bw 2bw', '98', 'D4^10'],
['P4mm', 'P 4 -2', '99', 'C4v^1'],
['P4bm', 'P 4 -2ab', '100', 'C4v^2'],
['P42cm', 'P 4c -2c', '101', 'C4v^3'],
['P42nm', 'P 4n -2n', '102', 'C4v^4'],
['P4cc', 'P 4 -2c', '103', 'C4v^5'],
['P4nc', 'P 4 -2n', '104', 'C4v^6'],
['P42mc', 'P 4c -2', '105', 'C4v^7'],
['P42bc', 'P 4c -2ab', '106', 'C4v^8'],
['I4mm', 'I 4 -2', '107', 'C4v^9'],
['I4cm', 'I 4 -2c', '108', 'C4v^10'],
['I41md', 'I 4bw -2', '109', 'C4v^11'],
['I41cd', 'I 4bw -2c', '110', 'C4v^12'],
['P-42m', 'P -4 2', '111', 'D2d^1'],
['P-42c', 'P -4 2c', '112', 'D2d^2'],
['P-421m', 'P -4 2ab', '113', 'D2d^3'],
['P-421c', 'P -4 2n', '114', 'D2d^4'],
['P-4m2', 'P -4 -2', '115', 'D2d^5'],
['P-4c2', 'P -4 -2c', '116', 'D2d^6'],
['P-4b2', 'P -4 -2ab', '117', 'D2d^7'],
['P-4n2', 'P -4 -2n', '118', 'D2d^8'],
['I-4m2', 'I -4 -2', '119', 'D2d^9'],
['I-4c2', 'I -4 -2c', '120', 'D2d^10'],
['I-42m', 'I -4 2', '121', 'D2d^11'],
['I-42d', 'I -4 2bw', '122', 'D2d^12'],
['P4/mmm', '-P 4 2', '123', 'D4h^1'],
['P4/mcc', '-P 4 2c', '124', 'D4h^2'],
['P4/nbm:1', 'P 4 2 -1ab', '125:1', 'D4h^3'],
['P4/nbm:2', '-P 4a 2b', '125:2', 'D4h^3'],
['P4/nnc:1', 'P 4 2 -1n', '126:1', 'D4h^4'],
['P4/nnc:2', '-P 4a 2bc', '126:2', 'D4h^4'],
['P4/mbm', '-P 4 2ab', '127', 'D4h^5'],
['P4/mnc', '-P 4 2n', '128', 'D4h^6'],
['P4/nmm:1', 'P 4ab 2ab -1ab', '129:1', 'D4h^7'],
['P4/nmm:2', '-P 4a 2a', '129:2', 'D4h^7'],
['P4/ncc:1', 'P 4ab 2n -1ab', '130:1', 'D4h^8'],
['P4/ncc:2', '-P 4a 2ac', '130:2', 'D4h^8'],
['P42/mmc', '-P 4c 2', '131', 'D4h^9'],
['P42/mcm', '-P 4c 2c', '132', 'D4h^10'],
['P42/nbc:1', 'P 4n 2c -1n', '133:1', 'D4h^11'],
['P42/nbc:2', '-P 4ac 2b', '133:2', 'D4h^11'],
['P42/nnm:1', 'P 4n 2 -1n', '134:1', 'D4h^12'],
['P42/nnm:2', '-P 4ac 2bc', '134:2', 'D4h^12'],
['P42/mbc', '-P 4c 2ab', '135', 'D4h^13'],
['P42/mnm', '-P 4n 2n', '136', 'D4h^14'],
['P42/nmc:1', 'P 4n 2n -1n', '137:1', 'D4h^15'],
['P42/nmc:2', '-P 4ac 2a', '137:2', 'D4h^15'],
['P42/ncm:1', 'P 4n 2ab -1n', '138:1', 'D4h^16'],
['P42/ncm:2', '-P 4ac 2ac', '138:2', 'D4h^16'],
['I4/mmm', '-I 4 2', '139', 'D4h^17'],
['I4/mcm', '-I 4 2c', '140', 'D4h^18'],
['I41/amd:1', 'I 4bw 2bw -1bw', '141:1', 'D4h^19'],
['I41/amd:2', '-I 4bd 2', '141:2', 'D4h^19'],
['I41/acd:1', 'I 4bw 2aw -1bw', '142:1', 'D4h^20'],
['I41/acd:2', '-I 4bd 2c', '142:2', 'D4h^20'],
['P3', 'P 3', '143', 'C3^1'],
['P31', 'P 31', '144', 'C3^2'],
['P32', 'P 32', '145', 'C3^3'],
['R3:H', 'R 3', '146:H', 'C3^4'],
['R3:R', 'P 3*', '146:R', 'C3^4'],
['P-3', '-P 3', '147', 'C3i^1'],
['R-3:H', '-R 3', '148:H', 'C3i^2'],
['R-3:R', '-P 3*', '148:R', 'C3i^2'],
['P312', 'P 3 2', '149', 'D3^1'],
['P321', 'P 3 2"', '150', 'D3^2'],
['P3112', 'P 31 2c (0 0 1)', '151', 'D3^3'],
['P3121', 'P 31 2"', '152', 'D3^4'],
['P3212', 'P 32 2c (0 0 -1)', '153', 'D3^5'],
['P3221', 'P 32 2"', '154', 'D3^6'],
['R32:H', 'R 3 2"', '155:H', 'D3^7'],
['R32:R', 'P 3* 2', '155:R', 'D3^7'],
['P3m1', 'P 3 -2"', '156', 'C3v^1'],
['P31m', 'P 3 -2', '157', 'C3v^2'],
['P3c1', 'P 3 -2"c', '158', 'C3v^3'],
['P31c', 'P 3 -2c', '159', 'C3v^4'],
['R3m:H', 'R 3 -2"', '160:H', 'C3v^5'],
['R3m:R', 'P 3* -2', '160:R', 'C3v^5'],
['R3c:H', 'R 3 -2"c', '161:H', 'C3v^6'],
['R3c:R', 'P 3* -2n', '161:R', 'C3v^6'],
['P-31m', '-P 3 2', '162', 'D3d^1'],
['P-31c', '-P 3 2c', '163', 'D3d^2'],
['P-3m1', '-P 3 2"', '164', 'D3d^3'],
['P-3c1', '-P 3 2"c', '165', 'D3d^4'],
['R-3m:H', '-R 3 2"', '166:H', 'D3d^5'],
['R-3m:R', '-P 3* 2', '166:R', 'D3d^5'],
['R-3c:H', '-R 3 2"c', '167:H', 'D3d^6'],
['R-3c:R', '-P 3* 2n', '167:R', 'D3d^6'],
['P6', 'P 6', '168', 'C6^1'],
['P61', 'P 61', '169', 'C6^2'],
['P65', 'P 65', '170', 'C6^3'],
['P62', 'P 62', '171', 'C6^4'],
['P64', 'P 64', '172', 'C6^5'],
['P63', 'P 6c', '173', 'C6^6'],
['P-6', 'P -6', '174', 'C3h^1'],
['P6/m', '-P 6', '175', 'C6h^1'],
['P63/m', '-P 6c', '176', 'C6h^2'],
['P622', 'P 6 2', '177', 'D6^1'],
['P6122', 'P 61 2 (0 0 -1)', '178', 'D6^2'],
['P6522', 'P 65 2 (0 0 1)', '179', 'D6^3'],
['P6222', 'P 62 2c (0 0 1)', '180', 'D6^4'],
['P6422', 'P 64 2c (0 0 01)', '181', 'D6^5'],
['P6322', 'P 6c 2c', '182', 'D6^6'],
['P6mm', 'P 6 -2', '183', 'C6v^1'],
['P6cc', 'P 6 -2c', '184', 'C6v^2'],
['P63cm', 'P 6c -2', '185', 'C6v^3'],
['P63mc', 'P 6c -2c', '186', 'C6v^4'],
['P-6m2', 'P -6 2', '187', 'D3h^1'],
['P-6c2', 'P -6c 2', '188', 'D3h^2'],
['P-62m', 'P -6 -2', '189', 'D3h^3'],
['P-62c', 'P -6c -2c', '190', 'D3h^4'],
['P6/mmm', '-P 6 2', '191', 'D6h^1'],
['P6/mcc', '-P 6 2c', '192', 'D6h^2'],
['P63/mcm', '-P 6c 2', '193', 'D6h^3'],
['P63/mmc', '-P 6c 2c', '194', 'D6h^4'],
['P23', 'P 2 2 3', '195', 'T^1'],
['F23', 'F 2 2 3', '196', 'T^2'],
['I23', 'I 2 2 3', '197', 'T^3'],
['P213', 'P 2ac 2ab 3', '198', 'T^4'],
['I213', 'I 2b 2c 3', '199', 'T^5'],
['Pm-3', '-P 2 2 3', '200', 'Th^1'],
['Pn-3:1', 'P 2 2 3 -1n', '201:1', 'Th^2'],
['Pn-3:2', '-P 2ab 2bc 3', '201:2', 'Th^2'],
['Fm-3', '-F 2 2 3', '202', 'Th^3'],
['Fd-3:1', 'F 2 2 3 -1d', '203:1', 'Th^4'],
['Fd-3:2', '-F 2uv 2vw 3', '203:2', 'Th^4'],
['Im-3', '-I 2 2 3', '204', 'Th^5'],
['Pa-3', '-P 2ac 2ab 3', '205', 'Th^6'],
['Ia-3', '-I 2b 2c 3', '206', 'Th^7'],
['P432', 'P 4 2 3', '207', 'O^1'],
['P4232', 'P 4n 2 3', '208', 'O^2'],
['F432', 'F 4 2 3', '209', 'O^3'],
['F4132', 'F 4d 2 3', '210', 'O^4'],
['I432', 'I 4 2 3', '211', 'O^5'],
['P4332', 'P 4acd 2ab 3', '212', 'O^6'],
['P4132', 'P 4bd 2ab 3', '213', 'O^7'],
['I4132', 'I 4bd 2c 3', '214', 'O^8'],
['P-43m', 'P -4 2 3', '215', 'Td^1'],
['F-43m', 'F -4 2 3', '216', 'Td^2'],
['I-43m', 'I -4 2 3', '217', 'Td^3'],
['P-43n', 'P -4n 2 3', '218', 'Td^4'],
['F-43c', 'F -4c 2 3', '219', 'Td^5'],
['I-43d', 'I -4bd 2c 3', '220', 'Td^6'],
['Pm-3m', '-P 4 2 3', '221', 'Oh^1'],
['Pn-3n:1', 'P 4 2 3 -1n', '222:1', 'Oh^2'],
['Pn-3n:2', '-P 4a 2bc 3', '222:2', 'Oh^2'],
['Pm-3n', '-P 4n 2 3', '223', 'Oh^3'],
['Pn-3m:1', 'P 4n 2 3 -1n', '224:1', 'Oh^4'],
['Pn-3m:2', '-P 4bc 2bc 3', '224:2', 'Oh^4'],
['Fm-3m', '-F 4 2 3', '225', 'Oh^5'],
['Fm-3c', '-F 4c 2 3', '226', 'Oh^6'],
['Fd-3m:1', 'F 4d 2 3 -1d', '227:1', 'Oh^7'],
['Fd-3m:2', '-F 4vw 2vw 3', '227:2', 'Oh^7'],
['Fd-3c:1', 'F 4d 2 3 -1cd', '228:1', 'Oh^8'],
['Fd-3c:2', '-F 4cvw 2vw 3', '228:2', 'Oh^8'],
['Im-3m', '-I 4 2 3', '229', 'Oh^9'],
['Ia-3d', '-I 4bd 2c 3', '230', 'Oh^10'],
]

print """
# This module has been generated automatically from space group information
# obtained from the Computational Crystallography Toolbox
#

from Scientific.Geometry import Vector, Tensor
from Scientific.Geometry.Transformation import Rotation, Translation
from Scientific import N

class SpaceGroup(object):

    def __init__(self, number, labels, transformations):
        self.number = number
        self.labels = labels
        self.transformations = []
        for rot, trans in transformations:
            self.transformations.append(Translation(trans)*Rotation(Tensor(rot)))

_space_group_table = {}

def getSpaceGroupTransformations(space_group_label_or_number):
    try:
        return _space_group_table[space_group_label_or_number].transformations
    except KeyError:
        pass
    space_group_label = ''.join(space_group_label_or_number.split())
    return _space_group_table[space_group_label].transformations
"""

# The space group label list has many groups more than once (there are only
# 230 space groups), so we need to identify the unique space groups first.
space_groups = {}
space_group_names = {}
for labels in space_group_labels:
    for i in range(len(labels)):
        if not ':' in labels[i]:
            l = labels[i]
            del labels[i]
            labels.insert(0, l)
    sgi = space_group_info(labels[0])
    unique_name = sgi.symbol_and_number() 
    start = unique_name.find('(No. ')
    end = unique_name.find(')')
    number = int(unique_name[start+5:end])
    space_groups[number] = sgi
    space_group_names[number] = \
              space_group_names.get(number, []) + labels

# Space groups with reflection operations cannot occur in biomolecular
# crystals, so remove them.
to_remove = []
for number in space_groups.keys():
    sgi = space_groups[number]
    for symmetry_transformation in sgi.group():
        rot =  symmetry_transformation.as_rational().r
        rot = N.array(rot)
        rot.shape = (3, 3)
        det = rot[0,0] * (rot[1, 1]*rot[2,2]-rot[1,2]*rot[2,1]) \
            - rot[0,1] * (rot[1, 0]*rot[2,2]-rot[1,2]*rot[2,0]) \
            + rot[0,2] * (rot[1, 0]*rot[2,1]-rot[1,1]*rot[2,0])
        assert abs(det) == 1
        if det < 0:
            to_remove.append(number)
            break

for number in to_remove:
    del space_groups[number]
    del space_group_names[number]

# Generate the code lines for the remaining space groups
for number in space_groups.keys():
    space_group_table_entry(number, space_group_names[number],
                            space_groups[number])

print """
del transformations
del rot
del trans
"""
