#import <IOBluetooth/objc/IOBluetoothDevice.h>
#import <BTUtil/BTUtil.h>

#import "SimpleOBEXServer.h"

@implementation SimpleOBEXServer

- (void)awakeFromNib
{
	myOBEXServer = [[BBOBEXServer alloc] initWithDelegate:self];
	[BBOBEXServer setDebug:YES];	// print lots of debug messages
    
    [rootDirectory setStringValue:[[NSFileManager defaultManager] currentDirectoryPath]];
}

- (IBAction)startOrStop:(id)sender
{
	if ([myOBEXServer isRunning]) {
		[self stop:nil];
	} else {
		[self start:nil];		
	}
}

//
// Starts the OBEX server.
//
- (IBAction)start:(id)sender
{
	[self log:@"Starting server..."];
	
	// advertise an OBEX service so other devices can find us
	BluetoothRFCOMMChannelID serverChannelID;
	IOReturn advertised = 	
		[BBServiceAdvertiser addRFCOMMServiceDictionary:[BBServiceAdvertiser objectPushProfileDictionary]
											   withName:[serviceName stringValue]
												   UUID:nil
											  channelID:&serverChannelID
									serviceRecordHandle:&serviceRecordHandle];
	if (advertised != kIOReturnSuccess) {
		[self log:@"Error advertising service!"];
		return;
	}

	// start the server
	[myOBEXServer startOnChannelID:serverChannelID];
	
	[self log:[NSString stringWithFormat:@"Server started on channel %d.", serverChannelID]];
	[startOrStopButton setTitle:@"Stop server"];
}

//
// Stops the OBEX server.
//
- (IBAction)stop:(id)sender
{
	[self log:@"Stopping server..."];	
	
	// stop the server
	[myOBEXServer stop];
	
	// stop advertising the OBEX service
	[BBServiceAdvertiser removeService:serviceRecordHandle];
	
	[self log:@"Server stopped."];	
	[startOrStopButton setTitle:@"Start server"];	
}

//
// Displays debug text.
//
- (void)log:(NSString *)logText
{
	[logTextView insertText:logText];
	[logTextView insertText:@"\n"];	
}

//
// Creates a new file with the given filename, and returns its file handle.
//
- (NSFileHandle *)getFileHandleForReceivingFile:(NSString *)filename
{
	NSString *fullPath = [[rootDirectory stringValue] stringByAppendingPathComponent:filename];
	
	// create a local file
	BOOL created = [[NSFileManager defaultManager] createFileAtPath:fullPath
														   contents:nil 
														 attributes:nil];
	if (created) {
		return [NSFileHandle fileHandleForWritingAtPath:fullPath];
	} else {
		[self log:@"Error creating file to receive data, can't accept the file."];
		return nil;
	}
}

//
// Finds the file with the given filename, and returns its file handle.
//
- (NSFileHandle *)getFileHandleForSendingFile:(NSString *)filename
{
	NSString *fullPath = [[rootDirectory stringValue] stringByAppendingPathComponent:filename];
	
	// this method will return nil if the file does not exist, which is fine
	return [NSFileHandle fileHandleForReadingAtPath:fullPath];	
}


#pragma mark -
#pragma mark BBOBEXServerSession delegate methods

// These delegate methods will be called by the BBOBEXServerSession instance 
// when an OBEX server event occurs. This is where you can actually react and 
// perform operations when a client connects, sends files, etc. 
//
// For the sake of simplicity, the delegate methods relating to ABORT and 
// SETPATH operations have not been implemented here, which means that this 
// server won't support these particular operations.


//
// Called when a connection is succesfully established.
//
- (void)obexServerSessionConnectComplete:(BBOBEXServerSession *)session
{
	IOBluetoothDevice *remoteDevice = [[session getRFCOMMChannel] getDevice];
	[self log:[NSString stringWithFormat:@"A client (%@) has connected.", 
		[remoteDevice getAddressString]]];
}

//
// Called when a session is succesfully disconnected.
//
- (void)obexServerSessionDisconnectComplete:(BBOBEXServerSession *)session
{
	[self log:@"A client has disconnected."];
}

//
// Called when a PUT request is received.
//
- (int)obexServerSessionPutRequest:(BBOBEXServerSession *)session
						  withName:(NSString *)name
							  type:(NSString *)type
							length:(unsigned)totalLength
					   writeToFile:(NSFileHandle **)destFile
{
	[self log:[NSString stringWithFormat:@"A client wants to send a file '%@' (%d bytes).", 
		name, totalLength]];		
	
	// Create a local file to receive the file data from the client.
	NSFileHandle *localDestFile = [self getFileHandleForReceivingFile:name];
	
	if (localDestFile == nil) {
		// there was an error creating the file
		return kOBEXResponseCodeInternalServerErrorWithFinalBit;
	} else {
		// tell the server session to write received data into this new file
		*destFile = localDestFile;
		return kOBEXSuccess;
	}
}

//
// Called each time data is received for a PUT request.
//
- (int)obexServerSessionPutProgress:(BBOBEXServerSession *)session
							forFile:(NSFileHandle *)destFile
							byAmount:(unsigned)bytecount
{
	[self log:[NSString stringWithFormat:@"Received another %d bytes...", bytecount]];
	return kOBEXSuccess;
}

//
// Called when a PUT operation is successfully completed.
//
- (void)obexServerSessionPutComplete:(BBOBEXServerSession *)session
							 forFile:(NSFileHandle *)destFile;
{
	[self log:@"File successfully received."];
}

//
// Called when a GET request is received.
//
- (int)obexServerSessionGetRequest:(BBOBEXServerSession *)session
						  withName:(NSString *)name
							  type:(NSString *)type
					  readFromFile:(NSFileHandle **)sourceFile
{
	[self log:[NSString stringWithFormat:@"A client wants to retrieve the file '%@'.", name]];		

	NSFileHandle *localSourceFile = [self getFileHandleForSendingFile:name];
	if (localSourceFile == nil) {
		// the requested file cannot be found
		return kOBEXResponseCodeNotFoundWithFinalBit;
	} else {
		// found the local file, tell the server session to send this file to 
		// the client
		*sourceFile = localSourceFile;
		return kOBEXSuccess;		
	}
}

//
// Called each time data is sent for a GET request.
//
- (int)obexServerSessionGetProgress:(BBOBEXServerSession *)session
							forFile:(NSFileHandle *)sourceFile
						   byAmount:(unsigned)bytecount
{
	[self log:[NSString stringWithFormat:@"Sent another %d bytes...", bytecount]];	
	return kOBEXSuccess;
}

//
// Called when a GET operation is successfully completed.
//
- (void)obexServerSessionGetComplete:(BBOBEXServerSession *)session
							 forFile:(NSFileHandle *)sourceFile;
{
	[self log:@"File successfully sent."];
}

//
// Called when an error occurs while running the obex server.
//
- (void)obexServerSessionError:(BBOBEXServerSession *)session
						 error:(OBEXError)error 
					   summary:(NSString *)summary
{
	[self log:[NSString stringWithFormat:@"An error has occured! Error was: %@ (%d)", summary, error]];
}


#pragma mark -

- (void)windowWillClose:(NSNotification *)aNotification
{
	// Make sure the server is stopped before shutting down this app.
	if ([myOBEXServer isRunning]) {
		[self stop:nil];
	}
}

#pragma mark -

- (void)dealloc
{
	[myOBEXServer release];
	[super dealloc];
}

@end
