"""
  Filename:  glpk.py (define a class for calling GLPK from Python)

-- This code is part of the Python-GLPK interface.
--
-- Copyright (C) 2005-2009, Joao Pedro Pedroso
-- Faculdade de Ciencias, Universidade do Porto
-- Porto, Portugal. All rights reserved. E-mail: <jpp@fc.up.pt>.
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
-- 02110-1301, USA.
"""

from glpkpi import *
from array import *
LOG = True

class glpk:
    def __del__(self):
        if self.tran != None:
            glp_mpl_free_wksp(self.tran);
        glp_delete_prob(self.lp);

    def __init__(self, mod_file = None, dat_file = None):
        ### # parameter structure:	NOT USED YET
	### self.parm = glp_smcp()
	### self.parm.msg_lev = GLP_MSG_OFF;	# silent modes
        self.tran = None

        if mod_file == None:
            print "creating problem -- no file associated"
            self.lp = glp_create_prob();
        elif mod_file.find(".mps") != -1:
            print "creating problem from mps file"
            self.lp = glp_create_prob();
            ret = glp_read_mps(self.lp, GLP_MPS_FILE, None, mod_file);	# self.parm
            if ret != 0:
                print "Error reading model", mod_file
                raise IOError
        elif mod_file.find(".mod") != -1:
            print "creating problem from mod file"
            self.lp = glp_create_prob();
            self.tran = glp_mpl_alloc_wksp();
            if dat_file != None:
                print "reading model 1"
                ret = glp_mpl_read_model(self.tran, mod_file, 1)
                if ret != 0:
                    print "Error reading model", mod_file
                    raise IOError
                print "reading data"
                ret = glp_mpl_read_data(self.tran, dat_file)
                if ret != 0:
                    print "Error reading data", dat_file
                    raise IOError
            else:
                print "reading model 0"
                ret = glp_mpl_read_model(self.tran, mod_file, 0);
                if ret != 0:
                    print "Error reading model", mod_file
                    raise IOError
            ret = glp_mpl_generate(self.tran, None)
            if ret != 0:
                print "Error generating model"
                raise IOError
            glp_mpl_build_prob(self.tran, self.lp)

        elif mod_file.find(".lp") != -1:
            print "creating problem from lp file"
            self.lp = glp_create_prob();
            ret = glp_read_lp(self.lp, None, mod_file);	# !!! self.parm

        self.names = {}		# map variable names to indices (for integer vars)
        self.indices = {}	# map variable indices to names (for integer vars)
        self.cvars = []		# continuous variables
        self.ivars = []		# integer variables
        self.bvars = []		# binary variables
        self.nvars = glp_get_num_cols(self.lp)
        if LOG: print "number of vars:", self.nvars
        self.lb = []		# lower bounds
        self.ub = []		# upper bounds
        for i in range(1,self.nvars+1):
            name = glp_get_col_name(self.lp, i)
            if glp_get_col_kind(self.lp, i) == GLP_CV:
                self.cvars.append(i)
            elif glp_get_col_kind(self.lp, i) == GLP_IV:
                self.names[i] = name
                self.indices[name] = i
                self.ivars.append(i)
                self.lb.append(glp_get_col_lb(self.lp, i))
                self.ub.append(glp_get_col_ub(self.lp, i))
            elif glp_get_col_kind(self.lp, i) == GLP_BV:
                self.names[i] = name
                self.indices[name] = i
                self.bvars.append(i)
                self.lb.append(glp_get_col_lb(self.lp, i))
                self.ub.append(glp_get_col_ub(self.lp, i))
            else:
                print "unkown col kind", glp_get_col_kind(self.lp, i)
                raise AttributeError
        

    def solve(self):
        if glp_get_num_int(self.lp) == 0:	# problem is integer
            glp_simplex(self.lp, None)	# self.parm !!!
        else:	# problem is MIP
            if self.tran:
                glp_mpl_build_prob(self.tran, self.lp);
            glp_simplex(self.lp, None);  # ??? should use dual simplex ???
            glp_intopt(self.lp, None);
            if self.tran:
                ret = glp_mpl_postsolve(self.tran, self.lp, GLP_MIP);
                if ret != 0:
                    print "Error on postsolving model"
                    raise AttributeError
        print "solved problem"
        self.instanciate_solution()


    def instanciate_solution(self):
        """update self.sol and self.x"""
        sol = {}
        x = array('d', [0. for i in range(self.nvars+1)])
        name = glp_get_obj_name(self.lp)
        if glp_get_num_int(self.lp) == 0:	# problem is integer
            value = glp_get_obj_val(self.lp)
            x[0] = value
            sol[name] = value
            for i in range(1,glp_get_num_cols(self.lp)+1):
                name = glp_get_col_name(self.lp, i)
                value = glp_get_col_prim(self.lp, i)
                sol[name] = value
                x[i] = value
        else:	# problem is MIP
            value = glp_mip_obj_val(self.lp)
            x[0] = value
            sol[name] = value
            for i in range(1,glp_get_num_cols(self.lp)+1):
                name = glp_get_col_name(self.lp, i)
                if glp_get_col_kind(self.lp, i) == GLP_CV:
                    value = glp_mip_col_val(self.lp, i)
                elif glp_get_col_kind(self.lp, i) in [GLP_IV, GLP_BV]:
                    # to avoid finite precision problems:
                    value = int(round(glp_mip_col_val(self.lp, i),0))
                else:
                    print "unkown col kind"
                    raise AttributeError
                sol[name] = value
                x[i] = value

        self.sol = sol
        self.x = x

if __name__ == "__main__":
    print "starting..."
    print "testing MATH models"
    glp_no_print()
    print "deactivating glpk terminal print"
    example = glpk("examples/assign.mod")
    example.solve()
    print "solution:", example.sol
    print "solution is also here:", example.x
    print "succeeded\n\n\n"
    
    print "testing MPS models"
    print "reactivating glpk terminal print"
    glp_do_print()
    example = glpk("examples/samp1.mps")
    example.solve()
    print "solution:", example.sol
    print "solution is also here:", example.x
    print "succeeded\n\n\n"

    print "testing LP models"
    example = glpk("examples/plan.lp")
    example.solve()
    print "solution:", example.sol
    print "solution is also here:", example.x
    print "succeeded\n\n\n"
        
