package AtomicData::List;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: List.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtomicData::List - virtual class which handles output and managemant of data

=head1 SYNOPSIS

  ...
  use AtomicData::List;

  my $output = "";
  my $field1 = AtomicData::List->('AtomicData::Text',$cgi->param('example1'));
  $field1->set_parameter(blank_ok => 0);
  my $errors = 0;
  my ($success,$problems) = $field1->verify();
  if (!$success) {
    $output .= 'You entered invalid data: '.join("\n",@$problems);
    $errors += @$problems;
  }
  $errors and return $output;

  my $field2 = new AtomicData::List->('AtomicData::Integer', 1, 2, 3);
  ...

=head1 DESCRIPTION

This class provides much of the functionality needed data encapsulating
objects.

Note that when discussing data here, we mean not only primitives such
as integer, float, character, string, etc., but also data best
distinguished by its function, e.g., social security numbers, routing
and trace numbers, street addresses, U.S. zipcodes, etc. We do not,
however, mean complex data such as programming objects.

AtomicData::List allows one to store, retrieve, verify and manage data
being used in HTML pages. The methods are fairly straight forward. The
actual data encapsulating is handled by another object class which has a
'used' relationship with AtomicData::List. AtomicData::List serves mostly
as a special I/O go between for the user and this data encapsulating
class, but does, however, do some verification of data and storage of
error messages. The necessary interface required of the data encapsulating
class by AtomicData::List is speced out below. For further details on data
verification, see the SEE ALSO section below.

Note that the data returned by retrieval functions is not guaranteed
to be exactly the same as the data entered. It may be canonicalized
(for its type), or reformated in some way, but will always be
equivalent to the original in intended value as per the encapsulating
data type specified. This will occur in one of the following two
cases: 1) some sort of formating request that would affect the value
of the underlying data encapsulator has been performed, or 2) the
underlying data type chooses to return data in its canonical form in
some cases.

=head1 DATA SPEC

AtomicData::List relies on atomic data encapsulators. It will internally
maintain a homogenous list of these corresponding to received
values. All data received by a single AtomicData::List object should be of the
same general type (be able to be stored in homogeneous encapsulators).

The data encapsulating class must have the following methods:

 ($object) new ($value)
  accepts a value and returns the data encapsulating object

 () set_value ($value)
  sets the value encapsulated by that object

 ($value) value ()
  returns the value stored by that object

 (boolean $success, string[] $problems) verify () 
  returns a boolean value for success and a list detailing the
  errors found.

 () set_format (\%hash)
  accepts a hash of key value pairs that specifies formating
  preferences of the value maintained; the meaningful keys and values
  are specified by the data encapsulating object itself

 () format (bool) 
  causes the data to be formated (see above); if set_format has never
  been called or has been reset, then the behavior is unspecifide
  except that the new data must be equivalent, if not in exactly the
  same form as the old WRT its type; this must affect the return of
  value called on the datatype when appropriate. This function will
  raise an exception if it is unable to cannocalize the current data
  form (see below), unless bool is set to true in which case errors
  will be ignored and the old form will be maintained unchanged..

 () parameters (\%hash)
  will set parameters to constrain what is considered valid data for
  the data type; the key/value pairs are specified by the data type
  itself; this will affect verification of data

 ($value) canonicalize ($value, bool)
  will accept a value proper the data encapsulators type and return it
  in a set canonical form (used in conjuction with test_equality
  below); will raise an exception if it is unable to canonicalize
  unless the bool is set to true, in which case errors will be ignored
  and the old form will be maintanined unchanged.

 ($bool) test_equality ($val1, $val2)
  will return true if $val1 and $val2 are canonically equivalent WRT
  the data types purpose; that is if we have a data type encapsulating
  U.S. dollars, then $1 aned 1.00 would be equivalent

=cut

use AtomicData;
@AtomicData::List::ISA = qw (AtomicData);

=head1 METHODS

In order to derive a non-virtual subclass, the public fucntion
C<html_input> must be implemented.

=head2 new

 class
 (AtomicData::List $obj) new
   (string $name, string $data_class, variable @values)

DESCRIPTION:

Returns an AtomicData::List object. This function will populate the internal
list of data with the values passed in via the @values array. If no
values are passed in, then the internal data will be populated with a
single data encapsulator encapsulating a null string. The data
encapsulating object class to be used to encapsulate is specified in
C<$data_class>. C<$name> is the value to be placed in the name
attribute of an input, select, submit, etc., HTML tag.

=cut

sub new {
  my ($proto, @arguments) = @_;
  my $this = $proto->SUPER::new(@arguments);
  return $this;
}

=head2 init

=cut

sub init {
  my ($this, $data_class, @values) = @_;

  # assign the data class.
  $data_class ||= "AtomicData";
  $this->set_data_class($data_class);

  # initialize data to clear.
  $this->clear();

  @values and $this->set_value(@values);
}

=head2 clear

=cut

sub clear {
  my ($this) = @_;
  $this->{_data}       = undef;
  $this->{_orig_data}  = undef;
}

=head2 create_datum

 instance
 (AtomicData $datum) create_datum ($value)

DESCRIPTION:

Returns an AtomicData instance.

=cut

sub create_datum {
  my ($this, $value) = @_;

  my $data_class = $this->data_class();
  my $datum = $data_class->new($value);

  $datum->reset_possible( $this->{_possible}, $this->{_ordering} );
  $datum->reset_parameters( $this->{_parameters} );
  $datum->reset_format( $this->{_format} );

  return $datum;
}

=head2 data

 instance
 (Data \@data) data ()

DESCRIPTION:

Will return a reference to a list of the internal data encapsulators.

=cut

sub data {
  my ($this) = @_;
  return $this->{_data} || [];
}

=head2 orig_data

 instance
 (Data \@data) orig_data ()

DESCRIPTION:

Will return a reference to a list of the internal data encapsulators.

=cut

sub orig_data {
  my ($this) = @_;
  return $this->{_orig_data} || $this->{_data} || [];
}

=head2 data_count

 instance
 (int $count) data_count ()

DESCRIPTION:

Returns the number of members of the internal data list; i.e. the
number of values stored; i.e. the number of values received by C<new>
or C<set_value> plus C<add_value> minus successful C<delet_values>'s.'

=cut

sub data_count {
  my ($this, $data) = @_;
  $data or $data = $this->data();
  return scalar @$data;
}

=head2 orig_data_count

 instance
 (int $count) data_count ()

DESCRIPTION:

Returns the number of members of the internal data list; i.e. the
number of values stored; i.e. the number of values received by C<new>
or C<set_value> plus C<add_value> minus successful C<delet_values>'s.'

=cut

sub orig_data_count {
  my ($this) = @_;
  return scalar @{ $this->orig_data() };
}

=head2 add_value

 instance
 () add_value (variable $value)

DESCRIPTION:

Will associate C<$value> with the proper data encapsulator and push it
to the back of the internal list.

=cut

sub add_value {
  my ($this, $value) = @_;
  my $datum = $this->create_datum($value);
  push @{$this->{_data}}, $datum;
}

=head2 delete_value

 instance
 () delete_value (variable $value)

DESCRIPTION:

Will delete C<$value> form the list of maintained values if found. If
the value passed in is not, however, canonicalizacle, then this
function might be unable to delete it.

=cut

sub delete_value {
  my ($this, $value) = @_;

  my @data = @{$this->data()};
  my $can_value = $data[0]->canonicalize($value) if @data;
  my (@new_data,@del_data);
  for my $datum (@{$this->{_data}}) {
    if ($datum->test_equality($can_value)) {
      push @del_data, $datum;
    } else {
      push @new_data, $datum;
    }
  }
  return unless @del_data;
  my $n_deleted = scalar(@del_data);

  if (! $this->{_orig_data}) {
    $this->{_orig_data} = $this->{_data};
  } else {
    map { $_->free_internals() } @del_data;
  }
  $this->{_data} = \@new_data;

  return $n_deleted;
}

=head2 set_value

 instance
 () set_value (variable @values)

DESCRIPTION:

Will reset the internal data list to correspond with the values passed
in.

=cut

sub set_value {
  my ($this, @values) = @_;

  my @data;
  for my $value (@values) {
    my $datum = $this->create_datum($value);
    push @data, $datum;
  }

  if (! $this->{_orig_data}) {
    $this->{_orig_data} = $this->{_data};
  } elsif ($this->{_data}) {
    map { $_->free_internals() } @{$this->{_data}};
  }
  $this->{_data} = \@data;

  return;
}

=head2 value

 instance
 (variable $value/@value) value ()

DESCRIPTION:

If called in a scalar context, will return the value maintained by the
first member of the internal list of data encapsulators, which is the
same as the order in which they where received unless they have been
reordered. If called in a list context, will return a list of values
maintained by said encapsulators whose order is subject the note
above.

See the note in DESCRIPTION above regarding retrieved values.

=cut

sub value {
  my ($this) = @_;

  my @values = map { $_->value() } @{ $this->data() };

  return @values if wantarray();
  return $values[0];
}

=head2 raw_value

 instance
 (variable $value/@value) value ()

DESCRIPTION:

If called in a scalar context, will return the value maintained by the
first member of the internal list of data encapsulators, which is the
same as the order in which they where received unless they have been
reordered. If called in a list context, will return a list of values
maintained by said encapsulators whose order is subject the note
above.

See the note in DESCRIPTION above regarding retrieved values.

=cut

sub raw_value {
  my ($this) = @_;

  my @values = map { $_->raw_value() } @{ $this->data() };

  return @values if wantarray();
  return $values[0];  
}

=head2 canonical_value

 instance
 (variable $canonical_value/@canonical_value) canonical_value ()

DESCRIPTION:

See the note in DESCRIPTION above regarding retrieved canonical
values.

=cut

sub canonical_value {
  my ($this) = @_;

  my @values = map { $_->canonical_value() } @{ $this->data() };

  return $values[0] unless wantarray;
  return @values;
}

=head2 formatted_value

 instance
 (string[] $values) format ()

DESCRIPTION:

This will return the values returned by format().  If called in a scalar
context, will return the first such value.

=cut

sub formatted_value {
  my ($this) = @_;

  my @values = map { $_->formatted_value() } @{ $this->data() };

  return @values if wantarray();
  return $values[0];  
}

=head2 set_orig_value

 instance
 () set_orig_value (variable @values)

DESCRIPTION:

Will reset the internal data list to correspond with the values passed
in.

=cut

sub set_orig_value {
  my ($this, @values) = @_;
  if (@values) {
    my @data;
    for my $value (@values) {
      my $datum = $this->create_datum($value);
      push @data, $datum;
    }

    if ($this->{_orig_data}) {
      map { $_->free_internals() } @{$this->{_orig_data}};
    }
    $this->{_orig_data} = \@data;
  } else {
    $this->{_orig_data} = $this->{_data};
  }

  return;
}

=head2 orig_value

 instance
 (variable $value/@value) orig_value ()

DESCRIPTION:

See the note in DESCRIPTION above regarding retrieved values.

=cut

sub orig_value {
  my ($this) = @_;

  my @values = map { $_->value() } @{ $this->orig_data() };

  return @values if wantarray();
  return $values[0];
}

=head2

=cut

sub changed_p {
  my ($this) = @_;
  return 0 unless defined $this->orig_value();
  return !$this->test_equality([ $this->orig_value() ]);
}

=head2 reset_possible

 instance
 () reset_possible ()

DESCRIPTION:

Will delete C<$value> form the list of maintained values if found. If
the value passed in is not, however, canonicalizacle, then this
function might be unable to delete it.

=cut

sub reset_possible {
  my ($this, $possible, $ordering) = @_;
  $ordering ||= [ sort keys %$possible ];

  $this->SUPER::reset_possible($possible,$ordering);

  for my $type (qw(_data _orig_data)) {
    next unless $this->{$type};
    for my $datum (@{$this->{$type}}) {
      $datum->reset_possible($possible,$ordering);
    }
  }
}

=head2 set_format

 instance
 () set_format (\%format)


DESCRIPTION:

This will call C<set_format> on each member data encapsulator, passing
C<%format> on.

=cut

sub set_format {
  my ($this, $format) = @_;
  map { $_->set_format($format) } @{$this->data()};
}

=head2 verify

 instance
 () verify ()

DESCRIPTION:

Will verify that all member data is 'good.' First will check that all
data is allowed by the possible values (if set) and will then call the
verification function on each data member.

=cut

sub verify {
  my ($this) = @_;
  my @problems;

  my @data = @{$this->data()};
  if (! @data) {
    if (! $this->{_parameters}->{blank_ok}) {
      push @problems, "Data cannot be blank.";
    }
  } else {
    for my $datum (@data) {
      # execute the datum's verify.
      my ($success,$problems) = $datum->verify();
      $success or push @problems, @$problems;
    }
  }

  @problems and return (0, \@problems);
  return (1, []);
}

=head2 canonicalize

 instance
 (variable $canonical_value/@canonical_value) canonical_value ()

DESCRIPTION:

If called in a scalar context, will return the canonical value
maintained by the first member of the internal list of data
encapsulators, which is the same as the order in which they where
received unless they have been reordered. If called in a list
context, will return a list of canonical values maintained by
said encapsulators whose order is subject the note above.

See the note in DESCRIPTION above regarding retrieved canonical
values.

=cut

sub canonicalize {
  my ($this,$data) = @_;
  $data ||= $this->data();

  my @values;
  for my $datum (@$data) {
    push @values, $datum->canonicalize();
  }
  return @values;
}

=head2 format

 instance
 (string[] $values, string[] $problems) format ()

DESCRIPTION:

This will call format on each member data encapsulator, and it will
return an array reference of vales, and an array references of errors
encountered.

=cut

sub format {
  my ($this) = @_;

  my (@values,@errors);
  for my $datum (@{$this->{_data}}) {
    my ($value,$problems) = $datum->format();
    defined $value or $value = $datum->value();
    push @values, $value;
    $problems and push @errors, @$problems;
  }

  return (\@values,\@errors);
}

=head2 test_equality

 instance
 (boolean) test_equality ()

DESCRIPTION:

blah

=cut

sub test_equality {
  my ($this,$in1,$in2) = @_;
  ref($in1) or die "test_equality() requires at least one ref parameter.\n";

  # if the input references are equal, the current is the original.
  return 1 if ref($in2) and $in1 eq $in2;

  my ($val1,$val2);
  if (ref($in1) eq 'ARRAY') {
    $val1 = [ map { (ref $_) ? $_->value() : $_ } @$in1 ];
  } else {
    $val1 = [ $in1->value() ];
  }

  if (!$in2) {
    $val2 = [ $this->value() ];
  } elsif (ref($in2) eq 'ARRAY') {
    $val2 = [ map { (ref $_) ? $_->value() : $_ } @$in2 ];
  } else {
    $val2 = [ $in2->value() ];
  }

  # if the two arrays do not have the same number of elements..
  @$val1 != @$val2 and return 0;

  # compare each datum.
  my ($datum,$value,$orig_value);
  for my $n (0..$#{$val1}) {
    if ($val1->[$n] ne $val2->[$n]) {
      return 0;
    }
  }
  return 1;
}

=head2 free_internals

 instance
 () free_internals (variable $value)

DESCRIPTION:

Will delete C<$value> form the list of maintained values if found. If
the value passed in is not, however, canonicalizacle, then this
function might be unable to delete it.

=cut

sub free_internals {
  my ($this) = @_;

  if ($this->{_orig_data}) {
    if ($this->{_data} ne $this->{_orig_data}) {
      for my $datum (@{$this->{_orig_data}}) {
	$datum->free_internals();
      }
    }
    delete $this->{_orig_data};
  }

  for my $datum (@{$this->{_data}}) {
    $datum->free_internals();
  }
  delete $this->{_data};

  $this->SUPER::free_internals();
  return;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
