package PSP::share;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: share.pm,v 1.2 2000/11/27 09:18:49 muaddib Exp $

use strict;
use Error qw(:try);

=head1 NAME

 PSP::share - implements package variable sharing for objects.

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 new

=cut

sub new {
  my ($proto,@args) = @_;
  my $this = {};
  bless $this, ref($proto)||$proto;
  return $this;
}

=head2 dest_share_class

 instance
 (string) dest_share_class ()

=cut

sub dest_share_class {
  my ($this,$name) = @_;
  $this->{_dest_class} = $name if defined $name;
  return $this->{_dest_class} if $this->{_dest_class};
  if (! defined $this->{_dest_class}) {
    if ($this->can("class_name")) {
      $name = $this->class_name();
    } else {
      $name = ref($this);
    }
    $this->{_dest_class} = $name;
  }
  return $name;
}

=head2 source_share_class

 instance
 (string) source_share_class ()

=cut

sub source_share_class {
  my ($this,$name) = @_;
  $this->{_source_class} = $name if defined $name;
  defined $this->{_source_class} or throw
    Error::Simple("Could not determine the source share class name.");
  return $this->{_source_class};
}

=head2 provide

 instance
 () provide (string[] \@var_list)

DESCRIPTION:

This function will provide a variable or variables to be shared
between the parent pile (the one which Loader is associated
with) and all its immediate subpiles.

Note that for efficiency\'s sake, the code invoked by C<share> and
C<unshare> is actually defined when C<provide> is called. This
is because, generally, provide is expected to be called one per pile
and it is cheaper to build the code here than to dynamically build on
each call to C<share> and C<unshare> (which are expected to
be called before and after every shared page).

=cut

sub provide {
  my ($this, @vars) = @_;
  #print STDERR $this->dest_share_class()." provides @vars\n";
  for my $var (@vars) {
    $this->{_provided}->{$var}++;
  }
  return @vars;
}
sub register { shift->provide(@_) }

sub expect {
  my ($this,@vars) = @_;
  #print STDERR $this->dest_share_class()." expects @vars\n";
  my %to_expect;
  map { $to_expect{$_}++ } @{$this->{_expected}}, @vars;
  for my $var (@{$this->{_expected}}) {
    delete $to_expect{$var};
  }
  for my $var (@vars) {
    delete $to_expect{$var} and push @{$this->{_expected}}, $var;
  }
  return @vars;
}

sub parse_var {
  my ($this,$var) = @_;

  my ($type,$name) = ($var =~ /^([\$\@\%\*\&])?(.*)$/);

  $type or throw 
    Error::Simple("$var does not match ^[\\\$\\\@\\\%\\\*\\\&].");
  $name =~ /::/ and throw 
    Error::Simple("Can't declare another package's variables.");
  $name =~ /^\w+[[{].*[]}]$/ and throw
    Error::Simple("Can't declare individual elements of hash or array.");
  $^W and length($name) == 1 and $name !~ tr/a-zA-Z// and throw
    Error::Simple("No need to declare built-in vars.");

  return ($type,$name);
}

=head2 share

 intance
 () share ([string $sub_pile])

DESCRIPTION:

This function actually does the sharing of provided variables. It
should generally be called before a page in a mounted pile is
invoked. This is because many root piles might be sharing the same
pile and where we not to share/unshare on each call, the root pile
namespaces would be corrupted.

This expects variables from a possibly existing parent, and then
recurses into children allowing them to expect variables.

=cut

sub share {
  my ($this,$vars,$source,$dest_class) = @_;
  $vars ||= $this->{_expected} or return;
  $dest_class ||= $this->dest_share_class();

  # if we are not given a source, and we are parent's child..
  if (! $source and $this->isa("PSP::parent")) {
    $source = $this->parent();
    $source or return;
  }

  # if a source_class is specified, a parent is not necessary.
  my ($source_class,$source_obj);
  if (ref($source)) {
    $source_obj = $source if $source->isa("PSP::share");
    if ($source->can("dest_share_class")) {
      $source_class = $source->dest_share_class();
    } else {
      $source_class = ref($source);
    }
  } elsif ($source) {
    $source_class = $source;
  } else {
    $source_class = $this->source_share_class();
  }

  #print STDERR "expect to share (@$vars) from $source_class to $dest_class\n";

  # iterate through the expected variable names
  for my $var (@$vars) {

    # make sure this expected variable is provided by the source class.
    if ($source_obj and ! $source_obj->{_provided}->{$var}) {
      throw Error::Simple
	("$var is expected by $dest_class but not provided by $source_class");
    }

    # parse this variable name.
    my ($type,$name) = $this->parse_var($var);

    # if the variable is already shared, don't share it again.
    next if eval '\\'.$type.$dest_class.'::'.$name.
      ' eq \\'.$type.$source_class.'::'.$name;

    # save a reference to the previous variable.
    eval '$this->{_shared_vars}->{$var} = \\'.$type.$dest_class.'::'.$name;

    # map the destination variable to the source variable.
    eval '*{'.$dest_class.'::'.$name.'} = \\'.$type.$source_class.'::'.$name;
  }
  return @$vars;
}

=head2 unshare

 intance
 () unshare ([string $sub_pile])

DESCRIPTION:

This function unshares all provided variables. For an explination of
use, see C<share> above.

=cut

sub unshare {
  my ($this,$vars,$dest_class) = @_;
  $vars ||= [ sort keys %{$this->{_shared_vars}||{}} ];
  $dest_class ||= $this->dest_share_class();

  #print STDERR "unsharing (@$vars) from $dest_class\n";

  # iterate through the expected variable names
  for my $var (@$vars) {

    # parse input variable.
    my ($type,$name) = $this->parse_var($var);

    # undefine the glob entry or glob.
    my %types = qw($ SCALAR % HASH @ ARRAY & CODE);
    my $glob_type = $types{$type} ? "{$types{$type}}" : '';

    # recover the reference to the previous variable.
    eval '*{'.$dest_class.'::'.$name.'} '.
      '= delete $this->{_shared_vars}->{$var}';
  }

  # recurse into children with shared variables.
  if ($this->isa("PSP::parent")) {
    for my $child ($this->children()) {
      $child->{_shared_vars} or next;
      $child->unshare();
    }
  }

  return @$vars;
}

=head2 DESTROY

 class
 (PSP::FieldSpace $fs) DESTROY ()

DESCRIPTION:

=cut

sub DESTROY {
  my ($this) = @_;
  $this->unshare();
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<PSP::Utils>

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
